<?php

class ResourceBackup extends CFormModel {
    
    static $dbName = "db.json";
    protected $tmpFolder ;
    
    public $file = null;
    public $filepath = null;
    
    public $allowNewApps = false;
    
    public $ids = array();
    public $actions = array();
    public function init() {
        parent::init();
        $this->tmpFolder = GlobalSettings::$fileBase ."backup/";
    }
    public function rules() {
		return array(
            array( "ids", 'safe'),
            array( 'file', 'file', 'types'=>'7z, zip, bck' ),
		);
	}
    static function logErrors( $item, $model ) {
        foreach ( $model->getErrors() as $name=> $values ){
            Yii::log("Error extracting backup: '$model->name': $name => ". implode("; ",$values), 'warning', 'spx.assets');
            if( PHP_OS == "WINNT" ) {
                echo "Error extracting backup: '$model->name': $name => ". implode("; ",$values)."\n"; 
            }
        }
    }
    public function mustSync() {
        if ( !Actions::getAction( 'content not synced' ) ){
            $ar = new Resource();
            $db = $ar->getDbConnection();
            $db->createCommand('PRAGMA synchronous = FULL;')->execute();
            Actions::setAction( 'content not synced', true );
            $db->createCommand('PRAGMA synchronous = NORMAL;')->execute();
        }
    }
    function findDependencies( $ids ) {         
        $newIds = array();
        for ( $i = 0; $i < count($ids); $i++) {
            $id  = $ids[$i];            
            $resource = Resource::model()->findByPk($id);
            if ( $resource === null ){
                return false;
            }
            $data = $resource->getListDesc( false );
            $data['folder'] = $resource->getFolder();
            if ( $resource->uuid ) {
                $data['uuid'] = $resource->uuid;
                $data['version'] = $resource->version;
            }
            $newIds[] = $data;
            foreach ( $resource->children as $child ) {  
                $ids[] = $child->id;                
            }
        }
        
        $doneId = array();
        $result = array();
        foreach( array_reverse( $newIds ) as $item ) {
            if ( !in_array( $item['id'], $doneId) ){
                $doneId[] = $item['id'];
                $result[] = $item;
            }
        };        
        return $result;
    }

    function createDB( $ids ) {
        $dbFile = GlobalSettings::$fileBase . self::$dbName;
        $db = array();
        foreach ( $ids as $data ) {
            if ( $data['type'] == 'media' || $data['type'] == 'apps' || $data['type'] == 'project' ) {
                unset( $data['manage'] );
                unset( $data['folder'] );
                $db[] = $data;            
            } else {
                $model = Resource::loadModel( $data['type'], $data['id'], false );
                if ( !$model ) {
                    $this->addError('ids', "Cannot find id: ".$data['id'] );
                    return false;
                }
                $allData = $model->getDesc( false, true );
                unset( $allData['href'] );
                unset( $allData['manage'] );   
                unset( $allData['previews'] );   
                $db[] = $allData;
            }
        }
        file_put_contents( $dbFile, json_encode( $db ) );
        return $dbFile;
    }
    function backup(  ) {
        
        $ids = $this->findDependencies( $this->ids );
        if ( $ids === false ) {
            $this->addError('ids', "not found" );
            return false;
        }
        $tempName = GlobalSettings::$fileBase . "backup.7z";
        if ( file_exists($tempName) ){
            unlink($tempName);
        }
        // create backup
        $db = $this->createDB( $ids );
        if ( $db === false ) {
            $this->addError('ids', "cannot save database" );
            return false;
        }
        $folders = array( "-xr!previews", $db );
        
        foreach ( $ids as $data ) {
            if ( $data['type'] == 'media' || $data['type'] == 'apps' || $data['type'] == 'project' ) {
                $folders[] = $data['folder'];                   
            }
            $this->actions[] = "Added ".$data['type']." ".$data['name'];
        }
        $zipHandler = new ZipHandler;
        $result = $zipHandler->createArchive( $tempName, $folders, GlobalSettings::$fileBase );
        
        unlink( $db );
        if ( $result === false ){
            $this->addError('ids', $zipHandler->error );            
        }
        return $result;
    }
    function clear( ) {        
        GlobalSettings::rrmdir( $this->tmpFolder );
    }
    function switchFolders( $name, $path, $target ){        
        if ( !@rename( $target, $target ."bck" ) ){
            Yii::log("rename failed: rename( $target, $target.bck )", 'warning', 'spx.media');
            throw new Exception( "update failed" .": ". $name );            
        }
        if ( !@rename( $path, $target ) ){
            @rename( $target . "bck", $target );
            Yii::log("replace failed: rename( $path, $target )", 'warning', 'spx.media');
            throw new Exception( "update failed" .": ". $name );
        }        
    }
    function revertFolders( $name, $target ){ 
        
        GlobalSettings::rrmdir( $target );
        if ( !@rename( $target ."bck", $target ) ){
            Yii::log("revert failed: rename( $target.bck, $target )", 'warning', 'spx.media');
            throw new Exception( "update failed".": ".$name );
        }
        
    }
    function cleanFolders( $target ){ 
        GlobalSettings::rrmdir( $target ."bck" );
    }
    function checkVersion( $path ){
        if ( !file_exists( $path . self::$dbName ) ){
            return "db error".": " . "missing";            
        }
        $file = file_get_contents( $path . self::$dbName );
        if ( !$file ){
            return  "db error".": " . "empty";
        }
        $db = json_decode( $file, true );
        if ( !$db ){
            return  "db error".": " . "expecting json"; 
        }
        if ( !isset( $db['uuid'])  ) {
            return "db error".": " . "uuid missing"; 
        }
        if ( !isset( $db['version']) ) {
            return "db error".": " . "version missing"; 
        }
        if ( !isset( $db['type']) ) {
            return "db error".": " . "type missing";
        }
        $criteria = new CDbCriteria();
        $criteria->addColumnCondition( array('uuid' => $db['uuid']) );
        $models = Resource::loadAll( 'resource', $criteria );
        if ( count( $models ) == 0){
            return true;
        }
        foreach ( $models as $model ){
            if ( version_compare( $db['version'], $model->version, ">" ) ){
                return true;
            }
        }
        return $model->version;
    }
    public function newApps( $item, $path ){

        $name = $item['name'];
        // find if the apps is already present
        if ( !file_exists( $path . Apps::$appsFile ) ){
            throw new Exception( "config file missing" .": apps->".$name);
        }
        
        if ( !isset( $item['uuid']) ) {
            throw new Exception( "uuid missing" .": apps->".$name);
        }
        if ( !isset( $item['version']) ) {
            throw new Exception( "version missing" .": apps->".$name);
        }
        
        // get the existing apps and check the uuid and versions
        $criteria = new CDbCriteria();
        $criteria->addColumnCondition( array('uuid' => $item['uuid']) );
        $models = Apps::model()->findAll( $criteria );
        foreach ( $models as $model ){                      
            if ( $model->uuid == $item['uuid'] ){  
                
                if ( version_compare( $item['version'], $model->version,"<=" ) ){
                    // the apps is already up to date
                    $this->actions[] = "Skipping apps $name";
                    return $model;
                }
                if ( !$this->allowNewApps ){
                    throw new Exception( "version is not up to date".": apps->".$name );
                }
                
                // we need to update the apps from what we have
                $model->version = $item['version'];
                $id = $model->id;
                $target = substr( $model->getFolder(), 0, -1 );
                $this->switchFolders( $name, $path, $target );
                // reload the apps
                $this->actions[] = "Updated apps $name";
                if( !$model->save() ) {
                    self::logErrors( $item, $model );
                    $model->rollback();
                    $this->revertFolders( $name, $target );
                    throw new Exception( "save failed".": apps->".$name );
                }
                $this->cleanFolders( $target );
                return Resource::loadModel('apps', $id );
            }
        }
        if ( !$this->allowNewApps ){
            throw new Exception(I18N::t("adding apps not allowed").": apps->".$name);
        }
        $model = new Apps;        
        $model->attributes = $item;
        $model->uuid = $item['uuid'];
        $model->version = $item['version'];
        $model->setAppsFile( array( 'format' => '7z', 'folder' => $path, 'name' => $item['name'] ) );
        
        if( !$model->save() ){     
            self::logErrors( $item, $model );
            throw new Exception( "save failed".": apps->".$name );
        } 
        $this->actions[] = "Added apps $name";
        return $model;
    }
    public function newMedia( $item, $path ){
        
        $name = $item['name'];
        $model = null;
        if ( isset( $item['uuid']) && isset( $item['version']) ){
            // get the existing apps and check the uuid and versions
            $models = Media::model()->findAll( );
            foreach ( $models as $testModel ){     
                if ( $testModel->resource->uuid == $item['uuid'] ){  
                    if ( version_compare( $item['version'], $testModel->resource->version ,"<=" ) ){
                        // the media is already up to date
                        $this->actions[] = "Skipping media $name";
                        return $testModel;
                    }
                    $testModel->resource->version = $item['version'];                    
                    $this->actions[] = "Updated media $name\n";    
                    $model = $testModel;
                    break;
                }
            }
        }      
        if ( !$model ) {
            $resource = new Resource;
            $resource->attributes = $item;
            $resource->type = "media";
            if ( isset( $item['uuid']) && isset( $item['version'] ) ){
                $resource->uuid = $item['uuid'];
                $resource->version = $item['version'];
            }
            if( !$resource->save() ){   
                self::logErrors( $item, $resource );
                throw new Exception( "save failed".": media->".$name );
            }

            $model = new Media;
            $model->id = $resource->id;
            
            $this->actions[] = "Added media ".$item['name'];            
        } else {
            if ( isset( $item['uuid']) && isset( $item['version'] ) ){
                $model->resource->uuid = $item['uuid'];
                $model->resource->version = $item['version'];
            }
            if( !$model->resource->save() ){   
                self::logErrors( $item, $model );
                throw new Exception( "save failed".": media->".$name );
            }
        }
        if ( isset( $item['hidden']) &&  $item['hidden'] ){
            $model->hidden = 1;
        }
        $options = array( 'format' => 'file' );
        if ( !file_exists($path)){
            throw new Exception( I18N::t("file not found").": media->".$name );
        }
        foreach ( scandir($path) as $object) {
            if ( filetype($path . $object) == 'file' && in_array(substr($object, -3), Media::$mime2ext ) ){
                $options['file'] = $path . $object;                
                break;
            }
        }
        if( !$model->processFile( $options ) ){
            $model->rollback();
            $this->addErrors( $model->getErrors() );
            array_pop( $this->actions );
            throw new Exception("processing failed".": media->".$name);
        }
        
        $model->resource->delayPreview = true;
        if( !$model->save() ) {
            self::logErrors( $item, $model );
            $model->rollback();
            array_pop( $this->actions );
            throw new Exception( "save failed".": media->".$name );
        }  
        
        return $model;
    }
    public function newProject( $item, $path ){
        
        $name = $item['name'];        
        
        $model = new Project;   
        
        unset( $item['query'] );
        $model->attributes = $item;
        
        $model->processFile( array( 'format' => '7z', 'folder' => $path, 'name' => $item['name'] ) );
        
        if( !$model->save() ){     
            self::logErrors( $item, $model );
            throw new Exception( "save failed".": project->".$name );
        } 
        $this->actions[] = "Added project $name";
        return $model;
    }
    public function fixId( &$item, $idMap ) {
        $ret = true;
        foreach ( $item as $key => &$value ){
            if ( is_array( $value ) ){
                $ret &= $this->fixId( $value, $idMap );
            } else if ( $key === 'id' ){
                if ( isset( $idMap[$value]) ){
                    $this->actions[count($this->actions)-1] .= ", '$value' to '".$idMap[$value]."'";
                    $item['id'] = $idMap[$value];                    
                } else {
                    $this->actions[count($this->actions)-1] .= ", '$value' not found";
                    $ret = false;
                }                               
            }
        }
        return $ret;
    }
    public function newItem( $item, $idMap  ){
        
        $name = $item['name'];
        // fixing the ids first        
        if ( isset( $item['uuid']) && isset( $item['version']) ){
            // get the existing apps and check the uuid and versions
            $criteria = new CDbCriteria();
            $criteria->addColumnCondition( array('uuid' => $item['uuid']) );
            $models = Resource::loadAll( $item['type'], $criteria );
            foreach ( $models as $model ){     
                if ( $model->uuid == $item['uuid'] ){  
                    if ( version_compare( $item['version'], $model->version, "<=" ) ){
                        // the media is already up to date
                        $this->actions[] = "Skipping ".$item['type']." $name";
                        return $model;
                    }
                    $this->actions[] = "Fixing id for ".$item['type']." ".$item['name'];                    
                    if ( !$this->fixId( $item, $idMap ) ) {
                        Yii::log("Error extracting backup: '$name': ".$this->actions[count($this->actions)-1], 'warning', 'spx.assets');
                        throw new Exception( "update failed".": ".$item['type']."->".$name );
                    }
                    $model->version = $item['version'];
                    $model->attributes = $item;                    
                    if( !$model->save() ) {
                        self::logErrors( $item, $model );
                        $model->rollback();                        
                        throw new Exception( "save failed".": ".$item['type']."->".$name );
                    }

                    $this->actions[] = "Updated ".$item['type']." $name\n";
                    
                    Actions::setAction("preview", true);
                    return $model;
                }
            }
        }   
        
        $model = Resource::newModel( $item['type'] );
        $this->actions[] = "Fixing id for ".$item['type']." ".$item['name'];                                        
        if ( !$this->fixId( $item, $idMap ) ) {
            Yii::log("Error extracting backup: '$name': ".$this->actions[count($this->actions)-1], 'warning', 'spx.assets');
            throw new Exception( "update failed".": ".$item['type']."->".$name );
        }
        $model->attributes = $item;        
        if ( isset( $item['uuid']) && isset( $item['version'] ) ){
            $model->uuid = $item['uuid'];
            $model->version = $item['version'];
        }
        $model->delayPreview = true;
        
        if( !$model->save() ){ 
            self::logErrors( $item, $model );
            throw new Exception( "save failed".": ".$item['type']."->".$name );
        }
        $this->actions[] = "Added ".$item['type']." ".$item['name'];
        Actions::setAction("preview", true);
        return $model;
    }
    
    function parseDb( $file ) {
        $this->actions[count($this->actions)-1] .= " ".pathinfo( $file, PATHINFO_BASENAME );
        $base = pathinfo( $file,  PATHINFO_DIRNAME ) ."/";
        $maindb = json_decode(file_get_contents( $file ), true );
        if ( $maindb === null ){
            return null;
        }
        if ( array_values($maindb) !== $maindb ){
            $maindb = array ( $maindb );
        }
        $db = array();
        // load includes
        foreach ( $maindb as $item ){
            if ( isset ($item['include'] ) ){
                $incDb = $this->parseDb( $base . $item['include'] );
                if ( $incDb === null ){
                    return null;
                }
                foreach ( $incDb  as $data ){
                    $db[] = $data;
                }
            } else {
                if ( isset( $item['id']) ){
                    $item['path'] = $base . $item['id'] ."/";
                } else {
                    $item['path'] = $base;
                }
                $db[] = $item;
            }
        }                 
        return $db;
    }
    function restore( ) {
        // Extract the backup to start
        if ( !file_exists( $this->tmpFolder ) ){
            mkdir( $this->tmpFolder );
        }
        $zip = new ZipHandler;
        if ( $this->filepath ){
            $path = $this->filepath;
        } else {
            $path = $this->file->tempName;
        }   
        if ( !$zip->open( $path, "7z", true ) ){
            $folder = substr($path, 0, -6);
            if ( file_exists( $folder . self::$dbName ) ){
                if ( !GlobalSettings::rcopy( $folder, $this->tmpFolder ) ) {
                    $this->addError('file', $zip->error );
                    return false;
                } 
            } else {
                $this->addError('file', $zip->error );
                return false;
            }
        } else {
            if ( $zip->extractTo( $this->tmpFolder )===false ){
                $this->addError('file', $zip->error );
                return false;
            }
        }
        // read the db
        if ( !file_exists( $this->tmpFolder . self::$dbName ) ){
            $this->addError('file', "file not found" );
            $this->clear();            
            return false;
        }
        $this->actions[] = "Opening";
        $db = $this->parseDb( $this->tmpFolder . self::$dbName );
        if ( $db === null ){
            $this->addError('file', "invalid db" );
            $this->clear();            
            return false;
        }
        $idMap = array();
        $doneMmodel = array();
        
        // prepare the IdMap:
        $models = Resource::model()->findAll( );
        foreach ( $models as $model ){
            if ( $model->uuid ){
                $idMap[ $model->uuid ] = $model->id;
            }
        }
        
        $this->mustSync();        
        $transaction = Yii::app()->db->beginTransaction();
        
        foreach ( $db as $item ){
            if ( isset( $item['id'] ) ) {
                $id = $item['id'];
                unset( $item['id'] );
            } else {
                $id = 0;
            }
            try {
                if ( $item['type'] == 'apps' ){
                    $model = $this->newApps( $item, $item['path'] );
                } else if ( $item['type'] == 'schedule' ){

                } else if ( $item['type'] == 'media' ){
                    $model = $this->newMedia( $item, $item['path'] );                    
                } else if ( $item['type'] == 'project' ){
                    $model = $this->newProject( $item, $item['path'] );                    
                } else {
                    $model = $this->newItem( $item, $idMap );
                }
                if ( $model ) {
                    if ( $id ){
                        $idMap[ $id ] = $model->id;
                    } 
                    if ( isset( $item['uuid'] ) ){
                        $idMap[ $item['uuid'] ] = $model->id;
                    }
                    $doneMmodel[] = $model;
                }
            } catch(Exception $e) { 
                $this->addError('file', $e->getMessage() );
            }
        }        
        $transaction->commit();        
        $this->clear();
        if ( $this->hasErrors() ){
            return false;
        }
        return count( $doneMmodel );
    }
    
}
