<?php

/**
 * This is the model class for table "Playlist".
 *
 * The followings are the available columns in table 'Playlist':
 * @property integer $id
 * @property string $name
 * @property double $defaultDur
 * @property string $transition
 * @property integer $looping
 * @property integer $shuffle
 */
class Playlist extends Resource
{
    
    protected $playlistFile = "playlist.js";
    
    
    protected $_optionsAttr = array( 'defaultDur', 'looping', 'shuffle', 'transition' );
    protected $_options = null;
    protected $_options_ok = true;
    protected $_media = null;
    protected $_media_modified = true;
    
    protected $_dur = null;
    var $sourceId = null;
    
    protected $_old_media = null;
	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
        $rules = parent::rules();
        
        $rules[] = array('looping, shuffle', 'boolean');
        $rules[] = array('defaultDur', 'numerical');
        
        $rules[] = array('options', 'safe');
        $rules[] = array('transition', 'validateTransition');
        $rules[] = array('media', 'validateMedia');
        $rules[] = array('sourceId', 'validateSourceId', 'on'=>'insert');
		return $rules;
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id' => 'id',
			'name' => 'name',
			'defaultDur' => 'options.defaultDur',
			'transition' => 'options.transition',
			'looping' => 'options.looping',
			'shuffle' => 'options.shuffle',
            'media' => 'media',
		);
	}
    public function init(){
        parent::init();
        $this->type = "playlist"; 
    }
    public function defaultScope(){
        return array(
            'condition'=>"type='playlist'",
        );
    }

    public function loadPlaylist( $force=false ) {
        $path = $this->getFolder();
        if ( !$path )
            return false;
        if ( !file_exists( $path . $this->playlistFile ))
            return false;
        $playlist = file_get_contents( $path . $this->playlistFile );
        if ( empty($playlist) )
            return false;
        $items = explode(";", $playlist );
        foreach ($items as $item){
            $setings = explode("=", $item, 2);
            if ( trim( $setings[0] )=="options" && ( $force || !$this->_options ))
                $this->_options = json_decode( trim( $setings[1] ), true );
            if ( trim( $setings[0] )=="media" && ( $force || !$this->_media )){
                $this->_media = json_decode( trim( $setings[1] ), true );
                $this->_old_media = $this->_media;
                $this->_media_modified = false;
            }
        }        
        return true;
    }
    public function savePlaylist( $path ) {
        $playlist = "";
        if ( !empty(  $this->_options ) )
            $playlist .= "options = ".json_encode( $this->_options ). ";\n";
        else
            $playlist .= "options = {};\n";
        $playlist .= "media = ".json_encode( $this->_media ). ";\n";
        $filepath = $path . $this->playlistFile ;
        return Resource::saveAtomic( $filepath, $playlist);
    }
     public function getDur() {
        if ( $this->_dur === null ){
            $dur = 0 ;
            $mediaSet = $this->getMedia(); 
            foreach( $mediaSet as $media ) {
                if ( isset($media['dur']) && floatval($media['dur']) != 0 ){
                    $dur += floatval( $media['dur'] );
                } else if ( isset($media['mediaDur']) && floatval($media['mediaDur']) != 0 ){ 
                    $dur += floatval( $media['mediaDur'] );
                } else {
                    $dur += floatval( $this->_options['defaultDur'] );            
                }
            }
            $this->_dur = $dur;
        }
        return $this->_dur;
    }
    public function getOptions() {
        if ( $this->_options === null )
            $this->loadPlaylist();        
        if ( $this->_options === null ){
            $this->setOptions( array() );            
        }
        return $this->_options;
    }
    public function setOptions( $options ) {
        $default = array (
                'defaultDur' => 5, 
                'looping' => false,
                'shuffle' => false,
                'transition' => null
            );
        if ( is_array( $options ) ){
            $this->_options = array_merge( $default, $options );  
            $this->_options_ok = true;
        } else {
            $this->_options_ok = false;      
            $this->_options = $default;
        }
    }
    public function __get( $name ) {
        if ( in_array( $name, $this->_optionsAttr ) ){
            $options = $this->getOptions();
            return $options[$name];
        } else
            return parent::__get( $name );            
    }
    public function __set( $name, $value ) {
        if ( in_array( $name, $this->_optionsAttr ) ){
            $options = $this->getOptions();
            $options[$name] = $value;
            
            $this->setOptions( $options );
        } else
            return parent::__set( $name, $value );            
    }
    public function validateTransition( $attribute ){
        
        if ( !$this->_options_ok ) {            
            $this->addError( 'options', "wrong format" );
            return false;
        }
        $val = new Validator( $this );
        $val->validateTransition($this->$attribute, $attribute);
        
        return true;
    }
    
    public function getMedia(){
        if ( $this->_media === null ){
            $this->loadPlaylist();         
        }
        if ( $this->_media === null ){
            $this->_media = array();
        }
        /*
        // fix the description of the playlist to include slide description
        foreach( $this->_media as $idx => &$media ) {
            if ( $media['type'] === 'apps' ){
                $id = $media['id'];
                $model = Apps::model()->findByPk($id);
                if ( !$model ){
                    // we have a problem the apps is gone, 
                    // remove this from the list
                    unset( $this->_media[$idx] );
                    continue;                    
                }                
                // add the read only info for the slide
                $media['apps'] = $model->desc;
            }
            if ( isset($media['dur']) ){
                $dur += floatval( $media['dur'] );
            } else if ( isset($media['mediaDur']) ){ 
                $dur += floatval( $media['mediaDur'] );
            } else {
                $dur += floatval( $this->_options['defaultDur'] );            
            }
        }
        */
        return $this->_media;
    }
    public function setMedia( $media ){
        $this->_media = $media;  
        $this->_media_modified = true;
    }
    public function validateMedia( $attribute ){
        if ( $attribute!="media" ){
            $this->addError( $attribute, "should be a media" );
            return false;
        }
        $childrens = array();
        
        $val = new Validator( $this );
        if ( $this->_media !== null ) {
            $val->validateMedia( $this->_media, $childrens, $attribute, false, true);

            // Update the AR relations
            $this->children = $childrens;
        } else {
            // load the media from the disk
            $this->getMedia();
        }
    }
    public function validateSourceId( $attribute ){
        if ( $attribute!="sourceId" ){
            $this->addError( $attribute, "should be an id" );
            return false;
        }
        $id = $this->$attribute;
        if ( empty($id) ){
            return;
        }
        $model = Playlist::model()->findByPk($id);
        if ( !$model ){
             $this->addError( $attribute, "missing" );
             return false;
        }
        
        $this->setOptions( $model->getOptions() );
        $this->setMedia( $model->getMedia() );
        $this->validateMedia( 'media' ); // update the relations
    }
    public function removeChild( $childId, $delayPreview = false ) {
        $this->delayPreview = $delayPreview;
        $old_media = $this->getMedia();
        $new_media = array();
        foreach( $old_media as $media ){
            if ( $media['id'] != $childId )
                $new_media[] = $media;
        }
        $this->setMedia( $new_media  );
        
        if ( $this->save() ) {
            return true;
        } else
            return false;
    }
    public function rollback() {
        if ( $this->_media_modified ) {
            $this->setMedia($this->_old_media);
            if (!$this->savePlaylist( $this->getFolder() ) )
                throw new Exception( "rollback failed" );
        }
        parent::rollback( );
    }
    
    public function getHref() {
        return parent::getHref() ."playlist.svg";
    }
    
    public function getListDesc( $preview=true ) {
        $description = array(
            'id' => $this->id,
            'name' => $this->name,
            'type' => 'playlist',
            'dur' => $this->getDur(),
            'href' => $this->href,
            'manage' => $this->manage,  
            'modified' => $this->modified,
            'keywords' => $this->keywords
        );        
        if (  $preview ) {
            $previews = $this->previews;
            if ( count( $previews ) )
                $description['previews'] = $previews;
        }
        return $description;
    }
    public function getDesc( $parents = true, $short = false ) {
        $description = $this->getListDesc( $parents ); 
        $description['options']= $this->getOptions();

        $description['media']= $this->getMedia();
        if ( $parents ) {
            foreach ( $this->parents as $parent ){
                $description['parents'][] = $parent->getListDesc( false );
            }
        }
        return $description;
    }
    public function getPreviews() {
        $where = $this->getFolder() ;
        if ( !file_exists( $where . "previews/" ) ){
            $this->generatePreviews( $where );
        }
        return parent::getPreviews();
    }
    public function generatePreviews( $where ) {
        // clear up the preview folder and create it
        $path = $where."previews/";
        if (file_exists($path) )
            GlobalSettings::rrmdir( $path, false );            
        else
            mkdir( $path );
        if ( count($this->children)>0 ){
            $media = $this->children[0];
            // TODO: create a link instead of copy
            $dir = $media->getFolder() . "previews/";
            $objects = scandir( $dir );
            foreach ($objects as $filename) {
                if ( substr( $filename,0, 7 )=="preview" )
                    @copy( $dir.$filename, $path.$filename );
            }            
        } else {
            // default previews
            @copy( GlobalSettings::$systemMedia."playlist.png", $path."preview_98x58_0_generated.png" );
        }
        // for the moment we just copy the previews of the first media
        
    }
    
    public function recover( $file = null ){
        
        $path = $this->getFolder();
        $playlistFilename = $path . $this->playlistFile;   
        $recover = parent::recover($playlistFilename);     
        if ( !$recover ){
            $this->copyNew( GlobalSettings::$systemMedia."playlist.svg", $path . "playlist.svg" );
        }
        return $recover;        
    }
    public function saveFiles(){
        $path = $this->getFolder();
        if ( !$path ){
            $this->addError( "", "path not found" );
            return false;
        }
        $this->copyNew( GlobalSettings::$systemMedia."playlist.svg", $path."playlist.svg" );
        
        if ( $this->_media_modified ){
            if ( !$this->savePlaylist( $path ) ){
                $this->addError( "", "save failed" );
                return false;
            }
            $this->generatePreviews( $path );
            $this->etag = $this->computeEtag( $path . $this->playlistFile );
        }
        return true;
    }
    public function beforeSave(){
        if ( !$this->isNewRecord && !$this->saveFiles() ){
            return false;
        }
        return parent::beforeSave();
    }
    public function afterSave(){ 
        parent::afterSave();
        
        if ( $this->isNewRecord && !$this->saveFiles() ){
            throw new Exception( "save failed" );
        }        
    }
    /**
     * Overload the find function to make sure that only playlist are returned 
     * when searching for an item
     * 
    public function findAll( $condition='', $params=array() ){
        $criteria=$this->getCommandBuilder()->createCriteria( $condition, $params );
		$criteria->compare( 'type', 'playlist' );
        
        return parent::findAll( $criteria );
    }
   */
	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 *
	 * Typical usecase:
	 * - Initialize the model fields with values from filter form.
	 * - Execute this method to get CActiveDataProvider instance which will filter
	 * models according to data in model fields.
	 * - Pass data provider to CGridView, CListView or any similar widget.
	 *
	 * @return CActiveDataProvider the data provider that can return the models
	 * based on the search/filter conditions.
	 */
	public function search()
	{
		// @todo Please modify the following code to remove attributes that should not be searched.

		$criteria=new CDbCriteria;

		$criteria->compare('id',$this->id);
		$criteria->compare('name',$this->name,true);
		$criteria->compare('defaultDur',$this->defaultDur);
		$criteria->compare('transition',$this->transition,true);
		$criteria->compare('looping',$this->looping);
		$criteria->compare('shuffle',$this->shuffle);

		return new CActiveDataProvider($this, array(
			'criteria'=>$criteria,
		));
	}

	/**
	 * Returns the static model of the specified AR class.
	 * Please note that you should have this exact method in all your CActiveRecord descendants!
	 * @param string $className active record class name.
	 * @return Playlist the static model class
	 */
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}
}
