<?php
class MaintenanceJobs
{
    static $report = "/usr/sbin/spxreport";
    static $boot_config = "/boot-config";
    static $enrolfile = "/etc/default/spxenroll";
    static $iotfile = '/etc/default/spxiot';
    static $aryafile = '/etc/spxmanage/arya-settings';
    static $interfaceindexfile = "/srv/raperca/interface/public/index.svg";
    protected $ssh_init;
    protected $ccconfig;

    function __construct()
    {
        $this->ssh_init = file_exists('/etc/init.d/ssh-delayed')
            ? '/etc/init.d/ssh-delayed'
            : '/etc/init.d/sshd';

        $this->ccconfig = file_exists("/etc/spxmanage")
            ? "/etc/spxmanage/configured"
            : "/etc/spinetix/configured";
    }

    function clearConfigured()
    {
        if (file_exists($this->ccconfig)) {
            unlink($this->ccconfig);
        }
        return true;
    }
    function isNotConfigured()
    {
        return !file_exists($this->ccconfig);
    }

    function getWizard()
    {
        if ($this->isNotConfigured()) {
            return true;
        }

        $config = file_get_contents($this->ccconfig);

        if (preg_match("/^\s*WIZARD=(.*)/m", $config, $matches)) {
            return $matches[1] != 0;
        }
    }
    function setWizard($useWizard = false)
    {
        if ($this->isNotConfigured()) {
            $config = "";
        } else {
            $config = file_get_contents($this->ccconfig);
            if (preg_match("/^\s*WIZARD=(.*)/m", $config, $matches)) {
                if (
                    ($matches[1] == 1 && $useWizard) ||
                    ($matches[1] == 0 && !$useWizard)
                ) {
                    return true; // nothing changed, don't write anything
                }
            }
        }

        $str =
            "WIZARD=" .
            ($useWizard === true || $useWizard === "true" || $useWizard == 1
                ? "1"
                : "0");

        if (preg_match("/^\s*WIZARD=/m", $config, $matches)) {
            $config = preg_replace("/^\s*WIZARD=.*/m", $str, $config);
        } else {
            $config = $str . "\n" . $config;
        }

        return file_put_contents($this->ccconfig, $config);
    }

    static function bootID()
    {
        return trim(file_get_contents("/proc/sys/kernel/random/boot_id"));
    }

    static function saveSymlink($target, $link)
    {
        if (PHP_OS == "WINNT") {
            if ($target[0] == '/') {
                copy($target, $link);
            } else {
                copy(dirname($link) . "/" . $target, $link);
            }
            file_put_contents($link . ".link", $target);
            return true;
        }
        if (!@symlink($target, $link)) {
            unlink($link);
            if (!@symlink($target, $link)) {
                return "Cannot save settings- File: $link as symlink to ($target)";
            }
        }
        exec('sync');
        return true;
    }

    static function saveFile(
        $name,
        $data,
        $secret = false,
        $group = 'raperca',
        $ignore_empty = true
    ) {
        if ($data == "" && $ignore_empty) {
            return true;
        }

        if (PHP_OS == "WINNT") {
            if (file_put_contents($name, $data) === false) {
                return 'Cannot save settings- File: ($name).<br/>Data: ' .
                    htmlspecialchars($data);
            }
            return true;
        }
        $fname = tempnam(dirname($name), basename($name));
        if (file_put_contents($fname, $data) === false) {
            return 'Cannot save settings- File: ($name).<br/>Data: ' .
                htmlspecialchars($data);
        } else {
            if (!$secret) {
                chmod($fname, 0644);
            } else {
                chmod($fname, $group ? 0640 : 0600);
                if ($group) {
                    chgrp($fname, $group);
                }
            }
            exec('sync');
            if (rename($fname, $name)) {
                exec('sync');
                return true;
            } else {
                unlink($fname);
                return "Cannot save settings- File: ($name).<br/>Data: " .
                    htmlspecialchars($data);
            }
        }
        return true;
    }

    // Copies the given source file or data data to the destination file
    // under the failsafe-data filesystem. If srcdata is FALSE the
    // destination file is removed instead. If isfile is true srcdata is a
    // file name, otherwise it is the data to copy. Returns false if the
    // failsafe-data data could not be updated.
    static function setSafeModeData($dstfile, $srcdata, $isfile = true)
    {
        if (PHP_OS == "WINNT") {
            return true;
        }
        $out = "";
        $status = "";
        if ($srcdata) {
            if ($isfile) {
                $args =
                    "copy " .
                    escapeshellarg($dstfile) .
                    " " .
                    escapeshellarg($srcdata);
            } else {
                $args =
                    "set " .
                    escapeshellarg($dstfile) .
                    " " .
                    escapeshellarg($srcdata);
            }
        } else {
            $args = "remove " . escapeshellarg($dstfile);
        }

        exec("failsafe-data " . $args, $out, $status);
        return $status == 0;
    }

    static function setSafeModeDataExt($commands)
    {
        if (PHP_OS == "WINNT") {
            return true;
        }
        $out = "";
        $status = "";
        $args = "";
        foreach ($commands as $command) {
            if (isset($command['data'])) {
                $args .=
                    "set " .
                    escapeshellarg($command['dst']) .
                    " " .
                    escapeshellarg($command['data']) .
                    " ";
            } elseif (isset($command['src'])) {
                $args .=
                    "copy " .
                    escapeshellarg($command['dst']) .
                    " " .
                    escapeshellarg($command['src']) .
                    " ";
            } else {
                $args .= "remove " . escapeshellarg($command['dst']) . " ";
            }
        }
        exec("failsafe-data " . $args, $out, $status);
        return $status == 0;
    }

    function getReportFile($serial, $ext = false)
    {
        header("Content-type: application/x-gzip");
        if ($ext) {
            header(
                'Content-Disposition: attachment; filename="spxreport-ext-' .
                    $serial .
                    '.tar.gz"'
            );
        } else {
            header(
                'Content-Disposition: attachment; filename="spxreport-' .
                    $serial .
                    '.tar.gz"'
            );
        }

        if (PHP_OS == "WINNT") {
            sleep(3);
            echo file_get_contents("C:\srv\spxreport.tar.gz");
        } elseif ($ext) {
            passthru("/usr/sbin/spxreport -x", $ret);
        } else {
            passthru("/usr/sbin/spxreport", $ret);
        }
    }

    function checkSSH()
    {
        $ret = -1;
        if (PHP_OS == "WINNT") {
            if (
                file_exists("/srv/ssh") &&
                file_get_contents("/srv/ssh") == "start"
            ) {
                $ret = 0;
            }
        } else {
            exec(escapeshellarg($this->ssh_init) . ' status', $dummy, $ret);
        }
        return $ret === 0;
    }

    function manageSSH($start = true)
    {
        if (PHP_OS == "WINNT") {
            file_put_contents("/srv/ssh", $start ? "start" : "stop");
            sleep(3);
        }
        exec(
            escapeshellarg($this->ssh_init) . ($start ? ' start' : ' stop'), 
            $output, $ret);
        return $ret === 0;
    }
    function resetFactory()
    {
        syslog(LOG_NOTICE, 'requested reset to factory defaults');
        $actions = "";
        if (file_exists(self::$boot_config)) {
            $actions = file_get_contents(self::$boot_config);
        }
        $actions .= "reset\n";

        return file_put_contents(self::$boot_config, $actions) !== false;
    }
    function formatStorage()
    {
        syslog(LOG_NOTICE, 'requested to format storage');
        $actions = "";
        if (file_exists(self::$boot_config)) {
            $actions = file_get_contents(self::$boot_config);
        }
        $actions .= "format\n";

        return file_put_contents(self::$boot_config, $actions) !== false;
    }
    function clearWebstorage()
    {
        syslog(LOG_NOTICE, 'clearing webstorage');
        return touch('/var/spool/raperca/clean-webstorage');
    }
    function resetContent()
    {
        syslog(LOG_NOTICE, 'resetting content to default');
        exec('/usr/libexec/spxmanage/content-reset', $output, $ret);
        return $ret === 0;
    }
    function clearInterfaceContent()
    {
        syslog(LOG_NOTICE, 'clearing interface content');
        return touch('/var/spool/spxmanage/reset-interface-data');
    }

    static function enableARYAMode($enabled = null)
    {
        if ($enabled === null) {
            return unlink(self::$aryafile);
        }
        $config = file_exists(self::$aryafile)
            ? file_get_contents(self::$aryafile)
            : "";

        if (preg_match("/^\s*ENABLED=(.*)/m", $config, $matches)) {
            $config = preg_replace(
                "/^\s*ENABLED\s*=.*/m",
                "ENABLED=" . ($enabled ? "yes" : "no"),
                $config
            );
        } else {
            if ($config !== "") {
                $config .= "\n";
            }

            $config .= "ENABLED=" . ($enabled ? "yes" : "no");
        }
        if ($enabled) {
            // make sure content from the interface is not displayed
            if (file_exists(self::$interfaceindexfile)) {
                unlink(self::$interfaceindexfile);
            }
        }

        return file_put_contents(self::$aryafile, $config) !== false;
    }
    static function enableCloudConnection($enabled = null)
    {
        return self::updateConfig(self::$enrolfile, 'ENABLE_ENROLLMENT', 
            is_null($enabled) ? '' : ($enabled ? 'yes' : 'no'));
    }

    static function enableCloudDebug($enabled = null)
    {
        return self::updateConfig(self::$enrolfile, 'ENABLE_DEBUG', 
            $enabled ? 'yes' : '');
    }

    static function configureEnrollUrl($url = null)
    {
        if ($url) {
            $url = escapeshellarg($url);
        }
        return self::updateConfig(self::$enrolfile, 'BASE_ENROLL_URL', $url);
    }

    static function enableIoTDebug($enabled = null)
    {
        return self::updateConfig(self::$iotfile, 'ENABLE_DEBUG', 
            $enabled ? 'yes' : '');
    }

    function clearLogs()
    {
        syslog(LOG_NOTICE, 'clearing logs in /srv/raperca/log/');
        if (PHP_OS == "WINNT") {
            rename(
                "/srv/raperca/log/",
                "/srv/raperca/log-del-" . uniqid() . "/"
            );
            mkdir("/srv/raperca/log/");
            return;
        }
        exec('rm -rf /srv/raperca/log/*', $output, $ret);

        $pid = trim(file_get_contents('/var/run/raperca.pid'));
        posix_kill($pid, SIGUSR1);

        $pid = trim(file_get_contents('/var/run/uploader.pid'));
        posix_kill($pid, SIGUSR1);

        return $ret === 0;
    }
    function clearStreaming()
    {
        syslog(LOG_NOTICE, 'clearing streaming captures');
        if (PHP_OS == "WINNT") {
            rename(
                "/srv/raperca/capture/",
                "/srv/raperca/capture-del-" . uniqid() . "/"
            );
            mkdir("/srv/raperca/capture/");
            return;
        }
        exec('rm -rf /srv/raperca/capture/*', $output, $ret);
        return $ret === 0;
    }
    function clearCache()
    {
        syslog(LOG_NOTICE, 'requesting to clean cache');
        if (touch('/var/spool/raperca/clean-cache')) {
            exec('sync'); // make sure the clean-cache file is on stable storage
            return true;
        } else {
            return false;
        }
    }
    function clearNtp()
    {
        syslog(LOG_NOTICE, 'requesting to reset NTP data');
        return touch('/var/spool/spxutils/ntpreset');
    }
    function clearWebPageData()
    {
        syslog(LOG_NOTICE, 'clearing web page data');
        if (touch('/var/spool/raperca/clean-web-page-data')) {
            exec('sync'); // make sure the file is on stable storage
            return true;
        } else {
            return false;
        }
    }
    static function reloadHttpd()
    {
        # Push everything to http server before reloading the server's config
        ob_flush();
        flush();

        syslog(LOG_NOTICE, 'restarting httpd');
        exec("httpd -k graceful", $output, $ret);
        return $ret === 0;
    }
    static function triggerEnrollment()
    {
        syslog(LOG_NOTICE, 'trigger re-enrollment ');
        exec(
            "dbus-send --system --type=method_call --print-reply --reply-timeout=10000 --dest=com.spinetix.SpxPlayer.Enroller1 /com/spinetix/SpxPlayer/Enroller1 com.spinetix.SpxPlayer.Enroller1.trigger",
            $output,
            $ret
        );
        return $ret === 0;
    }

    static function setRpcPollFallbackTimes(
        $service,
        $concentratorId,
        $interval = 0,
        $startDelay = 0,
        $duration = 0
    ) {
        syslog(LOG_NOTICE, 'set rpc poll fallback times');

        exec(
        		join(" ", array(
        				"dbus-send", "--system", "--reply-timeout=5000", "--print-reply", "--dest=com.spinetix.SpxPlayer.IoT1",
        				"/com/spinetix/SpxPlayer/IoT1", "com.spinetix.SpxPlayer.IoT1.RPCNotifier.setRpcPollFallbackTimes",
        				escapeshellarg("string:$service"), escapeshellarg("string:$concentratorId"), escapeshellarg("uint32:$interval"),
        				escapeshellarg("uint32:$startDelay"), escapeshellarg("uint32:$duration")
        				)),
            $output,
            $ret
        );
        return $ret === 0;
    }

    // Returns TRUE on success and an error string on failure
    static function shutdown(
        $msg,
        $recovery = false,
        $delay = false,
        $halt = false
    ) {
        $opt = $halt ? "-h" : "-r";
        if ($msg == "") {
            $msg = "user initiated shutdown : unknown reason";
        }

        if ($delay == false) {
            exec(
                'exec shutdown ' . escapeshellarg($opt) . ' now 2>&1',
                $out,
                $ret
            );
            if ($ret > 1) {
                return "failed to spawn shutdown command";
            } elseif ($ret != 0) {
                return "a shutdown or restart is already in progress";
            }
        } else {
            // the shutdown command requires a + to prefix a delay (in minutes)
            $delay = '+' . $delay;
            if (file_exists('/var/run/shutdown.pid')) {
                $pid = file_get_contents('/var/run/shutdown.pid');
            } else {
                $pid = false;
            }
            if ($pid !== false) {
                $pid = trim($pid);
                if (is_numeric($pid) && posix_kill($pid, 0)) {
                    return "a shutdown or restart is already in progress";
                }
            }
            exec(
                "exec shutdown " .
                    escapeshellarg($opt) .
                    " " .
                    escapeshellarg($delay) .
                    " &> /dev/null < /dev/null &",
                $out,
                $ret
            );
            if ($ret != 0) {
                return "failed to spawn shutdown command";
            }
        }
        if ($recovery) {
            exec('/etc/init.d/bootcount noreset; bootcount write 100');
        }
        syslog(LOG_NOTICE, $msg);
        return true;
    }

    static function scanDBUS()
    {
        $result = array(
            "Network" => array(),
            "KnownNetwork" => array(),
            "Scan" => array(),
        );
        $d = new Dbus(Dbus::BUS_SYSTEM, false /* useIntrospection */);
        $p = $d->createProxy(
            "net.connman.iwd",
            "/",
            "org.freedesktop.DBus.ObjectManager"
        );
        foreach ($p->GetManagedObjects()->getData() as $path => $obj) {
            $object = array();
            $object['path'] = $path;
            $object['interface'] = array();
            $networkName = "";
            foreach ($obj->getData() as $iface => $obj_iface) {
                $o2 = array();
                $o2['iface'] = $iface;
                $o2['data'] = array();
                $names = explode(".", $iface);
                foreach ($obj_iface->getData() as $name => $value) {
                    $o3 = array();
                    $v = $value->getData();
                    if (is_scalar($v)) {
                        $o2['data'][$name] = $v;
                    } elseif ($v instanceof DbusObjectPath) {
                        $o2['data'][$name] = $v->getData();
                    } else {
                        $o2['data'][$name] = $v;
                    }
                }
                $object['interface'][] = $o2;

                if (count($names) === 4 && $names[0] === "net") {
                    if (
                        $names[3] === "Network" ||
                        $names[3] === "KnownNetwork"
                    ) {
                        $result[$names[3]][$path] = $o2['data'];
                        $result[$names[3]][$path]["proxy"] = $d->createProxy(
                            "net.connman.iwd",
                            $path,
                            $iface
                        );
                    } elseif ($names[3] === "Station") {
                        $result["Station"] = $o2['data'];
                        $result["Station"]["proxy"] = $d->createProxy(
                            "net.connman.iwd",
                            $path,
                            $iface
                        );
                    } else {
                        $result[$names[3]] = $o2['data'];
                    }
                }
            }
            $result['Scan'][] = $object;
        }
        return $result;
    }

    public static function scanWifi()
    {
        //$iwd = self::createIwdProxy();
        $data = self::scanDBUS();
        if (!isset($data['Station'])) {
            throw new Exception("Station not found");
        }
        try {
            $data['Station']['proxy']->Scan();
        } catch (Exception $e) {
            throw new Exception("Start scan failed");
        }
    }

    public static function getWifiInfo()
    {
        $result = array(
            "networks" => array(),
            "knownNetworks" => array(),
            "stationState" => array(),
        );
        //$iwd = self::createIwdProxy();
        $data = self::scanDBUS();
        if (!isset($data['Station'])) {
            throw new Exception("Station not found");
        }
        try {
            $networks = $data['Station']['proxy']->GetOrderedNetworks();
        } catch (Exception $e) {
            throw new Exception("Start scan failed");
        }
        foreach ($networks->getData() as $info) {
            $path = $info->getData()[0]->getData();
            $signal = $info->getData()[1];
            if (isset($data['Network'][$path])) {
                $result["networks"][] = array(
                    'signal' => $signal,
                    'name' => $data['Network'][$path]['Name'],
                    'type' => $data['Network'][$path]['Type'],
                    'connected' => $data['Network'][$path]['Connected'],
                );
            }
        }
        if (isset($data['KnownNetwork'])) {
            $result["knownNetworks"] = array();
            foreach ($data['KnownNetwork'] as $info) {
                $result["knownNetworks"][] = array(
                    'name' => $info['Name'],
                    'type' => $info['Type'],
                    'hidden' => $info['Hidden'],
                    'autoConnect' => $info['AutoConnect'],
                    'lastConnectedTime' => $info['LastConnectedTime'],
                );
            }
        }
        $result["stationState"]["state"] = $data['Station']['State'];
        $result["stationState"]["scanning"] = $data['Station']['Scanning'];
        if (
            isset($data['Station']['ConnectedNetwork']) &&
            isset($data['Network'][$data['Station']['ConnectedNetwork']])
        ) {
            $info = $data['Network'][$data['Station']['ConnectedNetwork']];
            $result["stationState"]["connectedNetwork"] = array(
                'name' => $info['Name'],
                'type' => $info['Type'],
            );
        }
        return $result;
    }

    static function queryEnrollment()
    {
        $d = new Dbus(Dbus::BUS_SYSTEM, false /* useIntrospection */);
        $p = $d->createProxy(
            "com.spinetix.SpxPlayer.Enroller1", // connection name
            "/com/spinetix/SpxPlayer/Enroller1", // object
            "org.freedesktop.DBus.Properties" // interface
        );
        return array(
            "enrolled" => !!$p
                ->Get("com.spinetix.SpxPlayer.Enroller1", "enrolled")
                ->getData(),
            "enabled" => !$p
                ->Get("com.spinetix.SpxPlayer.Enroller1", "disabled")
                ->getData(),
        );
    }
    static function licensecheck($context = null, $revoke = false)
    {
        if ($revoke) {
            $args = "-R";
        } else {
            $args = "-d";
        }
        if ($context) {
            $fname = tempnam("/tmp", "context");
            if (file_put_contents($fname, json_encode($context)) === false) {
                throw new Exception("Internal error: Saving context failed");
            }
            $args .= " -A " . $fname;
        }
        exec("licensecheck " . $args, $out, $status);

        if ($status !== 0) {
            throw new Exception("Internal error: Check failed");
        }

        $json = implode("\n", $out);

        return json_decode($json, true);
    }

    static function licenseupdate($license = null)
    {
        $licensefile = "/etc/spinetix/licenses";

        if (file_exists($licensefile)) {
            $conf = trim(file_get_contents($licensefile));
            if (trim($conf) === "") {
                $conf = null;
            }
        } else {
            $conf = null;
        }

        $featuresFile = '/run/licensecheck/license-features.json';
        $lastModificationTime = @filemtime($featuresFile);

        if ($license === null) {
            if ($conf === null) {
                return true;
            }
            if (file_exists($licensefile)) {
                unlink($licensefile);
            }
        } else {
            if ($conf == trim($license)) {
                return true;
            }
            Tools::save_file($licensefile, trim($license) . "\n");
        }

        // Wait at most 100 * 50000 microseconds (5 seconds)
        for ($maxWait = 100; $maxWait; $maxWait--) {
            usleep(50000);
            clearstatcache();
            if (@filemtime($featuresFile) !== $lastModificationTime) {
                break;
            }
        }
        if (!$maxWait) {
            syslog(
                LOG_WARNING, 
                "Timeout waiting for {$featuresFile} to change"
            );
        }

        return true;
    }

    private static function updateConfig($file, $name, $value)
    {
        $config = file_get_contents($file);
        if (preg_match("/{$name}\s*=.*/", $config)) {
            $config = preg_replace(
                "/^\h*{$name}\s*=.*/m",
                "{$name}=${value}",
                $config
            );
        } else {
            $config .= "\n{$name}=${value}\n";
        }
        if (self::saveFile($file, $config) === true) {
            return true;
        } else {
            syslog(LOG_ERR, "Failed to save {$file}");
            return false;
        }
    }
}
