#!/bin/bash

#
# Copyright (C) 2017 SpinetiX
#

NAME=bootspx.sh

#
# Assorted initialization things specific to the normal firmware,
# run late in rcS.
# It should run after bootmisc.sh
#

RCINIT=/usr/share/resources/default/init/spxsysconf

# Load resources
[ -f "$RCINIT" ] && . "$RCINIT"

# The location where to mount the persistent data store
PERSISTENT_DATA_MPT=/var/lib/spinetix/persistent-data
# The name of the persistent data partition (MTD)
PERSISTENT_DATA_PNAME=persistent-data
# Flag file to skip reset of persistent data partition
PERSISTENT_DATA_NORESET=/var/lib/spinetix/persistent-data.noreset

# acquires a lock for the serial port on which the console is running;
# the lock is acquired for init's PID (1) since it as long lived as
# the kernel. It also sets the closing_wait parameter to 100 (i.e. 1
# second) so that if the serial port is suspended due to flow control
# it does not hang the shutdown process (default is 30 seconds).
lock_serial_console() {
    local port

    port="$(sed -e 's/.*\bconsole=\(tty[A-Z][[:alnum:]]*\).*/\1/p;d' < /proc/cmdline)"
    [ -z "$port" ] && return

    echo -n "Registering $port for system console: lock"
    lockdev -l -p1 "$port" || echo -n " failed"
    case "$port" in
        ttyO*)
            # OMAP serial ports fail with setserial (it returns "invalid argument")
            ;;
        *)
            echo -n ", "
            echo -n "port settings"
            setserial /dev/"$port" closing_wait 100 || echo -n " failed"
            ;;
    esac
    echo "."
}

# verifies that if a ssh host key exists then the public key pair
# exists and both have non-zero size, if the check fails then the keys
# are removed so that they are generated when sshd starts.
verify_ssh_keys() {
    local kf
    for kf in /etc/ssh/ssh_host_*_key ; do
        [ -f "$kf" ] || continue
        [ -s "$kf" -a -s "$kf".pub ] && continue
        rm -f "$kf" "$kf".pub
    done
}

mount_persistent_data_mtd() {
    local mtdnr
    local eret
    local dev path junk

    if ! [ -d "$PERSISTENT_DATA_MPT" ]; then
        echo 'missing mount point for persistent data storage'
        return 1
    fi
    while read dev path junk; do
        if [ "$dev" = mtd:"$PERSISTENT_DATA_PNAME" ]; then
             echo 'persistent data storage already mounted at '"$path"
             return 0
        fi
    done < /proc/mounts
    # originally the mount point dir was readable to all, fix it up if needed
    [ "$(stat -c %a "$PERSISTENT_DATA_MPT")" = 700 ] || chmod 700 "$PERSISTENT_DATA_MPT"
    if mount -t jffs2 mtd:"$PERSISTENT_DATA_PNAME" "$PERSISTENT_DATA_MPT"; then
        echo 'mounted persistent data storage'
        # originally the root dir of the persistent-data fs was readable to all, fix it up if needed
        [ "$(stat -c %a "$PERSISTENT_DATA_MPT")" = 700 ] || chmod 700 "$PERSISTENT_DATA_MPT"
        return 0
    fi

    echo "persistent data storage mount failed, attempting reset"

    mtdnr="$(sed -ne '/\"'"$PERSISTENT_DATA_PNAME"'\"$/{s/mtd\([0-9]\+\).*/\1/;p}' /proc/mtd)"
    if [ -z "$mtdnr" ]; then
        echo "persistent data storage partition $PERSISTENT_DATA_PNAME not found"
        return 1
    fi
    if [ -f "$PERSISTENT_DATA_NORESET" ]; then
        echo "persistent data storage reset already done and unsuccessful"
        return 1
    fi
    flash_erase -q -j /dev/mtd"$mtdnr" 0 0
    eret=$?
    touch "$PERSISTENT_DATA_NORESET"
    fsync "${PERSISTENT_DATA_NORESET%/*}" 2>/dev/null || sync
    if ! [ $eret -eq 0 ]; then
        echo "persistent data storage reset failed"
        return 1
    fi
    mount -t jffs2 mtd:"$PERSISTENT_DATA_PNAME" "$PERSISTENT_DATA_MPT"
    if [ $? -eq 0 ]; then
        echo "persistent data storage reset successful, mounted"
        chmod 700 "$PERSISTENT_DATA_MPT" || echo "failed setting persistent data storage as private"
        return 0
    fi
    echo "persistent data storage mount failed after reset"
    return 1
}

mount_persistent_data_bdev() {
    local mret
    local dev path junk

    if ! [ -d "$PERSISTENT_DATA_MPT" ]; then
        echo 'missing mount point for persistent data storage'
        return 1
    fi
    [ -z "$SPX_PERSISTENT_DATA_BDEV" ] && [ -f /etc/spinetix/identifiers ] && . /etc/spinetix/identifiers
    if [ -z "$SPX_PERSISTENT_DATA_BDEV" ]; then
        echo 'no block device for persistent storage'
        return 1
    fi
    while read dev path junk; do
        if [ "$dev" = "$SPX_PERSISTENT_DATA_BDEV" ]; then
             echo 'persistent data storage already mounted at '"$path"
             return 0
        fi
    done < /proc/mounts
    fsck.ext4 -p "$SPX_PERSISTENT_DATA_BDEV" < /dev/null
    if [ $? -le 1 ] && mount -t ext4 -o noatime "$SPX_PERSISTENT_DATA_BDEV" "$PERSISTENT_DATA_MPT"; then
        echo 'mounted persistent data storage'
        return 0
    fi

    echo "persistent data storage filesystem check or mount failed, attempting reset"

    if ! [ -b "$SPX_PERSISTENT_DATA_BDEV" ]; then
        echo "the persistent data block device '$SPX_PERSISTENT_DATA_BDEV' is not a block device"
        return 1
    fi
    if [ -f "$PERSISTENT_DATA_NORESET" ]; then
        echo "persistent data storage reset already done and unsuccessful"
        return 1
    fi
    # NOTE: the ext4 "64bit" feature is necessary to enable full-strength checksumming
    wipefs --quiet --all "$SPX_PERSISTENT_DATA_BDEV" && \
        mkfs.ext4 -q -b 1024 -O metadata_csum,64bit -m 0 -L persistent-data "$SPX_PERSISTENT_DATA_BDEV"
    mret=$?
    touch "$PERSISTENT_DATA_NORESET"
    fsync "${PERSISTENT_DATA_NORESET%/*}" 2>/dev/null || sync
    if ! [ $mret -eq 0 ]; then
        echo "persistent data storage reset failed"
        return 1
    fi
    mount -t ext4 -o noatime "$SPX_PERSISTENT_DATA_BDEV" "$PERSISTENT_DATA_MPT"
    if [ $? -eq 0 ]; then
        echo "persistent data storage reset successful, mounted"
        chmod 700 "$PERSISTENT_DATA_MPT" || echo "failed setting persistent data storage as private"
        sync
        return 0
    fi
    echo "persistent data storage mount failed after reset"
    return 1
}

# Mounts the persistent-data storage. If mounting fails it attempts to reset
# the storage partition and mount it again, but only if such a reset has not
# been attempted before (this avoids attempting a reset on each reboot, which
# would cause flash wear).
mount_persistent_data() {
    if [ "$HOSTTYPE" = "arm" ]; then
        mount_persistent_data_mtd
    else
        mount_persistent_data_bdev
    fi
}

# Set Intel Energy Performance Bias
set_energy_performance_bias() {
    local file has_epb

    for file in /sys/devices/system/cpu/cpu*/power/energy_perf_bias; do
        [ -f "$file" ] || continue
        has_epb=1
        # Set EPB to value NORMAL=6
        echo "6" > "$file"
    done
    if [ -n "$has_epb" ]; then
        echo "ENERGY_PERF_BIAS set to normal (6)"
    fi
}

set_msr_register() {
    if [ -n "$rc_x86_msr_address" ] && [ -n "$rc_x86_msr_new_value" ]; then
        echo "$rc_x86_msr_name (MSR $rc_x86_msr_address) set to value from branding"
        wrmsr "$rc_x86_msr_address" "$rc_x86_msr_new_value"
    fi
}

case "$1" in
    start|restart)
        lock_serial_console
        verify_ssh_keys
        mount_persistent_data
        set_energy_performance_bias
        set_msr_register
        ;;
    stop|reload|force-reload)
        ;;
    *)
esac
