<?php

class NetworkSource extends CModel
{
    var $name;
    var $uri;
    var $resourceId;
    public function attributeNames()
    {
        return array("name", "uri");
    }
    public function rules()
    {
        $res = array(
            array(
                'name',
                'length',
                'max' => 128,
                'min' => 3,
                "allowEmpty" => false,
            ),
            array(
                "uri",
                'SimpleURLValidator',
                'relativePath' => true,
                "allowEmpty" => false,
            ),
            array("resourceId", 'safe'),
        );
        return $res;
    }
    public function attributeLabels()
    {
        return array(
            'name' => 'Friendly name',
            'uri' => 'Server URI',
        );
    }
}

class SourceConfig extends CFormModel
{
    var $mediaMountFile = "/etc/default/media-mount";

    var $useUSB = null;
    var $playerConf = null;
    var $message = "";

    var $liveSources = array();

    public function rules()
    {
        $res = array(
            array("liveSources", 'validateSource'),
            array("useUSB", 'forceInt'),
        );
        return $res;
    }
    public function forceInt($attribute, $params)
    {
        if ($this->$attribute !== null) {
            $this->$attribute = intval($this->$attribute);
        }
    }
    public function validateSource($attribute, $params)
    {
        $value = $this->$attribute;
        if (!is_array($value)) {
            $this->addError($attribute, "not an array");
            return;
        }
        foreach ($value as $source) {
            if (!$source->validate()) {
                if ($source->getError('name')) {
                    $this->addError($attribute, $source->getError('name'));
                }
                if ($source->getError('uri')) {
                    $this->addError($attribute, $source->getError('uri'));
                }
            }
        }
    }
    public function attributeLabels()
    {
        return array(
            'useUSB' => 'Use external USB drive to store content',
        );
    }

    function getErrors($attribute = null)
    {
        if ($this->playerConf !== null) {
            return array_merge(
                parent::getErrors($attribute),
                $this->playerConf->getErrors($attribute)
            );
        } else {
            return parent::getErrors($attribute);
        }
    }
    function resetLiveSource()
    {
        $this->liveSources = array();
    }
    function addLiveSource($source)
    {
        $this->liveSources[] = $source;
    }
    function load()
    {
        $svgNS = 'http://www.w3.org/2000/svg';
        $xlinkNS = 'http://www.w3.org/1999/xlink';
        if ($this->playerConf === null) {
            $this->playerConf = new PlayerConfig();
        }
        if (!$this->playerConf->loadConfig()) {
            return false;
        }

        $config = $this->playerConf->domdoc;

        $this->liveSources = array();
        $mediaSources = $config->getElementsByTagName('virtualHost');
        foreach ($mediaSources as $mediaSource) {
            if (
                substr($mediaSource->getAttribute("host"), 0, 11) ==
                "spx://Live-"
            ) {
                $source = new NetworkSource();
                $source->name = $mediaSource->getAttribute("name");
                if (empty($source->name)) {
                    // backward compatibility
                    $source->name = substr(
                        $mediaSource->getAttribute("host"),
                        11
                    );
                }
                $uri = $mediaSource->getAttributeNS($xlinkNS, "href");
                $source->resourceId = $mediaSource->getAttribute("resourceId");
                if (substr($uri, 0, 28) === 'file:///srv/raperca/content/') {
                    $uri = substr($uri, 28);
                }
                $source->uri = $uri;

                $this->liveSources[] = $source;
            }
        }
        // usb
        $mediaMount = file_get_contents($this->mediaMountFile);
        if (preg_match("/INTERNAL_USB_STORAGE=(.*)/", $mediaMount, $matches)) {
            $this->useUSB = $matches[1] == 'yes';
        }
    }

    function save()
    {
        $svgNS = 'http://www.w3.org/2000/svg';
        $xlinkNS = 'http://www.w3.org/1999/xlink';

        if ($this->playerConf === null) {
            $this->playerConf = new PlayerConfig();
        }
        if (!$this->playerConf->loadConfig()) {
            return false;
        }
        $config = $this->playerConf->domdoc;

        // Clean virtual hosts
        $cleanResourceId = array();
        $mediaSources = $config->getElementsByTagName('virtualHost');
        $mediaSourcesList = array();
        foreach ($mediaSources as $mediaSource) {
            $mediaSourcesList[] = $mediaSource;
        }
        foreach ($mediaSourcesList as $mediaSource) {
            if (
                substr($mediaSource->getAttribute("host"), 0, 10) ==
                "spx://Live"
            ) {
                if ($mediaSource->hasAttribute("resourceId")) {
                    $cleanResourceId[] = $mediaSource->getAttribute(
                        "resourceId"
                    );
                }
                $mediaSource->parentNode->removeChild($mediaSource);
            }
        }

        $parent = $config->getElementsByTagName('resource');
        if ($parent->length < 1) {
            $parent = $config->appendChild($config->createElement('resource'));
        } else {
            $parent = $parent->item(0);
        }

        // live source

        foreach ($this->liveSources as $source) {
            $uri = $source->uri;
            if (
                substr($source->uri, 0, 7) != "http://" &&
                substr($source->uri, 0, 8) != "https://"
            ) {
                // local folder
                $uri = "file:///srv/raperca/content/" . $uri;
            }
            if (substr($uri, -1) != '/') {
                $uri .= "/";
            }
            $model = null;
            if ($source->resourceId) {
                // update the resource
                $model = Resource::loadModel(
                    'external',
                    $source->resourceId,
                    false
                );
            }
            if ($model) {
                $key = array_search($source->resourceId, $cleanResourceId);
                if ($key !== false) {
                    unset($cleanResourceId[$key]);
                }
                $model->loadSource();
            } else {
                // create a new resource
                $model = Resource::newModel('external');
            }
            if ($model) {
                $model->delayPreview = true;
                $simplifiedName = preg_replace(
                    "/[^a-zA-Z\d-]/",
                    "",
                    $source->name
                );
                $host = "spx://Live-" . $simplifiedName . "-" . uniqid();
                $model->sourceHref = $host . "/index.svg";
                $model->sourceFullPath = $uri . "index.svg";
                $model->name = $source->name;
                $model->keywords = array('live');
                $model->clearPreviews();
                $transaction = $model->getDbConnection()->beginTransaction();
                try {
                    if (!$model->save()) {
                        throw new Exception("save resource failed");
                    }
                    $model->fixRights();
                    $transaction->commit();
                } catch (Exception $e) {
                    // an exception is raised if a query fails
                    $transaction->rollback();
                    foreach ($model->errors as $error) {
                        $this->addError('liveSources', $error);
                    }
                    $this->addError('liveSources', $e->getMessage());
                    continue;
                }
                $source->resourceId = $model->id;

                $vHost = $config->createElement('virtualHost');
                $vHost->setAttribute("host", $host);
                $vHost->setAttribute("resourceId", $source->resourceId);
                $vHost->setAttribute("name", $source->name);
                $vHost->setAttributeNS($xlinkNS, "xlink:href", $uri);
                $vHost->setAttribute("xml:id", "Live-" . $simplifiedName);
                $parent->appendChild($vHost);
            }
            //$parent->appendChild($config->createTextNode("\n"));
        }
        foreach ($cleanResourceId as $resourceId) {
            $model = Resource::loadModel('external', $resourceId);
            if ($model) {
                if (count($model->parents) > 0) {
                    Tools::addMessage(
                        "Deleted source (" .
                            $model->name .
                            ") has been removed from schedule"
                    );
                }
                $model->delete();
            }
        }

        $this->playerConf->saveConfig(true);

        //usb
        if ($this->useUSB !== null) {
            $mediaMount = file_get_contents($this->mediaMountFile);

            $skip = false;
            if (
                preg_match("/INTERNAL_USB_STORAGE=(.*)/", $mediaMount, $matches)
            ) {
                if ($matches[1] == 'yes' && $this->useUSB) {
                    $skip = true;
                }
                if ($matches[1] != 'yes' && !$this->useUSB) {
                    $skip = true;
                }
            }
            if (!$skip) {
                if ($this->useUSB) {
                    $mediaMount = preg_replace(
                        "/INTERNAL_USB_STORAGE=.*/",
                        "INTERNAL_USB_STORAGE=yes",
                        $mediaMount
                    );
                    $this->message = "<h2>Status:</h2>";
                    $this->message .=
                        "Unplug and replug attached USB storage devices, if any, for the change to take effect. Alternatively you may restart the system for the changes to take effect.<br/><br/>";
                } else {
                    $mediaMount = preg_replace(
                        "/INTERNAL_USB_STORAGE=.*/",
                        "INTERNAL_USB_STORAGE=no",
                        $mediaMount
                    );
                }
                Tools::save_file($this->mediaMountFile, $mediaMount);
                Tools::addReason(
                    ($this->useUSB ? "Enable" : "Disable") . " external USB"
                );
                if (PHP_OS == "WINNT") {
                    if ($this->useUSB) {
                        copy("/proc/mounts-used", "/proc/mounts");
                    } else {
                        copy("/proc/mounts-notused", "/proc/mounts");
                    }
                }
            }
        }
    }
}
