<?php
function hash_pbkdf2_safe(
    $algorithm,
    $password,
    $salt,
    $count,
    $key_length,
    $raw_output
) {
    if (function_exists("hash_pbkdf2")) {
        return hash_pbkdf2(
            $algorithm,
            $password,
            $salt,
            $count,
            $key_length,
            $raw_output
        );
    }

    $hash_length = strlen(hash($algorithm, "", true));
    $block_count = ceil($key_length / $hash_length);

    $output = "";
    for ($i = 1; $i <= $block_count; $i++) {
        // $i encoded as 4 bytes, big endian.
        $last = $salt . pack("N", $i);
        // first iteration
        $last = $xorsum = hash_hmac($algorithm, $last, $password, true);
        // perform the other $count - 1 iterations
        for ($j = 1; $j < $count; $j++) {
            $xorsum ^= $last = hash_hmac($algorithm, $last, $password, true);
        }
        $output .= $xorsum;
    }

    if ($raw_output) {
        return substr($output, 0, $key_length);
    } else {
        return bin2hex(substr($output, 0, $key_length));
    }
}

class ConfigBackup extends CModel
{
    protected $version = "2.3";
    static $resetString = "-#-#-reset-#-#-";
    static $passphraseFile = "/var/lib/spinetix/persistent-data/config-crypt.key";
    protected $passphraseSaltBase64 = 'JSxnSuxD8NA=';
    var $backupPassword = true;
    var $backupNTP = true;
    var $passphrase;
    protected $_cachedPassphrase;
    protected $_cachedSalt;
    protected $_privatePassphrase;
    protected $_privateSalt;

    var $shutdownDelay = false;
    var $uploadedfile;
    var $needreload = 0;
    static $passphrase_map = array(
        'passphrase-check' => array('passphraseDigest', 'encrypt')
    );
    static $wizard_map = array('wizard' => array('active', true));
    static $display_map = array(
        'output' => 'output',
        'display-custom-video-mode' => 'custom',
        'custom' => 'custom',
        'resolution' => 'resolution',
        'vertical-freq' => 'vfreq',
        'force-std-mode' => array('forcestdmode', true),
        'restrict-type' => 'type',
        'screen-aspect-ratio' => 'screeenar',
        'underscan-supported' => array('underscansupp', true),
        'hdmi-link-type' => 'hdmilink',
    );
    static $display_map2 = array(
        'overscan-percentage' => 'overscan',
        'display-orientation' => 'orientation',
        'video-output-selector' => 'videoOutputSelector',
        'enable-audio' => array('audio', true),
        'audio-output' => 'audioOutput',
        'volume' => 'volume',
        'audio-power-save' => array('audioPowerSave', true),
        'canvas' => 'canvas',
    );
    static $splash_map = array(
        'boot' => 'boot',
        'shutdown' => 'shutdown',
        'fimrware' => 'fimrware',
    );
    static $power_map = array(
        'display-power-save' => array('enablePowerSave', true),
        'on-time' => 'onTime',
        'off-time' => 'offTime',
        'sunday-on-time' => 'sundayOn',
        'sunday-off-time' => 'sundayOff',
        'monday-on-time' => 'mondayOn',
        'monday-off-time' => 'mondayOff',
        'tuesday-on-time' => 'tuesdayOn',
        'tuesday-off-time' => 'tuesdayOff',
        'wednesday-on-time' => 'wednesdayOn',
        'wednesday-off-time' => 'wednesdayOff',
        'thursday-on-time' => 'thursdayOn',
        'thursday-off-time' => 'thursdayOff',
        'friday-on-time' => 'fridayOn',
        'friday-off-time' => 'fridayOff',
        'saturday-on-time' => 'saturdayOn',
        'saturday-off-time' => 'saturdayOff',
    );
    static $interactivity_map = array(
        'accepts-events' => array('events', true),
        'maximum-latency' => 'latency',
        'reduce-interactive-latency' => array('interlatency', true),
        'overlay' => 'overlay',
        'enable-USB-IO-API' => array('usbIOAPI', true),
    );
    static $network_v4_map = array(
        'network-interface' => 'netiface',
        'dhcp' => array('dhcp', true),
        'address' => 'address',
        'netmask' => 'netmask',
        'gateway' => 'gateway',
    );
    static $network_v6_map = array(
        'address' => 'ipv6Address',
        'prefix' => 'ipv6Netmask',
        'gateway' => 'ipv6Gateway',
    );
    static $dns_map = array(
        'dns' => 'nameserver1',
        'dns2' => 'nameserver2',
        'dns3' => 'nameserver3',
        'domain' => 'domain',
    );
    static $slaac_map = array('disable-slaac' => array('disableSLAAC', true));
    static $id_map = array(
        'screen-id' => 'screenId',
        'device-name' => 'deviceName',
    );
    static $netapi_map = array(
        'network-api-enabled-secure' => array('NetAPIEnabled', true),
        'network-api-enabled' => array('LegacyNetAPIEnabled', true),
        'network-api-port' => 'sharedVarPort',
        'encrypted-network-api-psk' => array('sharedVarPSK', 'encrypt'),
        'network-api-psk' => array('sharedVarPSK', 'nosave'),
        'network-api-server' => 'sharedVarDefaultServer',
    );
    static $https_map = array(
        'https-validate-certificates' => array('HTTPSUseDefaultCA', true),
    );
    static $watchdog_map = array(
        'max-time-no-addr' => 'maxTimeNoAddr',
        'min-time-keep-addr' => 'minTimeKeepAddr',
    );
    static $credit_map = array(
        'server' => 'href',
        'username' => 'username',
        'encrypted-password' => array('password', 'encrypt'),
        'realm' => 'realm',
    );
    static $proxy_map = array(
        'server' => 'server',
        'port' => 'port',
        'bypass-proxy-local' => 'bypassLocalNetwork',
        'username' => 'username',
        'password' => 'password',
    );
    static $snmp_map = array(
        'snmp-rocommunity' => 'rocommunity',
        'address-range' => 'snmpaddress',
    );
    static $source_map = array('allow-usb-sources' => array('useUSB', true));
    static $time_map = array('timezone' => 'shortlink');
    static $manualtime_map = array(
        'manual' => true,
        'time' => 'time',
        'date' => 'date',
    );
    static $ntpserver_map = array(
        'ntp' => true,
        'boot-delay' => 'bootdelay',
        'server-1' => 'ntpserver1',
        'server-2' => 'ntpserver2',
        'server-3' => 'ntpserver3',
        'server-4' => 'ntpserver4',
        'server-5' => 'ntpserver5',
        'monitor-only-1' => 'ntpnosel1',
        'monitor-only-2' => 'ntpnosel2',
        'monitor-only-3' => 'ntpnosel3',
        'monitor-only-4' => 'ntpnosel4',
        'monitor-only-5' => 'ntpnosel5',
    );
    static $temperature_map = array('shutdown-temperature' => 'max_temp');
    static $ports_map = array(
        'disable-secondary-network-port' => array('disablePort2', true),
    );
    static $io_map = array(
        'protocol-name' => 'backupName', // backward compatibility only
        'protocol-file' => 'backupName',
        'protocol-system' => 'backupName',
        'enabled' => array('enableSerial', true),
        'port-number' => 'portNumber',
        'device' => 'device',
        'name' => 'name',
        'baud-rate' => 'baudRate',
        'byte-size' => 'byteSize',
        'parity' => 'parity',
        'stop-bits' => 'stopBits',
    );
    static $capture_device_map = array(
        'index' => 'index',
        'name' => 'name',
        'path' => 'path',
    );
    static $cockpit_map = array(
        'uri' => 'cockpitUri',
        'options' => 'cockpitOptions',
        'time' => 'cockpitTime',
    );
    static $pullmode_map = array(
        'project' => true,
        'logs' => true,
        'rpc' => true,
        'mode' => 'selectMode',
        'push-logs' => array('enable_log', true),
        'log-source' => 'log_source',
        'log-time' => 'timeLog',
        'log-type' => 'log_type',
        'schedule-uri' => 'icalSourceHref',
        'ics-file' => 'icsfile',
        'check-frequency' => 'checktime',
        'rpc-concentrator' => array('enable_rpc', true),
        'uri' => 'concentrator',
        'polling' => 'polling',
        'pooling' => 'polling', // backward compatibility
        'notification-only' => array('notification_only', true),
    );
    static $syncsource_map = array(
        "dest" => "dest",
        "no-subfolders" => array('norecurs', true),
        "uri" => "uri",
        "update" => "update",
        "time" => "time",
    );
    static $livesource_map = array(
        "name" => "name",
        "uri" => "uri",
    );
    static $firmware_map = array(
        'firmware-update-uri' => 'updateuri',
        'firmware-update-auto' => 'updateAutoString',
    );
    static $touch_map = array(
        'xx' => 'xx',
        'xy' => 'xy',
        'tx' => 'tx',
        'yx' => 'yx',
        'yy' => 'yy',
        'ty' => 'ty',
    );
    static $streaming_map = array(
        'protocol' => 'protocol',
        'resolution' => 'resolution',
        'frame-rate' => 'frameRate',
        'quality' => 'quality',
        'codec' => 'codec',
        'rtp' => array('RTP', true),
        'ip' => 'IP',
        'port' => 'port',
        'multicast' => array('multicast', true),
        'group' => 'group',
        'url' => 'URL',
        'base64-xml' => 'base64XML',
    );
    // WebRTC
    static $webrtc_map = array(
        'echo-cancellation' => array('echoCancellation', true),
    );
    static $security_map = array(
        'secure-admin' => array('admin', true),
        'secure-content' => array('content', true),
        'secure-monitoring' => array('monitoring', true),
    );
    static $advance_config_map = array(
        'update-fps' => 'update',
        'min-fps-step-factor' => 'step',
    );
    static $upnp_map = array('ssdp-upnp-enabled' => array('enabled', true));
    static $llmnr_map = array('llmnr-enabled' => array('enabled', true));
    static $bonjour_map = array(
        'bonjour-enabled' => array('enabledAvahi', true),
        'bonjour-discovery-enabled' => array('enabledDiscovery', true),
    );
    static $security_rpc_map = array('rpc-api-key' => 'APIKey');
    static $webstorage_api_map = array(
        'webstorage-token' => array('token', 'nosave'),
        'encrypted-webstorage-token' => array('token', 'encrypt'),
    );
    static $server_certificate_map = array(
        'pkcs12' => array('pfxText', 'nosave'),
        'certificate' => 'certificateText',
        'privatekey' => array('keyText', 'nosave'),
        'encrypted-privatekey' => array('keyText', 'encrypt'),
        'privatekey-passphrase' => array('passphrase', 'nosave'),
        'ca-chain' => array('chainText', 'noempty'),
        'name' => 'uid',
    );
    static $server_security_map = array(
        'http-server-security-access' => "level",
        'http-publish-secure-only' => array('secureDAVPortOnly', true),
    );

    static $license_map = array('license-file' => 'license');

    static $capture_map = array('http-capture-log' => 'httpLogs');

    static $wifi_map = array(
        'ssid' => 'ssid',
        'method' => 'method',
        'hidden' => array('hidden', true),
        'username' => 'username',
        'encrypted-password' => array('password', 'encrypt'),
        'password' => array('password', 'nosave'),
        'pem' => 'pem',
        'encrypted-file' => array('file', 'encrypt'),
        'file' => array('file', 'nosave'),
        'base64-file' => 'fileBase64',
    );
    static $wifi_map_save = array(
        'ssid' => 'ssid',
        'method' => 'method',
        'encrypted-file' => array('file', 'encrypt'),
    );
    static $wired_auth_map = array(
        'network' => 'network',
        'method' => 'method',
        'username' => 'username',
        'encrypted-password' => array('password', 'encrypt'),
        'password' => array('password', 'nosave'),
        'pem' => 'pem',
        'encrypted-file' => array('file', 'encrypt'),
        'file' => array('file', 'nosave'),
        'base64-file' => 'fileBase64',
    );
    static $wired_auth_map_save = array(
        'network' => 'network',
        'method' => 'method',
        'encrypted-file' => array('file', 'encrypt'),
    );

    static $logger_map = array(
        'name' => 'name',
        'level' => 'level',
    );
 
    static $iot_map = array(
        "debug-enroll" => array('enrollDebug', true),
        "enroll-base-url" => array('url'),
        "debug-iot" => array('iotDebug', true),
    );

    /**
     * Recursively removes a directory and all its contents.
     */
    static function cleanupdir($dir)
    {
        // implementing a recursive directory cleanup in PHP is tricky due
        // to symlink following, better delegate it to well-known shell
        // commands
        if ($dir && is_dir($dir)) {
            if (PHP_OS == "WINNT") {
                exec("rd /s /q " . escapeshellarg($dir));
            } else {
                exec("rm -rf " . escapeshellarg($dir));
            }
        }
    }

    /**
     * Creates a new private temporary directory, with a random name which
     * does not yet exist, which is automatically cleaned up at script end.
     */
    static function mktempdir()
    {
        if (PHP_OS == "WINNT") {
            $parent = sys_get_temp_dir();
        } else {
            $parent = '/var/tmp';
        }
        $created = false;
        $n = 0;
        do {
            $tempdir = $parent . DIRECTORY_SEPARATOR . substr(md5(random_bytes(8)), 0, 8);
            $created = @mkdir($tempdir, 0700); // silence generated E_WARNING if dir exists
        } while (!$created && $n++ < 10);
        if (!$created) {
            die("failed creating temporary directory: ". error_get_last()['message']);
        }
        register_shutdown_function('ConfigBackup::cleanupdir', $tempdir);
        return $tempdir;
    }

    protected $dom;
    public function attributeNames()
    {
        return array('uploadedfile', 'backupPassword');
    }
    public function attributeLabels()
    {
        return array(
            'uploadedfile' => 'Backup',
            'backupPassword' => 'Include passwords and other secrets',
            'backupNTP' => 'Include time server configuration',
            'passphrase' => 'Passphrase',
            'hasPassphrase' => 'Please select',
        );
    }
    public function rules()
    {
        $res = array(
            array(
                'uploadedfile',
                'file',
                'types' => 'spx, xml, 7z, zip, cfg',
                'on' => 'upload',
            ),
            array('backupPassword', 'safe'),
            array(
                'passphrase',
                'match',
                'pattern' => "/^[-a-zA-Z0-9 #!%&*+,.:;=_]*$/",
            ),
        );
        return $res;
    }
    function saveInternalPassphrase()
    {
        if (empty($this->_privatePassphrase)) {
            $this->_privatePassphrase = openssl_random_pseudo_bytes(32);
            $this->_privateSalt = openssl_random_pseudo_bytes(8);
        }
        $data = array(
            'privatePassphrase' => bin2hex($this->_privatePassphrase),
            'privateSalt' => base64_encode($this->_privateSalt),
        );
        if ($this->_cachedPassphrase !== "") {
            $data['userPassphrase'] = bin2hex($this->_cachedPassphrase);
            $data['userSalt'] = base64_encode($this->_cachedSalt);
        }
        Tools::save_file(
            self::$passphraseFile,
            json_encode($data),
            true,
            null,
            false
        );
    }
    function loadInternalPassphrase()
    {
        if (file_exists(self::$passphraseFile)) {
            $data = json_decode(file_get_contents(self::$passphraseFile), true);
            if (
                $data !== null &&
                is_array($data) &&
                isset($data['privatePassphrase'])
            ) {
                $this->_privatePassphrase = hex2bin($data['privatePassphrase']);
                $this->_privateSalt = base64_decode($data['privateSalt']);
                if (isset($data['userPassphrase'])) {
                    $this->_cachedPassphrase = hex2bin($data['userPassphrase']);
                    $this->_cachedSalt = base64_decode($data['userSalt']);
                } else {
                    $this->_cachedPassphrase = "";
                }
                return;
            }
        }
        // create new one
        $this->_privatePassphrase = "";
        $this->_cachedPassphrase = "";
        $this->saveInternalPassphrase();
    }
    function getInternalPassphrase()
    {
        if ($this->_privatePassphrase === null) {
            $this->loadInternalPassphrase();
        }
        if ($this->_cachedPassphrase == "") {
            return $this->_privatePassphrase;
        } else {
            return $this->_cachedPassphrase;
        }
    }
    function getInternalSalt()
    {
        if ($this->_privatePassphrase === null) {
            $this->loadInternalPassphrase();
        }
        if ($this->_cachedPassphrase == "") {
            return $this->_privateSalt;
        } else {
            return $this->_cachedSalt;
        }
    }
    function encrypt($clearText, $attribute = "passphrase")
    {
        $hashMethod = "pbkdf2";
        $hashAlgo = "sha256";
        $password = $this->getInternalPassphrase();
        if (empty($password)) {
            $this->addError($attribute, "Passphrase not set.");
            return false;
        }
        $salt = $this->getInternalSalt();
        $iterations = 10000;

        $key = hash_pbkdf2_safe(
            $hashAlgo,
            $password,
            $salt,
            $iterations,
            32,
            true
        );

        $cryptAlgo = "aes-256-cbc";
        $iv_size = openssl_cipher_iv_length($cryptAlgo);
        $iv = openssl_random_pseudo_bytes($iv_size);

        $cipherText = openssl_encrypt($clearText, $cryptAlgo, $key, 0, $iv);

        $encryptedText =
            "$hashMethod($hashAlgo," .
            base64_encode($salt) .
            ",$iterations):$cryptAlgo:" .
            base64_encode($iv) .
            ":" .
            $cipherText;

        return $encryptedText;
    }
    function decrypt($encryptedText, $attribute = "passphrase")
    {
        $data = explode(":", $encryptedText);
        if (count($data) < 4) {
            $this->addError(
                $attribute,
                "Encrypted string not valid, missing paramters"
            );
            return "";
        }
        if (
            !preg_match(
                "/^pbkdf2\(sha256,([^,]*),(\d*)\)$/",
                $data[0],
                $matches
            )
        ) {
            $this->addError(
                $attribute,
                "Encrypted string not valid, cannot find key derivation algorithm"
            );
            return "";
        }
        $hashMethod = "pbkdf2";
        $hashAlgo = "sha256";
        $password = $this->getInternalPassphrase();

        $iterations = intval($matches[2], 10);

        $salt = base64_decode($matches[1]);
        $key = hash_pbkdf2_safe(
            $hashAlgo,
            $password,
            $salt,
            $iterations,
            32,
            true
        );

        $cryptAlgo = "aes-256-cbc";
        if ($data[1] != $cryptAlgo) {
            $this->addError(
                $attribute,
                "Encrypted string not valid, cipher algorithm not valid"
            );
            return "";
        }

        $iv = base64_decode($data[2]);

        $cipherText = $data[3];

        $clearText = openssl_decrypt($cipherText, $cryptAlgo, $key, 0, $iv);

        if ($clearText === false) {
            $this->addError($attribute, "Could not decrypt string");
        }
        return $clearText;
    }
    function getHasPassphrase()
    {
        if ($this->_privatePassphrase === null) {
            $this->loadInternalPassphrase();
        }
        return $this->_cachedPassphrase == "" ? 0 : 1;
    }
    function save($runValidation = true, $attributes = null)
    {
        $this->loadInternalPassphrase();

        if ($this->passphrase === self::$resetString) {
            $this->passphrase = "";
            $this->_privatePassphrase = "";
        }
        if ($runValidation && !$this->validate($attributes)) {
            return false;
        }
        if (empty($this->passphrase)) {
            $this->_cachedPassphrase = "";
            if ($this->_privatePassphrase === "") {
                Tools::addMessage("Secret key regenerated sucessfully");
            } else {
                Tools::addMessage("Secret key configured sucessfully");
            }
            $this->saveInternalPassphrase();
        } else {
            $passphraseSalt = base64_decode($this->passphraseSaltBase64);
            $this->_cachedPassphrase = hash_pbkdf2_safe(
                "sha256",
                $this->passphrase,
                $passphraseSalt,
                10000,
                32,
                true
            );
            $this->_cachedSalt = openssl_random_pseudo_bytes(8);
            $this->saveInternalPassphrase();
            Tools::addMessage("Passphrase applied sucessfully");
        }
        Tools::$reload = true;

        $this->passphrase = "";
        return true;
    }
    public function getPassphraseDigest() {
        return hash('sha512', $this->getInternalPassphrase());
    }

    function dump()
    {
        $folder = ConfigBackup::mktempdir() . DIRECTORY_SEPARATOR;
        $fn = $folder . 'config.xml';
        $zip = $folder . "config.7z";

        file_put_contents($fn, $this->create());

        // create the zip file from this
        if (PHP_OS == "WINNT") {
            $exe = '"C:\Program Files\7-Zip\7z.exe"';
        } else {
            $exe = "7za";
        }

        $files = array();

        // splash screens
        $splash = new SplashScreen();
        $splash->load();
        foreach ($splash->filepath as $name => $path) {
            array_push($files, $path);
        }
        // fsm files
        $serial = new SerialConfig();
        $list = $serial->getFileList(false, true);
        foreach ($list as $name => $path) {
            array_push($files, $path);
        }
        $cmd = "$exe a " . escapeshellarg($zip) . " -t7z -mx1 -mhc=off " . escapeshellarg($fn);
        foreach ($files as $idx => $file) {
            if ($file !== null && file_exists($file)) {
                $cmd .= " " . escapeshellarg($file);
            }
        }
        $options = array();
        $ret = -1;
        exec($cmd, $options, $ret);
        if ($ret == 0) {
            $fp = fopen($zip, 'rb');
            header("Content-type: application/x-spx-config");
            header(
                'Content-Disposition: attachment; filename="config-' .
                    Yii::app()->device->serial .
                    '.cfg"'
            );

            fpassthru($fp);
            fclose($fp);
        } else {
            Yii::log(
                "Error generating backup file: $cmd\n" .
                    print_r($options, true),
                'warning',
                'spx.config'
            );
            $this->addError(
                'uploadedfile',
                "Cannot create backup file, please contact support"
            );
        }
        return $ret == 0;
    }
    function isXmlBackup($filename)
    {
        //@$a = simplexml_load_string( $somestring );
        $f = fopen($filename, 'r');
        $start = fread($f, 5);
        if ($start == "<?xml" || $start == "<conf") {
            return true;
        } else {
            return false;
        }
    }
    function applyBackupFile($filename)
    {
        if (PHP_OS == "WINNT") {
            $exe = '"C:\Program Files\7-Zip\7za"';
        } else {
            $exe = "ulimit -v 12288 && 7za";
        }

        if ($this->isXmlBackup($filename)) {
            $data = file_get_contents($filename);
            $this->apply($data);
        } else {
            // zip file
            $folder = ConfigBackup::mktempdir() . DIRECTORY_SEPARATOR;

            $dest = SerialConfig::$confpathusr;
            $cmd = "$exe e " . escapeshellarg($filename) . " " . escapeshellarg("-o$dest") . escapeshellcmd(" -aoa *.fsm");
            exec($cmd, $output, $ret);
            if ($ret == 2 || $ret == 8) {
                $this->addError(
                    'uploadedfile',
                    "The compression factor may be too high. Try re-generating the archive with the compression level set to Store or Fast."
                );
                return;
            } elseif ($ret != 0) {
                $this->addError(
                    'uploadedfile',
                    "File extraction of fsm failed. ($ret)"
                );
                return;
            }

            $cmd = "$exe e " . escapeshellarg($filename) . " " . escapeshellarg("-o$folder") . escapeshellcmd(" -aoa *.jpg *.png *.JPG *.PNG *.7z");
            exec($cmd, $output, $ret);
            if ($ret == 2 || $ret == 8) {
                $this->addError(
                    'uploadedfile',
                    "The compression factor may be too high. Try re-generating the archive with the compression level set to Store or Fast."
                );
                return;
            } elseif ($ret != 0) {
                $this->addError(
                    'uploadedfile',
                    "File extraction of images failed. ($ret)"
                );
                return;
            }

            $cmd = "$exe e " . escapeshellarg($filename) . " " . escapeshellarg("-o$folder") . " -aoa config.xml";
            exec($cmd, $output, $ret);
            if ($ret == 2 || $ret == 8) {
                $this->addError(
                    'uploadedfile',
                    "The compression factor may be too high. Try re-generating the archive with the compression level set to Store or Fast."
                );
                return;
            } elseif ($ret != 0) {
                $this->addError(
                    'uploadedfile',
                    "File extraction of config failed. ($ret)"
                );
                return;
            } else {
                if (file_exists($folder . "config.xml")) {
                    $data = file_get_contents($folder . "config.xml");
                    $this->apply($data, $folder);
                    unlink($folder . "config.xml");
                } else {
                    $this->addError(
                        'uploadedfile',
                        "Cannot find the config.xml in the backup file."
                    );
                    return;
                }
            }
        }
    }

    function setAll(&$main, $name, $obj, $table)
    {
        if ($name === false) {
            $node = $main;
        } else {
            $node = $main->appendChild($this->dom->createElement($name));
        }
        foreach ($table as $dst => $src) {
            if (is_array($src)) {
                if ($src[1] === 'nosave') {
                    continue;
                }
                $src = $src[0];
            }
            if ($src === true) {
                continue;
            }
            $this->set($node, $obj, $src, $table);
        }
        return $node;
    }

    function set(&$node, $obj, $elems, $table = null)
    {
        if (!is_array($elems)) {
            $elems = array($elems);
        }
        foreach ($elems as $elem) {
            if ($table === null) {
                $member = $elem;
                $name = $elem;
            } else {
                foreach ($table as $name => $member) {
                    if (
                        is_array($member) &&
                        $member[0] === $elem &&
                        $member[1] !== 'nosave'
                    ) {
                        break;
                    }
                    if (!is_array($member) && $member === $elem) {
                        break;
                    }
                }
            }
            $bool = false;
            if (is_array($member)) {
                $bool = $member[1];
                $member = $member[0];
            }
            if (is_array($obj)) {
                if (!isset($obj[$member])) {
                    continue;
                }
                $val = $obj[$member];
            } elseif (is_object($obj)) {
                $val = $obj->$member;
            } else {
                $val = $obj;
            }

            if ($bool === true) {
                $node->appendChild(
                    $this->dom->createElement($name, $val ? "yes" : "no")
                );
            } elseif ($bool === "encrypt") {
                if (!empty($val)) {
                    $node->appendChild(
                        $this->dom->createElement($name, $this->encrypt($val))
                    );
                }
            } elseif ($bool === "noempty") {
                if (!empty($val)) {
                    $node->appendChild(
                        $this->dom->createElement($name, htmlspecialchars($val))
                    );
                }
            } elseif ($bool !== "nosave") {
                $node->appendChild(
                    $this->dom->createElement($name, htmlspecialchars($val))
                );
            }
        }
    }
    function convertErrors($errors, $table, $prefix = "")
    {
        $myerr = array();

        foreach ($errors as $elem => $value) {
            $realName = $elem;
            if (!is_array($table)) {
                $realName = $table;
            } else {
                foreach ($table as $name => $member) {
                    if (is_array($member) && $member[0] === $elem) {
                        $realName = $name;
                        break;
                    }
                    if (!is_array($member) && $member === $elem) {
                        $realName = $name;
                        break;
                    }
                }
            }
            $myerr[$prefix . $realName] = $value;
        }
        return $myerr;
    }

    function getAll(&$obj, $scenario, $node, $table, $prefix = "")
    {
        $attr = $this->get($node, $table, $prefix);
        if ($scenario) {
            $obj->setScenario($scenario);
        }
        $obj->setAttributes($attr);
        if (!$obj->validate(null, false)) {
            $this->addErrors(
                $this->convertErrors($obj->getErrors(), $table, $prefix)
            );
            $obj->clearErrors();
        }

        return $attr;
    }
    function setAttr($node, $name, $table, &$attr)
    {
        $member = $table[$name];
        $bool = false;
        if (is_array($member)) {
            $bool = $member[1];
            $member = $member[0];
        }
        if ($member === true) {
            $a = $this->get($node, $table);
            $attr = array_merge($a, $attr);
            $attr[$name] = true;
        } else {
            $val = trim($node->textContent);
            if ($bool === true) {
                $attr[$member] = $val == "yes" ? 1 : 0;
            } elseif ($bool === 'encrypt') {
                $attr[$member] = $this->decrypt($val, $member);
            } elseif ($bool === 'array') {
                if (!isset($attr[$member])) {
                    $attr[$member] = array();
                }
                $attr[$member][] = htmlspecialchars_decode($val);
            } else {
                $attr[$member] = htmlspecialchars_decode($val);
            }
        }
    }

    function get($node, $table, $prefix = "")
    {
        $attr = array();

        $name = $node->localName;
        //echo $name;
        //print_r( $table );

        if (isset($table[$name]) && $table[$name] !== true) {
            $this->setAttr($node, $name, $table, $attr);
        }
        if ($node->firstChild == null) {
            return $attr;
        }

        for (
            $conf = $node->firstChild;
            $conf != null;
            $conf = $conf->nextSibling
        ) {
            if ($conf->nodeType != XML_ELEMENT_NODE) {
                continue;
            }

            $name = $conf->localName;
            if (isset($table[$name])) {
                if ($name !== 'fsm-file') {
                    //exception fort the special case of fsm-file wich is a full xml document inside the current doc.
                    $this->setAttr($conf, $name, $table, $attr);
                }
            } else {
                $this->addError(
                    $prefix . $node->localName . " > " . $conf->localName,
                    'Unknown command item'
                );
            }
        }
        //print_r( $attr );
        return $attr;
    }
    function create()
    {
        $backup = new DOMDocument();
        $backup->preserveWhiteSpace = false;
        $backup->formatOutput = true;

        $this->dom = $backup;

        $main = $backup->appendChild($backup->createElement('configuration'));
        $main->setAttribute("version", $this->version);
        if (count(Yii::app()->branding->getFeatures()) > 0) {
            $features = $main->appendChild(
                $backup->createElement('features-check')
            );
            foreach (Yii::app()->branding->getFeatures() as $feature) {
                $features->appendChild(
                    $backup->createElement("feature", $feature)
                );
            }
            if (Yii::app()->branding->hasValidLicense()) {
                $features->appendChild(
                    $backup->createElement(
                        "license-type",
                        Yii::app()->branding->getLicense()
                    )
                );
            }
        }

        // need to open the different components to get all the needed info
        if ($this->backupPassword) {
            // passphrase
            if ($this->getHasPassphrase()) {
                $main->appendChild(
                    $backup->createComment(
                        "\n" .
                            "     A passphrase has been configured on this player.\n" .
                            "     To restore this configuration backup on another player,\n" .
                            "     the same passphrase must be already configured on that player.\n" .
                            "     This can be done using Control Center or alternatively uncomment\n" .
                            "     the passphrase element below replacing 'Your Passphrase' with\n" .
                            "     the actual passphrase.\n  "
                    )
                );
                $main->appendChild(
                    $backup->createComment(
                        " <passphrase>Your passphrase</passphrase> "
                    )
                );
            }
            $this->setAll($main, false, $this, self::$passphrase_map);
        }

        // wizard
        $wiz = new Wizard();
        $this->setAll($main, false, $wiz, self::$wizard_map);

        // player config
        $config = new PlayerConfig();
        $config->backupMode = true;

        // display
        $screen = new ScreenSettings();
        $screen->load();
        $display = $main;

        if ($screen->modes) {
            $settings = $display->appendChild(
                $backup->createElement('display-video-mode')
            );
        }
        foreach ($screen->modes as $output => $mode) {
            $settings = $display->appendChild(
                $backup->createElement('display-video-mode')
            );
            if ($output !== 'default') {
                $this->set($settings, $output, 'output');
            }
            if ($mode->resolution === 'native') {
                $this->set($settings, $mode, 'resolution');
            } elseif ($mode->custom && $mode->advancedtype !== "builtin") {
                $this->set($settings, $mode, 'custom');
            } else {
                $this->set(
                    $settings,
                    $mode,
                    array('resolution', 'vfreq', 'forcestdmode', 'type'),
                    self::$display_map
                );
            }
        }

        $this->set($display, $screen, 'underscansupp', self::$display_map);
        $this->set($display, $screen, 'hdmilink', self::$display_map);

        $this->set($display, $config, 'overscan', self::$display_map2);
        $this->set($display, $config, 'orientation', self::$display_map2);
        if (Yii::app()->branding->hasVideoOutputSelector) {
            $this->set(
                $display,
                $config,
                'videoOutputSelector',
                self::$display_map2
            );
        }
        $this->set($display, $config, 'audio', self::$display_map2);
        $this->set($display, $config, 'audioOutput', self::$display_map2);
        $this->set($display, $config, 'volume', self::$display_map2);
        $this->set($display, $config, 'audioPowerSave', self::$display_map2);
        $this->set($display, $config, 'canvas', self::$display_map2);

        //interactivity
        if (Yii::app()->branding->hasRight('interactivity')) {
            $this->set(
                $main,
                $config,
                array('events', 'latency', 'interlatency', 'overlay'),
                self::$interactivity_map
            );
            if ($config->events) {
                $touchCalibration = new TouchCalibration();
                $touchCalibration->playerConf = $config;
                $touchCalibration->load();
                $this->setAll(
                    $main,
                    'touchscreen',
                    $touchCalibration,
                    self::$touch_map
                );
            }
        }
        if (Yii::app()->branding->hasRight('peripheralsUSBAPI')) {
            $this->set(
                $main,
                $config,
                array('usbIOAPI'),
                self::$interactivity_map
            );
        }

        // splash screens
        $splash = new SplashScreen();
        $splash->load();
        foreach ($splash->filepath as $name => $path) {
            $spl = $main->appendChild($backup->createElement("splash"));
            $this->set($spl, $name, 'name');
            if ($path !== null) {
                $this->set($spl, basename($path), 'file');
            }
        }

        // network
        $network = new IPConfig();
        $network->load();
        $networkxml = $main;
        $net_prefix = $network->netiface . "-";

        $ethernet = $networkxml; //->appendChild($backup->createElement('network-ethernet'));
        if (!$network->dhcp) {
            $fix = $ethernet->appendChild(
                $backup->createElement($net_prefix . 'static')
            );
            $this->set(
                $fix,
                $network,
                array('address', 'netmask', 'gateway'),
                self::$network_v4_map
            );
        } else {
            $fix = $ethernet->appendChild(
                $backup->createElement($net_prefix . 'dhcp')
            );
        }
        if ($network->ipv6Configuration === "static") {
            $this->setAll(
                $main,
                $net_prefix . 'v6-static',
                $network,
                self::$network_v6_map
            );
        } else {
            $main->appendChild(
                $backup->createElement(
                    $net_prefix . 'v6-' . $network->ipv6Configuration
                )
            );
        }
        if ($network->dnsConfiguration === "manual") {
            $this->setAll($main, 'dns-manual', $network, self::$dns_map);
        } else {
            $main->appendChild($backup->createElement('dns-automatic'));
        }

        $this->set($main, $config, 'httpLogs', self::$capture_map);

        //SLAAC
        $this->setAll($main, false, $network, self::$slaac_map);

        //identification
        $this->set($main, $config, array('deviceName'), self::$id_map);
        if (Yii::app()->branding->hasRight('multiscreen')) {
            $this->set($main, $config, array('screenId'), self::$id_map);
        }
        //net-api
        if (Yii::app()->branding->hasRight('netAPI')) {
            if ($config->getNetAPIEnabled()) {
                if ($config->getLegacyNetAPIEnabled()) {
                    $this->setAll($main, false, $config, self::$netapi_map);
                } else {
                    $this->set(
                        $main,
                        $config,
                        array(
                            'NetAPIEnabled',
                            'LegacyNetAPIEnabled',
                            'sharedVarPSK',
                            'sharedVarDefaultServer',
                        ),
                        self::$netapi_map
                    );
                }
            } else {
                $this->set(
                    $main,
                    $config,
                    array('NetAPIEnabled', 'sharedVarDefaultServer'),
                    self::$netapi_map
                );
            }
        }
        //CORS security
        if (Yii::app()->branding->hasRight('netAPI')) {
            $securityRPC = new SecurityRPC();
            $securityRPC->load();
            if (!$securityRPC->enableCORS) {
                $main->appendChild($backup->createElement('rpc-api-key'));
            } else {
                $this->setAll(
                    $main,
                    false,
                    $securityRPC,
                    self::$security_rpc_map
                );
            }
        }
        if (Yii::app()->branding->hasRight('netAPI')) {
            $webstorageAPIConfig = new WebstorageAPIConfig();
            $webstorageAPIConfig->load();
            if (!$webstorageAPIConfig->enableToken) {
                $main->appendChild($backup->createElement('webstorage-token'));
            } else {
                $this->setAll(
                    $main,
                    false,
                    $webstorageAPIConfig,
                    self::$webstorage_api_map
                );
            }
        }
        //https
        $this->setAll($main, false, $config, self::$https_map);

        $main->appendChild($backup->createElement('certificates-reset'));
        $ssl = new SSLConfig();
        foreach ($ssl->getFileList(false) as $file) {
            $cert_xml = $main->appendChild(
                $backup->createElement('certificate')
            );
            $cert_xml->appendChild(
                $backup->createCDATASection(
                    $ssl->backupCertificate($file['filename'])
                )
            );
        }
        if ($this->backupPassword) {
            $main->appendChild(
                $backup->createElement('server-certificates-reset')
            );
        }
        $activeId = null;
        $serverCertificates = new ServerCertificates();
        foreach ($serverCertificates->getCertificatesList(false) as $file) {
            $serverCertificates->loadForBackup(
                $file['uid'],
                $this->backupPassword
            );
            $this->setAll(
                $main,
                'server-certificate',
                $serverCertificates,
                self::$server_certificate_map
            );
            if ($file['activeCertificates']) {
                $activeId = $file['uid'];
            }
        }

        $main->appendChild(
            $backup->createElement('active-certificate', $activeId)
        );

        if ($this->backupPassword && Yii::app()->branding->hasRight('WIFI')) {
            $main->appendChild($backup->createElement('wifi-ap-reset'));
            $wifiList = WifiConfig::listAll();
            foreach ($wifiList as $wifiConf) {
                $wifiConf->load();
                $node = $this->setAll(
                    $main,
                    'wifi-ap-add',
                    $wifiConf,
                    self::$wifi_map_save
                );
            }
        }

        if (
            $this->backupPassword &&
            Yii::app()->device->model !== 'ikebana'
        ) {
            $main->appendChild($backup->createElement('wired-auth-reset'));
            $wiredAuthList = WiredAuth::listAll();
            foreach ($wiredAuthList as $wiredAuthConf) {
                $wiredAuthConf->load();
                $node = $this->setAll(
                    $main,
                    'wired-auth-add',
                    $wiredAuthConf,
                    self::$wired_auth_map_save
                );
            }
        }
        // server security
        $serverSecurity = new ServerSecurity();
        $serverSecurity->load();
        $this->setAll(
            $main,
            false,
            $serverSecurity,
            self::$server_security_map
        );

        //net-watchdog
        if (Yii::app()->branding->hasRight('networkWatchdog')) {
            $watchdog = new NetworkWatchdog();
            $watchdog->load();
            $this->setAll(
                $main,
                'network-watchdog',
                $watchdog,
                self::$watchdog_map
            );
        }
        //credential
        if ($this->backupPassword) {
            $credit = new CredentialsConfig();
            $credit->load();
            $main->appendChild($backup->createElement('credential-reset'));
            foreach ($credit->credits as $server) {
                $node = $this->setAll(
                    $main,
                    'credential',
                    $server,
                    self::$credit_map
                );
            }
        }
        //bonjour
        $bonjourConfig = new BonjourConfig();
        $bonjourConfig->load();
        $this->setAll($main, false, $bonjourConfig, self::$bonjour_map);

        //upnp
        $UPnPConfig = new UPnPConfig();
        $UPnPConfig->load();
        $this->setAll($main, false, $UPnPConfig, self::$upnp_map);

        // llmnr
        $LLMNRConfig = new LLMNRConfig();
        $LLMNRConfig->load();
        $this->setAll($main, false, $LLMNRConfig, self::$llmnr_map);

        //proxy
        $proxy = new ProxyConfig();
        $proxy->load();
        if ($proxy->server != "") {
            $this->setAll($main, 'proxy', $proxy, self::$proxy_map);
        } else {
            $this->setAll($main, 'proxy', $proxy, array('server' => 'server'));
        }

        //snmp
        $snmp = new SNMPConfig();
        $snmp->load();
        $snmpxml = $main->appendChild(
            $backup->createElement('snmp-' . $snmp->access)
        );
        if ($snmp->access == 'limited') {
            $this->set($snmpxml, $snmp, 'snmpaddress', self::$snmp_map);
        }
        $this->set($main, $snmp, 'rocommunity', self::$snmp_map);

        //sources
        $sources = new SourceConfig();
        $sources->playerConf = $config;
        $sources->load();
        $this->setAll($main, false, $sources, self::$source_map);

        // users
        if ($this->backupPassword) {
            $users = User::model()->findAll();
            $sec = $main; //->appendChild($backup->createElement('security'));
            $main->appendChild($backup->createElement('user-reset'));
            foreach ($users as $user) {
                $entry = $sec->appendChild($backup->createElement("user"));
                $entry->appendChild(
                    $backup->createElement("username", $user->username)
                );
                $entry->appendChild(
                    $backup->createElement(
                        'encrypted-crypt-password',
                        $this->encrypt($user->password)
                    )
                );
                if ($user->getHasRSP()) {
                    $entry->appendChild(
                        $backup->createElement(
                            'encrypted-srp',
                            $this->encrypt($user->srpPassword)
                        )
                    );
                }
                if ($user->digestPassword) {
                    $entry->appendChild(
                        $backup->createElement(
                            'encrypted-digest',
                            $this->encrypt($user->digestPassword)
                        )
                    );
                }
                $rights = $user->userRights->getUserRights();
                if ($rights['admin']) {
                    $entry->appendChild($backup->createElement("admin", "yes"));
                } else {
                    foreach ($rights as $name => $valid) {
                        if ($name != 'advanced') {
                            $entry->appendChild(
                                $backup->createElement(
                                    $name,
                                    $valid ? "yes" : "no"
                                )
                            );
                        }
                    }
                }
            }
        }
        if (Yii::app()->branding->hasRight('security')) {
            $securitySettings = new SecuritySettings();
            $this->setAll($main, false, $securitySettings, self::$security_map);
        }
        // timezone
        $time = new TimeSettings();
        $time->load();
        $this->setAll($main, false, $time, self::$time_map);

        //time
        $ntp = new NTPSettings();
        $ntp->load();
        $ntptime = $main; //->appendChild($backup->createElement('time'));
        if (!$ntp->active) {
            $ntptime->appendChild($backup->createElement('time-manual'));
        } else {
            $n = $ntptime->appendChild($backup->createElement('time-ntp'));
            $this->setAll($n, false, $ntp, self::$ntpserver_map);
        }

        // temperature
        if (Yii::app()->device->model === 'ikebana') {
            $hwd = new HardwareSetup();
            $hwd->load();
            $this->setAll($main, false, $hwd, self::$temperature_map);
            if (Yii::app()->branding->product == 'HMP350') {
                $this->setAll($main, false, $hwd, self::$ports_map);
            }
        }

        // power
        $power = new Power();
        $power->playerConf = $config;
        $power->load();
        $this->set($main, $power, 'enablePowerSave', self::$power_map);

        if ($power->fixedMonPower !== "no") {
            $set = $main->appendChild(
                $backup->createElement('display-power-schedule')
            );
            $this->set(
                $set,
                $power,
                $power->fixedMonPower === "daily"
                    ? array('onTime', 'offTime')
                    : array(
                        "sundayOn",
                        "sundayOff",
                        "mondayOn",
                        "mondayOff",
                        "tuesdayOn",
                        "tuesdayOff",
                        "wednesdayOn",
                        "wednesdayOff",
                        "thursdayOn",
                        "thursdayOff",
                        "fridayOn",
                        "fridayOff",
                        "saturdayOn",
                        "saturdayOff",
                    ),
                self::$power_map
            );
        } else {
            $set = $main->appendChild(
                $backup->createElement('display-power-schedule')
            );
        }

        // serial-io
        $serial = new SerialConfig();
        $serial->playerConf = $config;
        $serial->load();
        if (Yii::app()->branding->hasRight('peripheralsFSM')) {
            if (count($serial->getFileList(false)) === 0) {
                $main->appendChild(
                    $backup->createElement('protocol-file-reset')
                );
            }
            if ($serial->backupType != "disabled") {
                if ($serial->backupType == "inline") {
                    $this->set($main, $serial->backupName, 'protocol-file');
                } elseif ($serial->backupType == "simple") {
                    $protocolSimple = $main->appendChild(
                        $backup->createElement('protocol-simple')
                    );
                    foreach ($serial->simpleCommands as $simpleFSMCommand) {
                        $commandNode = $protocolSimple->appendChild(
                            $backup->createElement('command')
                        );
                        $commandNode->appendChild(
                            $backup->createElement(
                                'name',
                                htmlentities($simpleFSMCommand['name'])
                            )
                        );
                        $commandNode->appendChild(
                            $backup->createElement(
                                'data',
                                htmlentities($simpleFSMCommand['value'])
                            )
                        );
                    }
                } else {
                    $this->set($main, $serial->backupName, 'protocol-system');
                }
            } else {
                $main->appendChild($backup->createElement('protocol-disabled'));
            }
        }
        if (Yii::app()->branding->hasRight('peripheralsSerial')) {
            $set = $main->appendChild(
                $backup->createElement('com-port-settings')
            );
            if ($serial->enableSerial) {
                $items = array(
                    'baudRate',
                    'byteSize',
                    'parity',
                    'stopBits',
                    'enableSerial',
                );
                if (Yii::app()->device->model !== 'ikebana') {
                    $items[] = "device";
                    $items[] = "name";
                }
                $this->set($set, $serial, $items, self::$io_map);
            } else {
                $this->set($set, $serial, array('enableSerial'), self::$io_map);
            }
        }

        // capture
        if (Yii::app()->branding->hasRight('peripheralsCapture')) {
            $capture = new Capture();
            $capture->loadConfig();
            $node = $main->appendChild($backup->createElement('capture'));
            foreach ($capture->devices as $device) {
                $this->setAll(
                    $node, 'video', $device, self::$capture_device_map);
            }
        }

        // pull mode
        $pull = new PullModeConfig();
        $pull->load();
        $pullxml = $main; //->appendChild($backup->createElement('pull-mode'));

        if ($pull->cockpitUri) {
            $this->setAll($pullxml, "cockpit", $pull, self::$cockpit_map);
        } else {
            $pullxml->appendChild($backup->createElement("cockpit-disabled"));
        }

        if (Yii::app()->branding->hasRight('pullMode')) {
            $staticNode = $pullxml->appendChild(
                $backup->createElement("pull-mode-static")
            );
            $this->set(
                $staticNode,
                $pull,
                array('enable_log'),
                self::$pullmode_map
            );
            if ($pull->enable_log) {
                $lg = $staticNode->appendChild($backup->createElement('logs'));
                $this->set(
                    $lg,
                    $pull,
                    array('log_source', 'timeLog', 'log_type'),
                    self::$pullmode_map
                );
            }
            $this->set(
                $staticNode,
                $pull,
                array('enable_rpc'),
                self::$pullmode_map
            );
            if ($pull->enable_rpc) {
                $rpc = $staticNode->appendChild($backup->createElement('rpc'));
                $this->set(
                    $rpc,
                    $pull,
                    array('concentrator', 'polling', 'notification_only'),
                    self::$pullmode_map
                );
            }
        }
        if (Yii::app()->branding->hasRight('pullMode')) {
            $node = $pullxml->appendChild(
                $backup->createElement("pull-mode-" . $pull->selectMode)
            );
            if ($pull->selectMode == PullModeConfig::PULL_FILE) {
                $n = $node->appendChild($backup->createElement('ics-file'));
                $n->appendChild(
                    $backup->createCDATASection($pull->getSchedule())
                );
            } elseif ($pull->selectMode == PullModeConfig::PULL_REMOTE) {
                $this->set(
                    $node,
                    $pull,
                    array('icalSourceHref', 'checktime'),
                    self::$pullmode_map
                );
            }
        }
        if (Yii::app()->branding->hasRight('pullMode')) {
            $pullxml->appendChild(
                $backup->createElement("scheduled-download-reset")
            );
            foreach ($pull->syncSources as $source) {
                $n = $pullxml->appendChild(
                    $backup->createElement("scheduled-download-add")
                );
                $this->set(
                    $n,
                    $source,
                    array('uri', 'update'),
                    self::$syncsource_map
                );
                if ($source->norecurs) {
                    $this->set(
                        $n,
                        $source,
                        array('norecurs'),
                        self::$syncsource_map
                    );
                }
                if ($source->update == 'daily') {
                    $this->set(
                        $n,
                        $source,
                        array('time'),
                        self::$syncsource_map
                    );
                }
                if (!empty($source->dest)) {
                    $this->set(
                        $n,
                        $source,
                        array('dest'),
                        self::$syncsource_map
                    );
                }
            }
        }

        // live sources
        if (Yii::app()->branding->hasRight('contentLiveSource')) {
            $main->appendChild($backup->createElement("live-source-reset"));
            $sourceConfig = new SourceConfig();
            $sourceConfig->load();
            foreach ($sourceConfig->liveSources as $source) {
                $this->setAll(
                    $main,
                    "live-source-add",
                    $source,
                    self::$livesource_map
                );
            }
        }
        //firmware
        if (Yii::app()->branding->hasRight('firmware')) {
            $firmware = new FirmwareUpdate();
            $firmware->scenario = "backup";
            $firmware->load();
            $this->setAll($main, false, $firmware, self::$firmware_map);
        }

        // streaming
        if (Yii::app()->branding->hasRight('IPTV')) {
            $streaming = new BackupStreaming();
            $streaming->load();
            $this->setAll(
                $main, 'streaming', $streaming->getAttributes(), 
                self::$streaming_map);
        }

        // WebRTC
        if (Yii::app()->branding->hasRight('webRtc')) {
             $this->set(
                $main,
                $config,
                'echoCancellation',
                self::$webrtc_map
            );
        }

        // Debug log
        $log = new DebugLog();
        $log->load();
        $node = $main->appendChild($backup->createElement('debug-log'));
        foreach ($log->loggers as $logger) {
            $this->setAll( $node, 'logger', $logger, self::$logger_map);
        }

        //arya
        if (Yii::app()->branding->hasRight('ARYA')) {
            $ARYAMode = new ARYAMode();
            if ($ARYAMode->getEnabled()) {
                $main->appendChild($backup->createElement('enable-arya'));
            }
        }

        if (!Tools::checkInstallationmode()) {
            $main->appendChild($backup->createElement("reboot"));
        }

        return $backup->saveXML();
    }

    function checkRight($item)
    {
        $map = array(
            'display-custom-video-mode' => 'displayCustom',
            'screen-aspect-ratio' => 'displayAR',
            "network-watchdog" => 'networkWatchdog',
            "live-sources-add" => 'contentLiveSource',
            'com-port-settings' => 'peripheralsSerial',
            "protocol-file-reset" => 'peripheralsFSM',
            "protocol-disabled" => 'peripheralsFSM',
            "protocol-simple" => 'peripheralsFSM',
            "protocol-system" => 'peripheralsFSM',
            "protocol-file" => 'peripheralsFSM',
            //"protocol-system" => 'peripheralsFSMAdvanced',
            //"protocol-file" => 'peripheralsFSMAdvanced',
            "enable-USB-IO-API" => 'peripheralsUSBAPI',
            'capture' => 'peripheralsCapture',
            "accepts-events" => 'interactivity',
            "maximum-latency" => 'interactivity',
            "reduce-interactive-latency" => 'interactivity',
            'overlay' => 'interactivity',
            "touchscreen" => 'interactivity',
            "network-api-enabled" => 'netAPI',
            "network-api-port" => 'netAPI',
            IPConfig::NET_WIFI . "-static" => 'WIFI',
            IPConfig::NET_WIFI . "-dhcp" => 'WIFI',
            IPConfig::NET_WIFI . "-v6-none" => 'WIFI',
            IPConfig::NET_WIFI . "-v6-static" => 'WIFI',
            "wifi-ap-reset" => 'WIFI',
            "wifi-ap-add" => 'WIFI',
            "rpc-api-key" => 'netAPI',
            "screen-id" => 'multiscreen',
            //'firmware-update-uri' => 'firmware',
            //'firmware-update-auto' => 'firmware',
            'pull-mode-' . PullModeConfig::PULL_STATIC => 'pullMode',
            'pull-mode-' . PullModeConfig::PULL_FILE => 'pullMode',
            'pull-mode-' . PullModeConfig::PULL_REMOTE => 'pullMode',
            'scheduled-download-add' => 'pullMode',
            'streaming' => 'IPTV',
            // WebRTC
            'echo-cancellation' => 'webRtc',
            'secure-admin' => 'security',
            'secure-content' => 'security',
            'secure-monitoring' => 'security',
        );

        if (
            isset($map[$item]) &&
            !Yii::app()->branding->hasRight($map[$item])
        ) {
            $this->addError($item, "Not supported by this model");
            return false;
        }
        return true;
    }
    function updateErrors(&$model, $item, $prefix = "")
    {
        $errors = $model->getErrors();
        $this->addErrors($this->convertErrors($errors, $item, $prefix));
        $model->clearErrors();
    }
    function apply($backupstr, $tmpfolder = "")
    {
        $backup = new DOMDocument();
        try {
            $ok = @$backup->loadXML($backupstr);
        } catch (Exception $e) {
            $this->addError(
                '',
                'Cannot parse backup file: ' . $e->getMessage()
            );
        }
        if (!$ok) {
            $this->addError('', 'Cannot parse backup file');
            return;
        }
        $main = $backup->documentElement;

        $version = $main->getAttribute("version");
        $f1a = explode(".", $version);
        $f2a = explode(".", $this->version);
        $versionOk = $f1a[0] == $f2a[0];
        if (!$versionOk) {
            $compatible = $main->getAttribute("compatible");
            if ($compatible) {
                $list = explode(";", $compatible);
                foreach ($list as $item) {
                    $value = explode(".", $item);
                    if ($value[0] == $f2a[0]) {
                        $versionOk = true;
                    }
                }
            }
        }
        if (!$versionOk) {
            $this->addError(
                '',
                'Backup version is not corect (' .
                    $f1a[0] .
                    "!=" .
                    $f2a[0] .
                    ')'
            );
            return;
        }

        $config_file = new PlayerConfig();

        $wiz = new Wizard();
        $welcome = $wiz->getWelcome();

        $screen = null;
        $splash = null;
        $power = null;
        $serial = null;
        $capture = null;
        $network = null;
        $watchdog = null;
        $credit = null;
        $proxy = null;
        $snmp = null;
        $sources = null;
        $securitySettings = null;
        $time = null;
        $ntp = null;
        $hwd = null;
        $pull = null;
        $pullModeConfigLogs = null;
        $firmware = null;
        $ssl = new SSLConfig();
        $serverCertificates = new ServerCertificates();
        $touchCalibration = null;
        $streaming = null;
        $bonjourConfig = null;
        $UPnPConfig = null;
        $LLMNRConfig = null;
        $securityRPC = null;
        $webstorageAPI = null;
        $serverSecurity = null;
        $log = null;
        $ARYAMode = null;
        $iot = null;
        $reboot = false;
        $playerLicenses = null;
        $licenseType = null;
        $failedFeatures = array();
        $maintenance = new Maintenance();

        $check_screen_config = false;
        $checkMultiOutput = false;

        for (
            $conf = $main->firstChild;
            $conf != null;
            $conf = $conf->nextSibling
        ) {
            if ($conf->nodeType != XML_ELEMENT_NODE) {
                continue;
            }
            if (!$this->checkRight($conf->localName)) {
                continue;
            }
            if (count($failedFeatures) > 0 && $licenseType) {
                Tools::$reload = false;
                $this->addError(
                    "license-type",
                    $licenseType . ' license required'
                );
                break;
            }
            switch ($conf->localName) {
                case 'features-check':
                    $vals = $this->get($conf, array(
                        'feature' => array('feature', 'array'),
                        'license-type' => 'license-type',
                    ));
                    if (count($vals['feature']) > 0) {
                        foreach ($vals['feature'] as $feature) {
                            if (strpos($feature, '#') === 0) {
                                continue;
                            }
                            if (
                                !in_array(
                                    $feature,
                                    Yii::app()->branding->getFeatures()
                                )
                            ) {
                                array_push($failedFeatures, $feature);
                            }
                        }
                    }
                    if (isset($vals['license-type'])) {
                        $licenseType = $vals['license-type'];
                    }
                    break;
                case 'regenerate-secret-key':
                    $this->passphrase = self::$resetString;
                    $this->save();
                    break;
                case 'passphrase-check':
                    $passphraseDigest = $this->get(
                        $conf, self::$passphrase_map)['passphraseDigest'];
                    if ($passphraseDigest !== $this->passphraseDigest) {
                        $this->addError(
                            "passphrase-check",
                            'Incorrect passphrase or secret key: The ' .
                            'password or secret key configured on the ' .
                            'player on which the configuration was ' .
                            'generated does not match the one configured on ' .
                            'this player.'
                        );
                        $this->clearErrors('passphraseDigest');
                        return;
                    }
                    break;
                case 'passphrase':
                    $this->setAttributes(
                        $this->get($conf, array('passphrase' => 'passphrase'))
                    );
                    $this->save();
                    break;
                case "wizard":
                    $this->getAll($wiz, false, $conf, self::$wizard_map);
                    break;
                case "display-custom-video-mode":
                case "display-video-mode":
                case "screen-aspect-ratio":
                case "underscan-supported":
                case "hdmi-link-type":
                    if (!$screen) {
                        $screen = new ScreenSettings();
                        $screen->load();
                    }
                    if ($conf->localName === 'display-video-mode'
                        && !$conf->hasChildNodes()
                    ) {
                        $screen->resetMode();
                        $checkMultiOutput = true;
                    } else {
                        $this->getAll($screen, 'backup', $conf, self::$display_map);
                        if ($conf->localName === 'display-video-mode'
                            || $conf->localName === 'display-custom-video-mode'
                        ) {
                            $check_screen_config = true;
                            $checkMultiOutput = true;
                        }
                    }
                    break;
                case "vga-dc-offset":
                    // Deprecated. Ignore parameters
                    break;
                case "overscan-percentage":
                case "display-orientation":
                case "enable-audio":
                case "audio-output":
                case "volume":
                case "audio-power-save":
                case "video-output-selector":
                case "canvas":
                    $this->getAll(
                        $config_file,
                        'display',
                        $conf,
                        self::$display_map2
                    );
                    if ($conf->localName === 'video-output-selector') {
                        $checkMultiOutput = true;
                    }
                    break;
                case "vga-power-mode":
                    // Deprecated. Ignore parameters
                    break;
                case "accepts-events":
                case "maximum-latency":
                case "reduce-interactive-latency":
                case 'overlay':
                case "enable-USB-IO-API":
                    $this->getAll(
                        $config_file,
                        'inter',
                        $conf,
                        self::$interactivity_map
                    );
                    break;
                case "update-fps":
                case "min-fps-step-factor":
                    $this->getAll(
                        $config_file,
                        'player',
                        $conf,
                        self::$advance_config_map
                    );
                    break;
                case "touchscreen":
                    if (!$touchCalibration) {
                        $touchCalibration = new TouchCalibration();
                    }
                    $this->getAll(
                        $touchCalibration,
                        'config',
                        $conf,
                        self::$touch_map
                    );
                    break;
                case "extended-USB":
                    // Deprecated
                    $this->addError(
                        $conf->localName,
                        'Not supported by this model'
                    );
                    break;
                case "splash":
                    if (!$splash) {
                        $splash = new SplashScreen();
                    }
                    $attr = $this->get($conf, array(
                        "name" => "name",
                        "file" => "file",
                    ));
                    if (isset($attr['file'])) {
                        $splash->addFile(
                            $attr['name'],
                            $tmpfolder . $attr['file']
                        );
                    } else {
                        $splash->addFile($attr['name']);
                    }
                    break;
                case IPConfig::NET_ETHERNET . "-static":
                case IPConfig::NET_ETHERNET . "-dhcp":
                case IPConfig::NET_WIFI . "-static":
                case IPConfig::NET_WIFI . "-dhcp":
                    if (!$network) {
                        $network = new IPConfig();
                        $network->load();
                    }
                    $options = explode("-", $conf->localName);
                    $network->netiface = $options[0];

                    if ($options[1] === 'dhcp') {
                        $network->dhcp = true;
                    } else {
                        $network->dhcp = false;

                        $attr = $this->get(
                            $conf,
                            self::$network_v4_map + self::$dns_map
                        );
                        // make sure that default are set to the right values
                        if (!isset($attr['gateway'])) {
                            $attr['gateway'] = "";
                        }
                        // backward compatibility
                        if (
                            isset($attr['nameserver1']) ||
                            isset($attr['domain'])
                        ) {
                            $defaults = array(
                                'nameserver1',
                                'nameserver2',
                                'nameserver3',
                                'domain',
                            );
                            foreach ($defaults as $name) {
                                if (!isset($attr[$name])) {
                                    $attr[$name] = "";
                                }
                            }
                        }
                        $network->setAttributes($attr); // for the fixed IP strings
                    }
                    if (!$network->validate() || !$network->checkConfig()) {
                        $this->updateErrors($network, $conf->localName);
                    }
                    break;
                case IPConfig::NET_ETHERNET . "-v6-none":
                case IPConfig::NET_ETHERNET . "-v6-static":
                case IPConfig::NET_WIFI . "-v6-none":
                case IPConfig::NET_WIFI . "-v6-static":
                    //case "ethernet-v6-dhcp":
                    if (!$network) {
                        $network = new IPConfig();
                        $network->load();
                    }
                    $options = explode("-", $conf->localName, 3);
                    $network->ipv6Configuration = $options[2];
                    $attr = array();
                    if ($network->ipv6Configuration === "static") {
                        $attr = $this->get($conf, self::$network_v6_map);
                        // make sure that default are set to the right values.
                        // Required values are set to "" to trigger an error is missing
                        $defaults = array(
                            'ipv6Address',
                            'ipv6Netmask',
                            'ipv6Gateway',
                        );
                        foreach ($defaults as $name) {
                            if (!isset($attr[$name])) {
                                $attr[$name] = "";
                            }
                        }
                    }
                    $network->setIpv6($options[0], $attr);
                    if (!$network->validate()) {
                        $this->updateErrors($network, $conf->localName);
                    }
                    break;
                case "dns-automatic":
                case "dns-manual":
                    if (!$network) {
                        $network = new IPConfig();
                        $network->load();
                    }
                    $network->dnsConfiguration = substr(
                        $conf->localName,
                        strlen("dns-")
                    );
                    if ($network->dnsConfiguration === "manual") {
                        $attr = $this->get($conf, self::$dns_map);
                        // make sure that default are set to the right values
                        $defaults = array(
                            'nameserver1',
                            'nameserver2',
                            'nameserver3',
                            'domain',
                        );
                        foreach ($defaults as $name) {
                            if (!isset($attr[$name])) {
                                $attr[$name] = "";
                            }
                        }
                        $network->setAttributes($attr); // for the fixed IP strings
                    }
                    if (!$network->validate()) {
                        $this->updateErrors($network, $conf->localName);
                    }
                    break;
                case "modem-3g-simple":
                case "modem-3g-advanced":
                    // Deprecated. Ignore parameters
                    break;
                case "wifi-ap-reset":
                    WifiConfig::clearAll();
                    break;
                case "wifi-ap-add":
                    $wifiConfig = new WifiConfig();
                    $this->getAll($wifiConfig, false, $conf, self::$wifi_map);
                    $wifiConfig->save();
                    if ($wifiConfig->hasErrors()) {
                        $this->updateErrors($wifiConfig, self::$wifi_map);
                    }
                    break;
                case "wired-auth-reset":
                    WiredAuth::clearAll();
                    break;
                case "wired-auth-add":
                    if (Yii::app()->device->model == 'ikebana') {
                        $this->addError(
                            $conf->localName,
                            'Not supported by this model'
                        );
                        break;
                    }
                    $wiredAuth = new WiredAuth();
                    $this->getAll(
                        $wiredAuth,
                        false,
                        $conf,
                        self::$wired_auth_map
                    );
                    $wiredAuth->save();
                    if ($wiredAuth->hasErrors()) {
                        $this->updateErrors($wiredAuth, self::$wired_auth_map);
                    }
                    break;
                case "http-capture-log":
                    $this->getAll(
                        $config_file,
                        'player',
                        $conf,
                        self::$capture_map
                    );
                    break;
                case "disable-slaac":
                    if (!$network) {
                        $network = new IPConfig();
                        $network->load();
                    }
                    $this->getAll($network, null, $conf, self::$slaac_map);
                    break;
                case "screen-id":
                case "device-name":
                    $this->getAll($config_file, 'id', $conf, self::$id_map);
                    break;
                case "network-api-enabled-secure":
                case "network-api-enabled":
                case "network-api-port":
                case "encrypted-network-api-psk":
                case "network-api-psk":
                case "network-api-server":
                    $this->getAll(
                        $config_file,
                        'netapi',
                        $conf,
                        self::$netapi_map
                    );
                    break;
                case "rpc-api-key":
                    if (!$securityRPC) {
                        $securityRPC = new SecurityRPC();
                    }
                    $securityRPC->enableCORS = trim($conf->textContent) !== "";
                    $this->getAll(
                        $securityRPC,
                        null,
                        $conf,
                        self::$security_rpc_map
                    );
                    break;
                case "webstorage-token":
                case "encrypted-webstorage-token":
                    if (!$webstorageAPI) {
                        $webstorageAPI = new WebstorageAPIConfig();
                    }
                    $webstorageAPI->enableToken =
                        trim($conf->textContent) !== "";
                    $this->getAll(
                        $webstorageAPI,
                        null,
                        $conf,
                        self::$webstorage_api_map
                    );
                    break;
                case "https-validate-certificates":
                    $this->getAll(
                        $config_file,
                        'https',
                        $conf,
                        self::$https_map
                    );
                    break;
                case "certificates-reset":
                    $ssl->clearCertificates();
                    break;
                case "certificate":
                    $ssl->restoreCertificate(trim($conf->textContent));
                    if ($ssl->hasErrors()) {
                        $this->updateErrors($ssl, $conf->localName);
                    }
                    break;
                case "server-certificates-reset":
                    $serverCertificates->clearCertificates();
                    break;
                case "server-certificate":
                    $this->getAll(
                        $serverCertificates,
                        'backup',
                        $conf,
                        self::$server_certificate_map
                    );
                    if (!$serverCertificates->restoreBackup()) {
                        $this->updateErrors(
                            $serverCertificates,
                            $conf->localName
                        );
                    }
                    break;
                case "active-certificate":
                    $this->getAll($serverCertificates, 'activate', $conf, array(
                        "active-certificate" => 'uid',
                    ));
                    if (!$serverCertificates->activate()) {
                        $this->updateErrors(
                            $serverCertificates,
                            $conf->localName
                        );
                    }
                    break;
                case 'http-server-security-access':
                case 'http-publish-secure-only':
                    if (!$serverSecurity) {
                        $serverSecurity = new ServerSecurity();
                    }
                    $this->getAll(
                        $serverSecurity,
                        false,
                        $conf,
                        self::$server_security_map
                    );
                    break;
                case "network-watchdog":
                    $watchdog = new NetworkWatchdog();
                    $this->getAll($watchdog, false, $conf, self::$watchdog_map);
                    break;
                case "credential-reset":
                    $credit = new CredentialsConfig();
                    break;
                case "credential":
                    if ($credit === null) {
                        $credit = new CredentialsConfig();
                        $credit->load();
                    }
                    $server = new Credentials();
                    $this->getAll(
                        $server,
                        false,
                        $conf,
                        self::$credit_map + array("password" => "password")
                    );

                    $credit->addCreditential($server);
                    break;
                case "proxy":
                    $proxy = new ProxyConfig();
                    $attr = $this->getAll(
                        $proxy,
                        false,
                        $conf,
                        self::$proxy_map
                    );
                    break;
                case "snmp-local":
                case "snmp-limited":
                case "snmp-all":
                case "snmp-rocommunity":
                    if (!$snmp) {
                        $snmp = new SNMPConfig();
                        $snmp->load();
                    }
                    if ($conf->localName == "snmp-rocommunity") {
                        $this->getAll(
                            $snmp,
                            'community',
                            $conf,
                            self::$snmp_map
                        );
                    } else {
                        $snmp->access = substr($conf->localName, 5);
                        if ($snmp->access == "limited") {
                            $snmp->snmpaddress = "";
                        } // reset the address to make it compulsory
                        $this->getAll(
                            $snmp,
                            'backup-' . $snmp->access,
                            $conf,
                            self::$snmp_map
                        );
                    }
                    break;
                case "allow-usb-sources":
                    if (!$sources) {
                        $sources = new SourceConfig();
                        $sources->playerConf = $config_file;
                        $sources->load();
                    }
                    $this->getAll($sources, false, $conf, self::$source_map);
                    break;
                case "user-reset":
                    $user = new User();
                    $user->clearAllUsers();
                    if ($user->needreload) {
                        $this->needreload = 1;
                    }
                    break;
                case "user":
                    $username = "";
                    $password = false;
                    $passcrypt = false;
                    $digest = null;
                    $srp = null;
                    $groups = array();
                    for (
                        $entry = $conf->firstChild;
                        $entry != null;
                        $entry = $entry->nextSibling
                    ) {
                        if ($entry->nodeType == XML_ELEMENT_NODE) {
                            if ($entry->localName == 'username') {
                                $username = $entry->textContent;
                            } elseif ($entry->localName == 'password') {
                                $password = $entry->textContent;
                            } elseif ($entry->localName == 'crypt') {
                                $passcrypt = $entry->textContent;
                            } elseif (
                                $entry->localName == 'encrypted-crypt-password'
                            ) {
                                $passcrypt = $this->decrypt(
                                    $entry->textContent,
                                    'encrypted-crypt-password'
                                );
                            } elseif ($entry->localName == 'encrypted-srp') {
                                $srp = $this->decrypt(
                                    $entry->textContent,
                                    "encrypted-srp"
                                );
                            } elseif ($entry->localName == 'encrypted-digest') {
                                $digest = $this->decrypt(
                                    $entry->textContent,
                                    'encrypted-digest'
                                );
                            } elseif ($entry->localName == 'digest') {
                                $digest = $entry->textContent;
                            } else {
                                $groups[$entry->localName] =
                                    $entry->textContent == 'yes';
                            }
                        }
                    }

                    //echo "$username : $password : $digest\n<br/>";
                    if (
                        $password === false &&
                        $passcrypt === false &&
                        count($groups) == 0
                    ) {
                        // clearing the user
                        $user = User::model()->findByAttributes(array(
                            'username' => $username,
                        ));
                        if ($user) {
                            if ($user->id == 1 || !$user->delete()) {
                                $this->addError("user", 'Cannot delete user');
                            }
                        } else {
                            $this->addError(
                                "user",
                                "username '$username' not found"
                            );
                        }
                    } else {
                        $user = User::model()->findByAttributes(array(
                            'username' => $username,
                        ));
                        $attrs = array();
                        if (!$user) {
                            $user = new User();
                            $attrs['username'] = $username;
                        }
                        if ($password !== false) {
                            $attrs['password'] = $password;
                            $attrs['verifyPassword'] = $password;
                            $user->lowSecurityPassword = true;
                        } elseif ($passcrypt !== false) {
                            $attrs['password'] = $passcrypt;
                            $attrs['verifyPassword'] = $passcrypt;
                            $user->crypted = true;
                        }
                        $attrs['srpPassword'] = $srp;
                        $attrs['digestPassword'] = $digest;
                        $user->setAttributes($attrs);

                        $transaction = $user
                            ->getDbConnection()
                            ->beginTransaction();
                        try {
                            if ($user->save()) {
                                if (count($groups) != 0) {
                                    if ($user->id == 1) {
                                        $groups['admin'] = true;
                                    }
                                    $user->userRights->attributes = $groups;
                                    if (!$user->userRights->save()) {
                                        $this->updateErrors(
                                            $user->userRights,
                                            $conf->localName
                                        );
                                        throw new Exception(
                                            "save user rights failed"
                                        );
                                    }
                                }
                                if (!$user->updateHttpdSecurity()) {
                                    throw new Exception(
                                        $user->getError("password")
                                    );
                                }
                            } else {
                                $this->updateErrors($user, $conf->localName);
                                throw new Exception("save user failed");
                            }
                            $transaction->commit();
                        } catch (Exception $e) {
                            // an exception is raised if a query fails
                            $this->addError("user", $e->getMessage());
                            $transaction->rollback();
                        }
                        if ($user->needreload) {
                            $this->needreload = 1;
                        }
                    }
                    break;
                case 'secure-admin':
                case 'secure-content':
                case 'secure-monitoring':
                    if (!$securitySettings) {
                        $securitySettings = new SecuritySettings();
                    }
                    $this->getAll(
                        $securitySettings,
                        false,
                        $conf,
                        self::$security_map
                    );
                    break;
                case 'timezone':
                    $time = new TimeSettings();
                    $this->getAll($time, 'timezone', $conf, self::$time_map);
                    break;

                case 'time-manual':
                    if (!$ntp) {
                        $ntp = new NTPSettings();
                        $ntp->load();
                    }
                    $ntp->active = false;
                    $attr = $this->getAll(
                        $ntp,
                        'manual',
                        $conf,
                        self::$manualtime_map
                    );
                    break;
                case 'time-ntp-restore':
                case 'time-ntp':
                    if (!$ntp) {
                        $ntp = new NTPSettings();
                        $ntp->load();
                    }
                    $ntp->active = true;
                    $ntp->setScenario('ntp');
                    $ntp->loadTime();
                    if ($conf->localName == 'time-ntp-restore') {
                        break;
                    }
                    $ntp->clearServers();
                    $this->getAll($ntp, null, $conf, self::$ntpserver_map);
                    break;
                case 'shutdown-temperature':
                    if (Yii::app()->device->model === 'ikebana') {
                        if (!$hwd) {
                            $hwd = new HardwareSetup();
                        }
                        $this->getAll(
                            $hwd,
                            false,
                            $conf,
                            self::$temperature_map
                        );
                    }
                    break;
                case 'disable-secondary-network-port':
                    if (Yii::app()->branding->product == 'HMP350') {
                        if (!$hwd) {
                            $hwd = new HardwareSetup();
                        }
                        $this->getAll($hwd, false, $conf, self::$ports_map);
                    } else {
                        $this->addError(
                            $conf->localName,
                            'Not supported by this model'
                        );
                    }
                    break;
                case "display-power-save":
                    if ($power == null) {
                        $power = new Power();
                        $power->playerConf = $config_file;
                        $power->load();
                    }
                    $attr = $this->getAll(
                        $power,
                        'backup',
                        $conf,
                        self::$power_map
                    );
                    break;
                case "display-power-schedule":
                    if ($power == null) {
                        $power = new Power();
                        $power->playerConf = $config_file;
                        $power->load();
                    }
                    $power->deleteSchedule();
                    $attr = $this->getAll(
                        $power,
                        'backup',
                        $conf,
                        self::$power_map
                    );
                    if (count($attr) == 0) {
                        $power->modified = true;
                    }
                    break;
                case 'com-port-settings':
                    if ($serial == null) {
                        $serial = new SerialConfig();
                        $serial->playerConf = $config_file;
                        $serial->load();
                    }
                    $attr = $this->getAll(
                        $serial,
                        'backup',
                        $conf,
                        self::$io_map
                    );
                    if (count($attr) == 0) {
                        $serial->modified = true;
                    }
                    break;
                case "protocol-file-reset":
                    if ($serial == null) {
                        $serial = new SerialConfig();
                        $serial->playerConf = $config_file;
                        $serial->load();
                    }
                    $serial->deleteAll();
                    break;
                case 'protocol-disabled':
                case 'protocol-system':
                case 'protocol-file':
                case 'protocol-simple':
                    if ($serial == null) {
                        $serial = new SerialConfig();
                        $serial->playerConf = $config_file;
                        $serial->load();
                    }
                    $attr = $this->get(
                        $conf,
                        self::$io_map + array(
                            'fsm-file' => true,
                            'command' => true,
                            'name' => true,
                            'data' => true,
                        )
                    );
                    $serial->setScenario('backup');
                    $serial->setAttributes($attr);

                    if (!isset($attr['backupName'])) {
                        $attr['backupName'] = "";
                    }
                    if ($conf->localName == 'protocol-disabled') {
                        $serial->setBackup('disabled');
                    } elseif ($conf->localName == 'protocol-system') {
                        $serial->setBackup('system', $attr['backupName']);
                    } elseif ($conf->localName == 'protocol-simple') {
                        $serial->setBackup('simple');
                        $commands = array();
                        for (
                            $node = $conf->firstChild;
                            $node != null;
                            $node = $node->nextSibling
                        ) {
                            if (
                                $node->nodeType == XML_ELEMENT_NODE &&
                                $node->localName == "command"
                            ) {
                                $command = array();
                                for (
                                    $n2 = $node->firstChild;
                                    $n2 != null;
                                    $n2 = $n2->nextSibling
                                ) {
                                    if ($n2->nodeType == XML_ELEMENT_NODE) {
                                        $command[$n2->localName] =
                                            $n2->nodeValue;
                                    }
                                }
                                $skip = false;
                                if (!isset($command['name'])) {
                                    $this->addError(
                                        $conf->localName . " > command",
                                        'Missing name'
                                    );
                                    $skip = true;
                                }
                                if (isset($command['data'])) {
                                    $command['value'] = $command['data'];
                                } else {
                                    $this->addError(
                                        $conf->localName . " > command",
                                        'Missing data'
                                    );
                                    $skip = true;
                                }
                                if (!$skip) {
                                    $commands[] = $command;
                                }
                            }
                        }
                        $serial->setSimpleCommands($commands);
                    } elseif ($conf->localName == 'protocol-file') {
                        $content = "";
                        for (
                            $node = $conf->firstChild;
                            $node != null;
                            $node = $node->nextSibling
                        ) {
                            if (
                                $node->nodeType == XML_ELEMENT_NODE &&
                                $node->localName == "fsm-file"
                            ) {
                                $newdoc = new DOMDocument();
                                $f = $node->firstChild;
                                while (
                                    $f != null &&
                                    $f->nodeType != XML_ELEMENT_NODE &&
                                    $f->localName != "protocol"
                                ) {
                                    $f = $f->nextSibling;
                                }
                                $newdoc->appendChild(
                                    $newdoc->importNode($f, true)
                                );
                                $content = $newdoc->saveXML();
                            }
                        }
                        $serial->setBackup(
                            'inline',
                            $attr['backupName'],
                            $content
                        );
                    }
                    if ($serial->hasErrors() || !$serial->validate()) {
                        $this->updateErrors($serial, $conf->localName);
                    }
                    break;
                case 'capture':
                    if (!$capture)
                        $capture = new Capture();
                    foreach ($conf->childNodes as $child) {
                        if ($child->nodeType == XML_ELEMENT_NODE && 
                            $child->localName == 'video') {
                            $device = new CaptureDevice();
                            $device->setAttributes($this->get(
                                $child, self::$capture_device_map, 
                                $conf->localName . ' > '), false);
                            $capture->devices[$device->path] = $device;
                        }
                    }
                    if (!$capture->validate()) {
                        $this->updateErrors($capture, $conf->localName);
                        foreach ($capture->devices as $device) {
                            $this->updateErrors(
                                $device, self::$capture_device_map, 
                                $conf->localName . ' > video > ');
                        }
                    }
                    break;
                case 'cockpit':
                case 'cockpit-disabled':
                    if (!$pull) {
                        $pull = new PullModeConfig();
                        $pull->load();
                    }
                    $pull->cockpitUri = null;
                    $this->getAll($pull, false, $conf, self::$cockpit_map);
                    break;
                case 'pull-mode-' . PullModeConfig::PULL_STATIC:
                    $hasLogs = false;
                    $hasRPC = false;
                    for (
                        $n = $conf->firstChild;
                        $n != null;
                        $n = $n->nextSibling
                    ) {
                        if ($n->nodeType == XML_ELEMENT_NODE) {
                            if ($n->localName == 'pull-project') {
                                $this->addError(
                                    $conf->localName . " > pull-project",
                                    'Unknown command item'
                                );
                                continue;
                            }
                            if ($n->localName == 'rpc') {
                                $hasRPC = true;
                            }
                            if ($n->localName == 'logs') {
                                $hasLogs = true;
                            }
                        }
                    }
                case 'pull-mode-' . PullModeConfig::PULL_FILE:
                case 'pull-mode-' . PullModeConfig::PULL_REMOTE:
                case 'pull-mode-' . PullModeConfig::PULL_DISABLED:
                    if (!$pull) {
                        $pull = new PullModeConfig();
                        $pull->load();
                    }
                    $pull->backup = true;
                    $pull->setSelectMode(substr($conf->localName, 10));

                    $attr = $this->get($conf, self::$pullmode_map);
                    if (
                        $conf->localName ==
                        'pull-mode-' . PullModeConfig::PULL_STATIC
                    ) {
                        if (!isset($attr['enable_log'])) {
                            if ($hasLogs) {
                                $this->addError(
                                    $conf->localName . " > logs",
                                    'Missing push-logs command'
                                );
                                break;
                            } else {
                                $attr['enable_log'] = false;
                            }
                        }
                        if ($attr['enable_log'] && !$hasLogs) {
                            $this->addError(
                                $conf->localName . " > logs",
                                'Missing mendatory parameters when push-logs is enabled'
                            );
                            break;
                        }
                        if (!$attr['enable_log'] && $hasLogs) {
                            $this->addError(
                                $conf->localName . " > logs",
                                'push-logs not enabled'
                            );
                            break;
                        }
                        if ($attr['enable_log'] && !isset($attr['timeLog'])) {
                            $attr['timeLog'] = "00:30";
                        }
                        if (!isset($attr['enable_rpc'])) {
                            if ($hasRPC) {
                                $this->addError(
                                    $conf->localName . " > rpc",
                                    'Missing rpc-concentrator command'
                                );
                                break;
                            } else {
                                $attr['enable_rpc'] = false;
                            }
                        }
                        if ($attr['enable_rpc'] && !$hasRPC) {
                            $this->addError(
                                $conf->localName . " > rpc",
                                'Missing mendatory paramters when rpc-concentrator is enabled'
                            );
                            break;
                        }
                        if (!$attr['enable_rpc'] && $hasRPC) {
                            $this->addError(
                                $conf->localName . " > rpc",
                                'rpc-concentrator not enabled'
                            );
                            break;
                        }
                        if ($attr['enable_rpc'] && !isset($attr['polling'])) {
                            $attr['polling'] = "60s";
                        }
                    }
                    $pull->setAttributes($attr);

                    if (!$pull->validate()) {
                        $this->updateErrors(
                            $pull,
                            self::$pullmode_map + array(
                                'log-time (hour)' => 'hourLog',
                                'log-time (min)' => 'minLog',
                            ),
                            $conf->localName . " > "
                        );
                    }
                    break;
                case 'pullmode-logs-level':
                    if (!$pullModeConfigLogs) {
                        $pullModeConfigLogs = new PullModeConfig();
                        $pullModeConfigLogs->loadLogConf();
                    }
                    $this->getAll($pullModeConfigLogs, false, $conf, array(
                        "pullmode-logs-level" => "priority",
                    ));
                    break;
                case 'pullmode-http-capture-log':
                    if (!$pull) {
                        $pull = new PullModeConfig();
                        $pull->load();
                    }
                    $this->getAll($pull, false, $conf, array(
                        "pullmode-http-capture-log" => "httpLogs",
                    ));
                    break;
                case 'scheduled-download-reset':
                    if (!$pull) {
                        $pull = new PullModeConfig();
                        $pull->load();
                    }
                    $pull->resetSyncSource();
                    break;
                case 'scheduled-download-add':
                    if (!$pull) {
                        $pull = new PullModeConfig();
                        $pull->load();
                    }
                    $syncsource = new SyncSource();
                    $this->getAll(
                        $syncsource,
                        false,
                        $conf,
                        self::$syncsource_map,
                        "scheduled-download-add > "
                    );
                    $pull->addSyncSource($syncsource);
                    break;
                case 'live-source-reset':
                    if (!$sources) {
                        $sources = new SourceConfig();
                        $sources->playerConf = $config_file;
                        $sources->load();
                    }
                    $sources->resetLiveSource();
                    break;
                case 'live-source-add':
                    if (!$sources) {
                        $sources = new SourceConfig();
                        $sources->playerConf = $config_file;
                        $sources->load();
                    }
                    $livesource = new NetworkSource();
                    $this->getAll(
                        $livesource,
                        false,
                        $conf,
                        self::$livesource_map,
                        "live-source-add > "
                    );
                    $sources->addLiveSource($livesource);
                    break;
                case 'firmware-update-uri':
                    if (!$firmware) {
                        $firmware = new FirmwareUpdate();
                        $firmware->load();
                    }

                    $attr = $this->getAll(
                        $firmware,
                        'backup',
                        $conf,
                        self::$firmware_map
                    );
                    if ($attr['updateuri'] == "") {
                        $firmware->resetUri();
                    }
                    break;
                case 'firmware-update-auto':
                    if (!$firmware) {
                        $firmware = new FirmwareUpdate();
                        $firmware->load();
                    }
                    $this->getAll(
                        $firmware,
                        'backup',
                        $conf,
                        self::$firmware_map
                    );
                    break;
                case 'streaming':
                    if (!$streaming)
                        $streaming = new BackupStreaming();
                    $attr = $this->get($conf, self::$streaming_map);
                    if (!$streaming->setAttributes($attr)) {
                        $this->updateErrors($streaming, $conf->localName);
                    }
                    break;
                // WebRTC
                case 'echo-cancellation':
                    $this->getAll(
                        $config_file,
                        'WebRTC',
                        $conf,
                        self::$webrtc_map
                    );
                    break;
                case 'bonjour-discovery-enabled':
                case 'bonjour-enabled':
                    if (!$bonjourConfig) {
                        $bonjourConfig = new BonjourConfig();
                    }
                    $this->getAll(
                        $bonjourConfig,
                        'backup',
                        $conf,
                        self::$bonjour_map
                    );
                    break;
                case 'ssdp-upnp-enabled':
                    if (!$UPnPConfig) {
                        $UPnPConfig = new UPnPConfig();
                    }
                    $this->getAll(
                        $UPnPConfig,
                        'backup',
                        $conf,
                        self::$upnp_map
                    );
                    break;
                case 'llmnr-enabled':
                    if (!$LLMNRConfig) {
                        $LLMNRConfig = new LLMNRConfig();
                    }
                    $this->getAll(
                        $LLMNRConfig,
                        'backup',
                        $conf,
                        self::$llmnr_map
                    );
                    break;
                case 'debug-log':
                    if (!$log)
                        $log = new DebugLog();
                    foreach ($conf->childNodes as $child) {
                        if ($child->nodeType == XML_ELEMENT_NODE && 
                            $child->localName == 'logger') {
                            $logger = new DebugLogger();
                            $logger->setAttributes($this->get(
                                $child, self::$logger_map, 
                                $conf->localName . ' > '), false);
                            $log->loggers[$logger->name] = $logger;
                        }
                    }
                    if (!$log->validate()) {
                        $this->updateErrors($log, $conf->localName);
                        foreach ($log->loggers as $logger) {
                            $this->updateErrors(
                                $logger, self::$logger_map, 
                                $conf->localName . ' > logger > ');
                        }
                    }
                    break;
                case "license-file":
                    if (!$playerLicenses) {
                        $playerLicenses = new PlayerLicenses();
                        $playerLicenses->load();
                    }
                    $this->getAll(
                        $playerLicenses,
                        false,
                        $conf,
                        self::$license_map
                    );
                    break;
                case 'enable-arya':
                    if (!$ARYAMode) {
                        $ARYAMode = new ARYAMode();
                    }
                    $ARYAMode->setEnabled();
                    break;
                case 'reset-arya':
                    if (!$ARYAMode) {
                        $ARYAMode = new ARYAMode();
                    }
                    $ARYAMode->resetArya();
                    break;
                case 'enable-cloud':
                    if (!$iot) {
                        $iot = new Iot();
                    }
                    $iot->enable = true;
                    break;
                case 'disable-cloud':
                    if (!$iot) {
                        $iot = new Iot();
                    }
                    $iot->enable = false;
                    break;
                case 'debug-enroll':
                case 'enroll-base-url':
                case 'debug-iot':
                    if (!$iot) {
                        $iot = new Iot();
                    }
                    $this->getAll($iot, false, $conf, self::$iot_map);
                    break;
                case 'reset-published-content':
                    $maintenance->resetPublishedContent();
                    break;
                case 'reset-content':
                    $maintenance->resetContent();
                    break;
                case 'reset-webstorage':
                    $maintenance->clearWebstorage();
                    break;
                case 'reset-log':
                    $maintenance->clearLogs();
                    break;
                case 'reset-cache':
                    $maintenance->clearCache();
                    break;
                case 'reset-ntp':
                    $maintenance->clearNtp();
                    break;
                case 'clear-captures':
                    $maintenance->clearStreaming();
                    break;
                case 'reboot':
                    $reboot = true;
                    break;
                default:
                    $this->addError($conf->localName, 'Unknown command');
            }
        }
        if ($screen && !$screen->hasErrors() && $check_screen_config) { 
            if (!$screen->check()) {
                $this->updateErrors($screen, self::$display_map);
            }
        }
        if (!$this->hasErrors() && $checkMultiOutput) {
            $this->checkMultiOutput($config_file, $screen);
        }
        if (!$this->hasErrors() && $screen) {
            if (!$screen->save()) {
                $this->updateErrors($screen, self::$display_map);
            }
        }
        if (!$this->hasErrors()) {
            $config_file->saveConfig();
            if ($power) {
                $power->save();
            }
            if ($serial) {
                $serial->save();
            }
            if ($splash) {
                $splash->save();
            }
            if ($network) {
                $network->save(!$credit, !$ntp);
            }
            if ($watchdog) {
                $watchdog->save();
            }
            if ($credit) {
                $credit->save($network);
            }
            if ($proxy) {
                $proxy->save();
            }
            if ($snmp) {
                $snmp->save();
            }
            if ($sources) {
                $sources->save();
            }
            if ($time) {
                $time->save();
            }
            if ($ntp) {
                $ntp->save($network);
            }
            if ($hwd) {
                $hwd->save();
            }
            if ($pull) {
                $pull->save();
            }
            if ($bonjourConfig) {
                $bonjourConfig->save();
            }
            if ($UPnPConfig) {
                $UPnPConfig->save();
            }
            if ($LLMNRConfig) {
                $LLMNRConfig->save();
            }
            if ($securityRPC) {
                $securityRPC->save();
            }
            if ($webstorageAPI) {
                $webstorageAPI->save();
            }
            if ($serverSecurity) {
                $serverSecurity->save();
            }
            if ($pullModeConfigLogs) {
                $pullModeConfigLogs->saveLogConf();
            }
            if ($firmware) {
                $firmware->save();
            }
            if ($touchCalibration) {
                $touchCalibration->save();
            }
            if ($streaming) {
                $streaming->save();
            }
            if ($capture) {
                $capture->save();
            }
            if ($log) {
                $log->save();
            }

            if ($playerLicenses) {
                $playerLicenses->save();
            }
            if ($securitySettings) {
                if ($securitySettings->needreload) {
                    $this->needreload = 1;
                }
            }

            if ($iot) {
                $iot->save();
            }

            if (Tools::doShutdown()) {
                Tools::shutdown(false);
            }
            if ( $reboot && ( $this->shutdownDelay === false || $this->shutdownDelay >= 0 ) ) {
                if (Tools::checkInstallationmode()) {
                    Tools::shutdown(true, false, $this->shutdownDelay);
                }
            }

            Tools::addMessage("Configuration applied successfully");
            Tools::$reload = true;
        }
    }
    function reloadConfig()
    {
        if ($this->needreload) {
            Tools::reloadHttpd();
        }
    }

    protected function checkMultiOutput($playerConfig, $screen) {
        if (!$playerConfig->hasMultipleVideoOutputs()) {
            return;
        }

        if (
            !($regions = $this->getMultiOutputRegions($playerConfig, $screen))
        ) {
            return;
        };

        $mode = self::getMultiOutputMode($regions);

        if (!self::hasMultiOutputRight($mode)) {
            $this->addError('', "No valid license found for multi-output " .
                "mode '" . ucfirst(trim($mode)) . "'");
        }
    }

    protected function getMultiOutputRegions($playerConfig, $screen) {
        if (!$screen) {
            $screen = new ScreenSettings();
            $screen->load();
        }

        foreach ($playerConfig->getVideoOutputs() as $name => $output) {
            if (!($mode = Mode::get($screen->modes, $name))) {
                $this->addError('', "No video mode configured for output " .
                    "'{$name}'");
                return array();
            }
            if ($mode->resolution) {
                if ($mode->resolution === 'native') {
                    $width = 0;
                    $height = 0;
                } else {
                    list($width, $height) = explode('x', $mode->resolution);
                }
            } else {
                list($width, $height) = explode(' ', $mode->custom);
            }
            $regions[] =
                PlayerConfig::getVideoOutputRegion($output, $width, $height);
        }

        return $regions;
    }

    protected static function getMultiOutputMode($regions) {
        foreach ($regions as $region) {
            if ($region->width()) {
                $regionsWidth = $region->width();
            }
            if ($region->height()) {
                $regionsHeight = $region->height();
            }
        }

        // Check if all regions have the same dimensions
        if (isset($regionsWidth)) {
            foreach ($regions as $region) {
                $width = $region->width();
                if ($width && $width !== $regionsWidth) {
                    return 'custom';
                }
            }
        }
        if (isset($regionsHeight)) {
            foreach ($regions as $region) {
                $height = $region->height();
                if ($height && $height !== $regionsHeight) {
                    return 'custom';
                }
            }
        }

        // Check mirror configuration (all regions are identical)
        for ($i = 1; $i < count($regions); $i++) {
            if (
                $regions[$i]->x1 !== $regions[0]->x1 ||
                $regions[$i]->y1 !== $regions[0]->y1
            ) {
                break;
            }
        }
        if ($i === count($regions)) {
            return 'mirroring';
        }

        // Sort by top-left coordinates to check for a grid pattern with no
        // duplicates
        foreach ($regions as $region) {
            if (
                isset($xy[$region->x1]) && 
                in_array($region->y1, $xy[$region->x1])
            ) {
                return 'custom';
            }
            $xy[$region->x1][] = $region->y1;
            $yx[$region->y1][] = $region->x1;
        }
        ksort($xy);
        ksort($yx);

        // Each row must have the same number of columns
        foreach ($yx as $row) {
            if (count($row) !== count($xy)) {
                return 'custom';
            }
        }

        // Each column must have the same number of rows
        foreach ($xy as $column) {
            if (count($column) !== count($yx)) {
                return 'custom';
            }
        }

        // Grid elements must not overlap and must be evenly spaced by no more
        // than 5/4 of the dimensions
        $horizontalStep = 0;
        $verticalStep = 0;

        if (count($xy) > 1) {
            $x = array_slice(array_keys($xy), 0, 2);
            $horizontalStep = $x[1] - $x[0];
            if (isset($regionsWidth)) {
                if (
                    $horizontalStep < $regionsWidth || 
                    $horizontalStep - $regionsWidth > $regionsWidth / 4
                ) {
                    return 'custom';
                }
            }
        }

        if (count($yx) > 1) {
            $y = array_slice(array_keys($yx), 0, 2);
            $verticalStep = $y[1] - $y[0];
            if (isset($regionsHeight)) {
                if (
                    $verticalStep < $regionsHeight ||
                    $verticalStep - $regionsHeight > $regionsHeight / 4
                ) {
                    return 'custom';
                }
            }
        }

        $nextLeft = key($xy);
        foreach($xy as $x => $column) {
            if ($x !== $nextLeft) {
                return 'custom';
            }
            $nextLeft += $horizontalStep;
        }

        $nextTop = key($yx);
        foreach($yx as $y => $row) {
            if ($y !== $nextTop) {
                return 'custom';
            }
            $nextTop += $verticalStep;
        }

        return 'grid';
    }

    protected static function hasMultiOutputRight($mode) {
        if ($mode === 'mirroring') {
            return Yii::app()->branding->hasRight('MIRROR-OUTPUT') ||
                Yii::app()->branding->hasRight('CUSTOM-OUTPUT');
        } elseif ($mode === 'grid') {
            return Yii::app()->branding->hasRight('GRID-OUTPUT') ||
                Yii::app()->branding->hasRight('CUSTOM-OUTPUT');
        } elseif ($mode === 'custom') {
            return Yii::app()->branding->hasRight('CUSTOM-OUTPUT');
        } else {
            return false;
        }
    }
}
