<?php
class CaptureDevice extends CFormModel {
    public function attributeLabels() {
        return array('index' => $this->name);
    }

    public function rules() {
        return array(array(
            'index', 'numerical', 'integerOnly' => true, 'min' => 1, 
            'message' => '{attribute} index must be an integer.', 
            'tooSmall' => '{attribute} index must be greater than 0.'));
    }

    public function hasErrors($attribute = null) {
        if ($attribute !== null) {
            $attribute = str_replace('[' . $this->path . ']', '', $attribute);
        }
        return parent::hasErrors($attribute);
    }    

    public $index = '';
    public $name = '';
    public $path = '';
}

class Capture extends CFormModel {

    const FILE = '/etc/raperca/spxconfig.xml';
    const V4L_DIR = '/sys/class/video4linux';

    public function rules() {
         return array(array('devices', 'validateDevices'));
    }

    public function validateDevices($attribute, $params) {
        $error = false;
        $devicesByIndex = array();
        foreach ($this->$attribute as $device) {
            if (!$device->validate()) {
                $error = true;
            }
            if ($device->index != '') {
                if (!isset($devicesByIndex[$device->index])) {
                    $devicesByIndex[$device->index] = array();
                }
                $devicesByIndex[$device->index][] = $device;
            }
        }
        foreach ($devicesByIndex as $devices) {
            if (count($devices) > 1) {
                $error = true;
                foreach ($devices as $device) {
                    $device->addError(
                        'index', $device->name . ' index must be unique.');
                }
            }
        }
        if ($error) {
            $this->addError($attribute, 'Invalid capture device indexes.');
        }
    }

    public function load() {
        $this->loadConfig();
        $this->loadDevices();
    }

    public function loadConfig() {
        $doc = new DOMDocument;
        if (!$doc->load(self::FILE)) {
            return;
        }
        //Use DOMXPath::query?
        if (!($capture = $doc->getElementsByTagName('capture')->item(0))) {
            return;
        }
        foreach ($capture->childNodes as $child) {
            if ($child->nodeType == XML_ELEMENT_NODE && $child->nodeName == 
                'video') {
                $device = new CaptureDevice();
                $device->index = $child->getAttribute('index');
                $device->name = $child->getAttribute('name');
                $device->path = $child->getAttribute('path');
                $this->devices[$device->path] = $device;
            }
        }
    }

    public function save() {
        $doc = new DOMDocument;
        $doc->preserveWhiteSpace = false;
        $doc->formatOutput = true;
        if (!$doc->load(self::FILE)) {
            return;
        }
        $changed = false;
        //Use DOMXPath::query?
        if (($capture = $doc->getElementsByTagName('capture')->item(0))) {
            $nodes = array();
            foreach ($capture->childNodes as $child) {
                if ($child->nodeType == XML_ELEMENT_NODE && $child->nodeName 
                    == 'video') {
                    $nodes[] = $child;
                }
            }
            foreach($nodes as $node) {
                $capture->removeChild($node);
            }
            if (!empty($nodes)) {
                $changed = true;
            }
        } 
        if ($this->hasIndex()) {
            if (!$capture) {
                if (!($node = $doc->getElementsByTagName('player')->item(0))) {
                    return;
                }
                $capture = $node->appendChild($doc->createElement('capture'));
            }
            foreach ($this->devices as $device) {
                if (!empty($device->index)) {
                    $node = $capture->appendChild(
                        $doc->createElement('video'));
                    $node->setAttribute('index', $device->index);
                    $node->setAttribute('name', $device->name);
                    $node->setAttribute('path', $device->path);
                }
            }
            $changed = true;
        } else if ($capture && !$capture->hasChildNodes()) {
            $capture->parentNode->removeChild($capture);
            $changed = true;
        }
        if ($changed) {
            Tools::save_file(self::FILE, $doc->saveXML());
            Tools::addReason('capture configuration change');
        }
    }

    public $devices = array();

    private function loadDevices() {
        $devices = array();
        if ($files = @scandir(self::V4L_DIR)) {
            Yii::log(
                'scandir(' . self::V4L_DIR . '): ' . print_r($files, true), 
                'debug', 'spx.Capture');
            foreach ($files as $file) {
                if (!strncmp($file, 'video', 5)) {
                    $path = self::V4L_DIR . '/' . $file;
                    $devices[$path] = substr($file, 5);
                }
            }
        }
        Yii::log(
            'Video devices: ' . print_r($devices, true), 'debug', 
            'spx.Capture');
        foreach ($devices as $path => $index) {
            unset($output);
            if (exec(
                'v4l2-ctl -d ' . escapeshellarg($index) . ' --list-formats', 
                $output, $status) !== false && $status === 0) {
                Yii::log(
                    'v4l2-ctl -d ' . $index . ' --list-formats output' . 
                    print_r($output, true), 'debug', 'spx.Capture');
                if (empty(preg_grep('/^\s*Type:\s+Video Capture/', $output)) 
                    || empty(preg_grep('/^\s*\[\d+\]:/', $output)) 
                    || !empty(preg_grep('/^\s*\[\d+\]:\s+\'GREY\'/', 
                    $output))) {
                    unset($devices[$path]);
                }
            } else {
                unset($devices[$path]);
            }
        }
        Yii::log('
            Capture devices: ' . print_r($devices, true), 'debug', 
            'spx.Capture');
        foreach ($devices as $path => $index) {
            unset($output);
            if (exec(
                'udevadm info -p ' . escapeshellarg($path) . ' -q property', 
                $output, $status) !== false && $status === 0) {
                //Yii::log(
                //    'udevadm info -p ' . $path . ' -q property' . 
                //    print_r($output, true), 'debug', 'spx.Capture');
                $properties = array();
                array_walk(
                    $output, function ($value, $index) use (&$properties) {
                        list($key, $value) = explode('=', $value);
                        $properties[$key] = $value;
                    });
                Yii::log(
                    $path . ' properties: ' . print_r($properties, true), 
                    'debug', 'spx.Capture');
                if (!isset($properties['DEVLINKS']) || !isset(
                    $properties['ID_SERIAL'])) {
                    continue;
                }
                $byId = strstr($properties['DEVLINKS'], '/dev/v4l/by-id');
                if ($byId === false) {
                    continue;                
                }
                $byId = strtok($byId, ' ');
                if (!isset($this->devices[$byId])) {
                    $device = new CaptureDevice();
                    $device->name = str_replace(
                        '_', ' ', $properties['ID_SERIAL']);
                    $device->path = $byId;
                    $this->devices[$byId] = $device;
                }
                $this->devices[$byId]->name = str_replace(
                    '_', ' ', $properties['ID_SERIAL']);
            }
        }
        if (defined('YII_DEBUG') && YII_DEBUG) {
            if (!isset($this->devices['/path/to/dummy0'])) {
                $device = new CaptureDevice();
                $device->name = 'Dummy Device 0';
                $device->path = '/path/to/dummy0';
                $this->devices[$device->path] = $device;
            }
            if (!isset($this->devices['/path/to/dummy1'])) {
                $device = new CaptureDevice();
                $device->name = 'Dummy Device 1';
                $device->path = '/path/to/dummy1';
                $this->devices[$device->path] = $device;
            }
        }
    }

    private function hasIndex() {
        foreach ($this->devices as $device) {
            if (!empty($device->index)) {
                return true;
            }
        }
        return false;
    }
}
