/*!
 * jSignage.Astronomy
 * http://www.spinetix.com/
 * Copyright SpinetiX S.A.
 * Released under the GPL Version 2 or later license.
 *
 * Includes code from the suncal library https://github.com/mourner/suncalc
 * Copyright (c) 2014, Vladimir Agafonkin 
 * 
 * Redistribution and use in source and binary forms, with or without modification, are 
 * permitted provided that the following conditions are met: 
 *    1. Redistributions of source code must retain the above copyright notice, this list of 
 *       conditions and the following disclaimer. 
 *    2. Redistributions in binary form must reproduce the above copyright notice, this list 
 *       of conditions and the following disclaimer in the documentation and/or other materials 
 *       provided with the distribution. 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR 
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 *
 * Includes code from http://praytimes.org/
 * PrayTimes.js: Prayer Times Calculator (ver 2.3)
 * Copyright (C) 2007-2011 PrayTimes.org
 * Developer: Hamid Zarrabi-Zadeh
 * License: GNU LGPL v3.0
 * TERMS OF USE
 *   Permission is granted to use this code, with or
 *   without modification, in any website or application
 *   provided that credit is given to the original work
 *   with a link back to PrayTimes.org.
 * This program is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY
 * PLEASE DO NOT REMOVE THIS COPYRIGHT BLOCK.
 *
 * Includes code by Peter Hayes http://www.aphayes.pwp.blueyonder.co.uk/
 * Copyright 2001-2010
 * Unless otherwise stated this code is based on the methods in
 * Astronomical Algorithms, first edition, by Jean Meeus
 * Published by Willmann-Bell, Inc.
 * This code is made freely available but please keep this notice.
 * The calculations are approximate but should be good enough for general use,
 * I accept no responsibility for errors in astronomy or coding.
 *
 * Includes code from http://tingletech.github.com/moon-phase/
 * License: BSD-3-clause
 *
 * $Date: 2021-02-01 09:38:38 +0000 (Mon, 01 Feb 2021) $
 * $Revision: 37509 $
 */
 
( function () {
var version = new String( "1.0.1" );
version.major = 1;
version.minor = 0;
version.revision = 1;
    
var reElevation = /^\s*((?:[+-])?\d+(?:\.\d*)?)\s*(m|M|Ft|FT|ft|')?\s*$/;
var reRelDay = /^(this|[+-]?\d+) ([a-z]+?)s?$/;
var wdays = { monday: 0, tuesday: 1, wednesday: 2, thursday: 3, friday: 4, saturday: 5, sunday: 6 };
var mdays = [ 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 ];

function lastDayOfTheMonth( month, year ) {
    month = month % 12;
    if ( month < 0 )
        month += 12;
    var day = mdays[ month ];
    if ( month == 1 && year % 4 == 0 && ( year % 100 != 0 || year % 400 == 0 ) )
        ++day;
    return day;
}

jSignage.Astronomy = {

    version: version,
    
    parseLatitude: function( dms ) {
        return jSignage.parseLatLon( dms, true );
    },

    parseLongitude: function( dms ) {
        return jSignage.parseLatLon( dms, false );
    },

    parseElevation: function( el ) {
        var x = NaN;
        var m = reElevation.exec( el );
        if ( m ) {
            x = parseFloat( m[1] );
            if ( !isFinite( x ) )
                x = NaN;
            if ( m[2]=='ft' || m[2]=='Ft' || m[2]=='FT' || m[2]=='\'' )
                x *= 0.3048;
        }
        return x;
    },

    adjustYMD: function( year, month, day ) {
        month -= 1;
        while ( day > lastDayOfTheMonth( month, year ) )
            day -= lastDayOfTheMonth( month++, year );
        while ( day <= 0 )
            day += lastDayOfTheMonth( --month, year );
        if ( month > 11 ) {
            year += Math.floor( month / 12 );
            month %= 12;
        } else if ( month < 0 ) {
            year += Math.floor( month / 12 );
            month = month % 12 + 12;
        }
        return [year, month + 1, day];
    },

    parseDay: function( date_string, today ) {
        var year = today.getFullYear();
        var month = today.getMonth() + 1;
        var day = today.getDate();
        if ( date_string != 'today' ) {
            var r = reRelDay.exec( date_string );
            if ( r ) {
                var x = r[1] == 'this' ? 0 : parseFloat( r[1] );
                var y = r[2];
                if ( y == 'day' ) {
                    day += x;
                } else if ( y in wdays ) {
                    var wday = wdays[y];
                    var tday = today.getDay() - 1;
                    if ( tday < 0 ) tday = 6;
                    if ( x == 0 ) {
                        var firstDayOfTheWeek = jSignage.getLocaleInfo().DateTimeSymbols.FIRSTDAYOFWEEK;
                        if ( ( wday - firstDayOfTheWeek + 7 ) % 7 < ( tday - firstDayOfTheWeek + 7 ) % 7 )
                            x = -1;
                        else if ( ( wday - firstDayOfTheWeek + 7 ) % 7 > ( tday - firstDayOfTheWeek + 7 ) % 7 )
                            x = 1;
                    }
                    if ( x > 0 ) {
                        if ( wday > tday )
                            day += wday - tday + ( x - 1 ) * 7;
                        else
                            day += wday - tday + x * 7;
                    } else if ( x < 0 ) {
                        if ( wday > tday )
                            day += wday - tday + x * 7;
                        else
                            day += wday - tday + ( x + 1 ) * 7;
                    }
                }
            } else {
                return false;
            }
        }
        return jSignage.Astronomy.adjustYMD( year, month, day );
    },

    /**
     *  Converts milliseconds into hours, minutes and seconds.
     *  @param {number} ms The time in milliseconds.
     *  @return {Object.<string, string>} Object containing hours, minutes and seconds.
     */
    getTimeUnitsfromMs: function ( ms ) {
        if ( isNaN( ms ) ) return false;        
        
        var nf = new jSignage.NumberFormat( '0' ),
            nf2 = new jSignage.NumberFormat( '00' ),
            seconds = Math.round( Math.abs( ms ) / 1000 ),
            minutes,
            hours = Math.floor( seconds / 3600 );
        
        seconds -= hours * 3600;
        minutes = Math.floor( seconds / 60 );
        seconds -= minutes * 60;
        
        return {  
            h: nf.format( hours ), 
            hh: nf2.format( hours ),
            m: nf.format( minutes ), 
            mm: nf2.format( minutes ),  
            s: nf.format( seconds ), 
            ss: nf2.format( seconds ), 
            sign: ( ms >= 0 ? '+' : '-' )
        };
    },
    
    dailyEventsScheduler: function ( triggers, daily_callback, trigger_callback, longitude ) {
        var schedule = {}, nextTimer = null, refreshTimer = null;

        function addToSchedule( item ) {
            for ( var i in item ) {
                if ( !schedule[i] )
                    schedule[i] = [];
                schedule[i].push( item[i] );
            }
        }

        function refreshSchedule() {
            var t = Date.now() + ( longitude || 0 ) * 240000;
            schedule = {};
            var yesterday = new Date( t - 86400000 ), today = new Date( t ), tomorrow = new Date( t + 86400000 );
            addToSchedule( daily_callback( yesterday.getUTCFullYear(), yesterday.getUTCMonth() + 1, yesterday.getUTCDate() ) );
            addToSchedule( daily_callback( today.getUTCFullYear(), today.getUTCMonth() + 1, today.getUTCDate() ) );
            addToSchedule( daily_callback( tomorrow.getUTCFullYear(), tomorrow.getUTCMonth() + 1, tomorrow.getUTCDate() ) );
        }

        function trig_current( t0 ) {
            var current = null, tCurrent = 0;
            if ( triggers ) for ( var i = 0; i < triggers.length; i++ ) {
                var trigger = triggers[i];
                if ( !schedule[trigger.start] )
                    continue;
                for ( var j = 0; j < schedule[trigger.start].length; j++ ) {
                    var t = schedule[trigger.start][j] + ( trigger.offset || 0 ) * 60000;
                    if ( t <= t0 ) {
                        if ( current === null || t > tCurrent ) {
                            current = trigger;
                            tCurrent = t;
                        }
                    }
                }
            }
            var tRelative = Math.min( current ? ( tCurrent - Date.now() ) / 1000 : -document.documentElement.getCurrentTime(), 0 );
            var set = jSignage.createElement( 'set', { attributeName: 'display', to: 'inherit', dur: 'indefinite', begin: 'indefinite' } );
            $( 'svg' ).add( set );
            $.beginEvent( set[0], function () {
                trigger_callback( current );
                set.remove();
            } );
            $.beginLayerAt( set[0], tRelative );
            trig_next( t0 );
        }

        function trig_next( t0 ) {
            if ( nextTimer !== null ) {
                $.clearTimeout( nextTimer );
                nextTimer = null;
            }
            var next = null, tNext = 0;
            if ( triggers ) for ( var i = 0; i < triggers.length; i++ ) {
                var trigger = triggers[i];
                if ( !schedule[trigger.start] )
                    continue;
                for ( var j = 0; j < schedule[trigger.start].length; j++ ) {
                    var t = schedule[trigger.start][j] + ( trigger.offset || 0 ) * 60000;
                    if ( t > t0 ) {
                        if ( next === null || t < tNext ) {
                            next = trigger;
                            tNext = t;
                        }
                    }
                }
            }
            if ( next ) {
                //alert( 'Next layer is for ' + next.start + ' in ' + ( tNext - Date.now() ) / 1000 + 's' );
                nextTimer = $.setTimeout( function () {
                    trigger_callback( next );
                    trig_next( tNext );
                }, Math.max( tNext - Date.now(), 0 ) );
            }
        }

        return {
            start: function () {
                refreshSchedule();
                refreshTimer = $.setInterval( function () {
                    refreshSchedule();
                    trig_next( Date.now() );
                }, 43200000 );
                trig_current( Date.now() );
            },

            stop: function () {
                if ( refreshTimer ) {
                    $.clearInterval( refreshTimer );
                    refreshTimer = null;
                }
                if ( nextTimer ) {
                    $.clearTimeout( nextTimer );
                    nextTimer = null;
                }
            }
        };
    }

};

} )();

( function () {
/*
    (c) 2011-2014, Vladimir Agafonkin
    SunCalc is a JavaScript library for calculating sun/mooon position and light phases.
    https://github.com/mourner/suncalc
*/

    // shortcuts for easier to read formulas

    var PI = Math.PI,
        sin = Math.sin,
        cos = Math.cos,
        tan = Math.tan,
        asin = Math.asin,
        atan = Math.atan2,
        acos = Math.acos,
        rad = PI / 180;

    // sun calculations are based on http://aa.quae.nl/en/reken/zonpositie.html formulas


    // date/time constants and conversions

    var dayMs = 1000 * 60 * 60 * 24,
        J1970 = 2440588,
        J2000 = 2451545;

    function toJulian( date ) { return date.valueOf() / dayMs - 0.5 + J1970; }
    function fromJulian( j ) { return new Date(( j + 0.5 - J1970 ) * dayMs ); }
    function toDays( date ) { return toJulian( date ) - J2000; }


    // general calculations for position

    var e = rad * 23.4397; // obliquity of the Earth

    function rightAscension( l, b ) { return atan( sin( l ) * cos( e ) - tan( b ) * sin( e ), cos( l ) ); }
    function declination( l, b ) { return asin( sin( b ) * cos( e ) + cos( b ) * sin( e ) * sin( l ) ); }

    function azimuth( H, phi, dec ) { return atan( sin( H ), cos( H ) * sin( phi ) - tan( dec ) * cos( phi ) ); }
    function altitude( H, phi, dec ) { return asin( sin( phi ) * sin( dec ) + cos( phi ) * cos( dec ) * cos( H ) ); }

    function siderealTime( d, lw ) { return rad * ( 280.16 + 360.9856235 * d ) - lw; }


    // general sun calculations

    function solarMeanAnomaly( d ) { return rad * ( 357.5291 + 0.98560028 * d ); }

    function eclipticLongitude( M ) {

        var C = rad * ( 1.9148 * sin( M ) + 0.02 * sin( 2 * M ) + 0.0003 * sin( 3 * M ) ), // equation of center
            P = rad * 102.9372; // perihelion of the Earth

        return M + C + P + PI;
    }

    function sunCoords( d ) {

        var M = solarMeanAnomaly( d ),
            L = eclipticLongitude( M );

        return {
            dec: declination( L, 0 ),
            ra: rightAscension( L, 0 )
        };
    }

    var SunCalc = {};

    // calculates sun position for a given date and latitude/longitude

    SunCalc.getPosition = function ( date, lat, lng ) {

        var lw = rad * -lng,
            phi = rad * lat,
            d = toDays( date ),

            c = sunCoords( d ),
            H = siderealTime( d, lw ) - c.ra;

        return {
            azimuth: azimuth( H, phi, c.dec ),
            altitude: altitude( H, phi, c.dec )
        };
    };


    // sun times configuration (angle, morning name, evening name)

    var times = SunCalc.times = [
        [-0.833, 'sunrise', 'sunset'],
        [-0.3, 'sunriseEnd', 'sunsetStart'],
        [-6, 'dawn', 'dusk'],
        [-12, 'nauticalDawn', 'nauticalDusk'],
        [-18, 'astronomicalDawn', 'astronomicalDusk'],
    ];

    // adds a custom time to the times config

    SunCalc.addTime = function ( angle, riseName, setName ) {
        times.push( [angle, riseName, setName] );
    };

    // calculations for sun times

    var J0 = 0.0009;

    function julianCycle( d, lw ) { return Math.round( d - J0 - lw / ( 2 * PI ) ); }

    function approxTransit( Ht, lw, n ) { return J0 + ( Ht + lw ) / ( 2 * PI ) + n; }
    function solarTransitJ( ds, M, L ) { return J2000 + ds + 0.0053 * sin( M ) - 0.0069 * sin( 2 * L ); }

    function hourAngle( h, phi, d ) {
        var cos_w = ( sin( h ) - sin( phi ) * sin( d ) ) / ( cos( phi ) * cos( d ) );
        if ( cos_w > 1 )
            return 0;
        else if ( cos_w < -1 )
            return Math.PI;
        else
            return acos( cos_w );
    }

    // returns set time for the given sun altitude
    function getSetJ( h, lw, phi, dec, n, M, L ) {

        var w = hourAngle( h, phi, dec ),
            a = approxTransit( w, lw, n );
        return solarTransitJ( a, M, L );
    }

    // calculates sun times for a given date and latitude/longitude

    SunCalc.getTimes = function ( date, lat, lng, elevation ) {

        var lw = rad * -lng,
            phi = rad * lat,

            d = toDays( date ),
            n = julianCycle( d, lw ),
            ds = approxTransit( 0, lw, n ),

            M = solarMeanAnomaly( ds ),
            L = eclipticLongitude( M ),
            dec = declination( L, 0 ),

            Jnoon = solarTransitJ( ds, M, L ),

            i, len, time, Jset, Jrise;


        var result = {
            solarNoon: fromJulian( Jnoon ),
            nadir: fromJulian( Jnoon - 0.5 )
        };

        for ( i = 0, len = times.length; i < len; i += 1 ) {
            time = times[i];

            Jset = getSetJ( ( time[0] - 2.076 * Math.sqrt(elevation) / 60 ) * rad, lw, phi, dec, n, M, L );
            Jrise = Jnoon - ( Jset - Jnoon );

            result[time[1]] = fromJulian( Jrise );
            result[time[2]] = fromJulian( Jset );
        }

        return result;
    };

    jSignage.Astronomy.sunriseEquation = function ( year, month, day, latitude, longitude, elevation, tune ) {
        if ( !elevation || elevation < 0 )
            elevation = 0;

        var date = new Date( Math.floor( Date.UTC( year, month - 1, day, 12, 0, 0 ) - longitude / 360 * 86400000 ) );
        var T = SunCalc.getTimes( date, latitude, longitude, elevation );
        for ( var i in T )
            if ( tune && tune[i] )
                T[i] = new Date( T[i].valueOf() + tune[i] * 60000 );
        return T;
    };

    jSignage.Astronomy.getSunPosition = function ( date, latitude, longitude ) {
        return SunCalc.getPosition( date, latitude, longitude );
    };

} )();

( function () {

//--------------------- Copyright Block ----------------------
/* 

PrayTimes.js: Prayer Times Calculator (ver 2.3)
Copyright (C) 2007-2011 PrayTimes.org

Developer: Hamid Zarrabi-Zadeh
License: GNU LGPL v3.0

TERMS OF USE:
	Permission is granted to use this code, with or 
	without modification, in any website or application 
	provided that credit is given to the original work 
	with a link back to PrayTimes.org.

This program is distributed in the hope that it will 
be useful, but WITHOUT ANY WARRANTY. 

PLEASE DO NOT REMOVE THIS COPYRIGHT BLOCK.
 
*/


//--------------------- Help and Manual ----------------------
/*

User's Manual: 
http://praytimes.org/manual

Calculation Formulas: 
http://praytimes.org/calculation



//------------------------ User Interface -------------------------


	getTimes (date, coordinates [, timeZone [, dst [, timeFormat]]]) 
	
	setMethod (method)       // set calculation method 
	adjust (parameters)      // adjust calculation parameters	
	tune (offsets)           // tune times by given offsets 

	getMethod ()             // get calculation method 
	getSetting ()            // get current calculation parameters
	getOffsets ()            // get current time offsets


//------------------------- Sample Usage --------------------------


	var PT = new PrayTimes('ISNA');
	var times = PT.getTimes(new Date(), [43, -80], -5);
	document.write('Sunrise = '+ times.sunrise)


*/


//----------------------- PrayTimes Class ------------------------


jSignage.Astronomy.PrayTimes = function ( method ) {


    //------------------------ Constants --------------------------
    var

	// Time Names
	timeNames = {
	    imsak: 'Imsak',
	    fajr: 'Fajr',
	    sunrise: 'Sunrise',
	    dhuhr: 'Dhuhr',
	    asr: 'Asr',
	    sunset: 'Sunset',
	    maghrib: 'Maghrib',
	    isha: 'Isha',
	    midnight: 'Midnight'
	},


	// Calculation Methods
	methods = {
	    MWL: {
	        name: 'Muslim World League',
	        params: { fajr: 18, isha: 17 }
	    },
	    ISNA: {
	        name: 'Islamic Society of North America (ISNA)',
	        params: { fajr: 15, isha: 15 }
	    },
	    Egypt: {
	        name: 'Egyptian General Authority of Survey',
	        params: { fajr: 19.5, isha: 17.5 }
	    },
	    Makkah: {
	        name: 'Umm Al-Qura University, Makkah',
	        params: { fajr: 18.5, isha: '90 min' }
	    },  // fajr was 19 degrees before 1430 hijri
	    Karachi: {
	        name: 'University of Islamic Sciences, Karachi',
	        params: { fajr: 18, isha: 18 }
	    },
	    Tehran: {
	        name: 'Institute of Geophysics, University of Tehran',
	        params: { fajr: 17.7, isha: 14, maghrib: 4.5, midnight: 'Jafari' }
	    },  // isha is not explicitly specified in this method
	    Jafari: {
	        name: 'Shia Ithna-Ashari, Leva Institute, Qum',
	        params: { fajr: 16, isha: 14, maghrib: 4, midnight: 'Jafari' }
	    }
	},


	// Default Parameters in Calculation Methods
	defaultParams = {
	    maghrib: '0 min', midnight: 'Standard'

	},


	//----------------------- Parameter Values ----------------------
	/*
	
	// Asr Juristic Methods
	asrJuristics = [ 
		'Standard',    // Shafi`i, Maliki, Ja`fari, Hanbali
		'Hanafi'       // Hanafi
	],


	// Midnight Mode
	midnightMethods = [ 
		'Standard',    // Mid Sunset to Sunrise
		'Jafari'       // Mid Sunset to Fajr
	],


	// Adjust Methods for Higher Latitudes
	highLatMethods = [
		'NightMiddle', // middle of night
		'AngleBased',  // angle/60th of night
		'OneSeventh',  // 1/7th of night
		'None'         // No adjustment
	],


	// Time Formats
	timeFormats = [
		'24h',         // 24-hour format
		'12h',         // 12-hour format
		'12hNS',       // 12-hour format with no suffix
		'Float'        // floating point number 
	],
	*/


	//---------------------- Default Settings --------------------

	calcMethod = 'MWL',

	// do not change anything here; use adjust method instead
	setting = {
	    imsak: '10 min',
	    dhuhr: '0 min',
	    asr: 'Standard',
	    highLats: 'NightMiddle'
	},

	timeFormat = '24h',
	timeSuffixes = ['am', 'pm'],
	invalidTime = '-----',

	numIterations = 3,
	offset = {},


	//----------------------- Local Variables ---------------------

	lat, lng, elv,       // coordinates
	timeZone, jDate;     // time variables


    //---------------------- Initialization -----------------------


    // set methods defaults
    var defParams = defaultParams;
    for ( var i in methods ) {
        var params = methods[i].params;
        for ( var j in defParams )
            if ( ( typeof ( params[j] ) == 'undefined' ) )
                params[j] = defParams[j];
    };

    // initialize settings
    calcMethod = methods[method] ? method : calcMethod;
    var params = methods[calcMethod].params;
    for ( var id in params )
        setting[id] = params[id];

    // init time offsets
    for ( var i in timeNames )
        offset[i] = 0;



    //----------------------- Public Functions ------------------------
    return {


        // set calculation method 
        setMethod: function ( method ) {
            if ( methods[method] ) {
                this.adjust( methods[method].params );
                calcMethod = method;
            }
        },


        // set calculating parameters
        adjust: function ( params ) {
            for ( var id in params )
                setting[id] = params[id];
        },


        // set time offsets
        tune: function ( timeOffsets ) {
            for ( var i in timeOffsets )
                offset[i] = timeOffsets[i];
        },


        // get current calculation method
        getMethod: function () { return calcMethod; },

        // get current setting
        getSetting: function () { return setting; },

        // get current time offsets
        getOffsets: function () { return offset; },

        // get default calc parametrs
        getDefaults: function () { return methods; },


        // return prayer times for a given date
        getTimes: function ( date, coords, timezone, dst, format ) {
            lat = 1 * coords[0];
            lng = 1 * coords[1];
            elv = coords[2] ? 1 * coords[2] : 0;
            timeFormat = format || timeFormat;
            if ( date.constructor === Date )
                date = [date.getFullYear(), date.getMonth() + 1, date.getDate()];
            if ( typeof ( timezone ) == 'undefined' || timezone == 'auto' )
                timezone = this.getTimeZone( date );
            if ( typeof ( dst ) == 'undefined' || dst == 'auto' )
                dst = this.getDst( date );
            timeZone = 1 * timezone + ( 1 * dst ? 1 : 0 );
            jDate = this.julian( date[0], date[1], date[2] ) - lng / ( 15 * 24 );

            return this.computeTimes();
        },


        // convert float time to the given format (see timeFormats)
        getFormattedTime: function ( time, format, suffixes ) {
            if ( isNaN( time ) )
                return invalidTime;
            if ( format == 'Float' ) return time;
            suffixes = suffixes || timeSuffixes;

            time = DMath.fixHour( time + 0.5 / 60 );  // add 0.5 minutes to round
            var hours = Math.floor( time );
            var minutes = Math.floor(( time - hours ) * 60 );
            var suffix = ( format == '12h' ) ? suffixes[hours < 12 ? 0 : 1] : '';
            var hour = ( format == '24h' ) ? this.twoDigitsFormat( hours ) : ( ( hours + 12 - 1 ) % 12 + 1 );
            return hour + ':' + this.twoDigitsFormat( minutes ) + ( suffix ? ' ' + suffix : '' );
        },


        //---------------------- Calculation Functions -----------------------


        // compute mid-day time
        midDay: function ( time ) {
            var eqt = this.sunPosition( jDate + time ).equation;
            var noon = DMath.fixHour( 12 - eqt );
            return noon;
        },


        // compute the time at which sun reaches a specific angle below horizon
        sunAngleTime: function ( angle, time, direction ) {
            var decl = this.sunPosition( jDate + time ).declination;
            var noon = this.midDay( time );
            var t = 1 / 15 * DMath.arccos(( -DMath.sin( angle ) - DMath.sin( decl ) * DMath.sin( lat ) ) /
                    ( DMath.cos( decl ) * DMath.cos( lat ) ) );
            return noon + ( direction == 'ccw' ? -t : t );
        },


        // compute asr time 
        asrTime: function ( factor, time ) {
            var decl = this.sunPosition( jDate + time ).declination;
            var angle = -DMath.arccot( factor + DMath.tan( Math.abs( lat - decl ) ) );
            return this.sunAngleTime( angle, time );
        },


        // compute declination angle of sun and equation of time
        // Ref: http://aa.usno.navy.mil/faq/docs/SunApprox.php
        sunPosition: function ( jd ) {
            var D = jd - 2451545.0;
            var g = DMath.fixAngle( 357.529 + 0.98560028 * D );
            var q = DMath.fixAngle( 280.459 + 0.98564736 * D );
            var L = DMath.fixAngle( q + 1.915 * DMath.sin( g ) + 0.020 * DMath.sin( 2 * g ) );

            var R = 1.00014 - 0.01671 * DMath.cos( g ) - 0.00014 * DMath.cos( 2 * g );
            var e = 23.439 - 0.00000036 * D;

            var RA = DMath.arctan2( DMath.cos( e ) * DMath.sin( L ), DMath.cos( L ) ) / 15;
            var eqt = q / 15 - DMath.fixHour( RA );
            var decl = DMath.arcsin( DMath.sin( e ) * DMath.sin( L ) );

            return { declination: decl, equation: eqt };
        },


        // convert Gregorian date to Julian day
        // Ref: Astronomical Algorithms by Jean Meeus
        julian: function ( year, month, day ) {
            if ( month <= 2 ) {
                year -= 1;
                month += 12;
            };
            var A = Math.floor( year / 100 );
            var B = 2 - A + Math.floor( A / 4 );

            var JD = Math.floor( 365.25 * ( year + 4716 ) ) + Math.floor( 30.6001 * ( month + 1 ) ) + day + B - 1524.5;
            return JD;
        },


        //---------------------- Compute Prayer Times -----------------------


        // compute prayer times at given julian date
        computePrayerTimes: function ( times ) {
            times = this.dayPortion( times );
            var params = setting;

            var imsak = this.sunAngleTime( this.eval( params.imsak ), times.imsak, 'ccw' );
            var fajr = this.sunAngleTime( this.eval( params.fajr ), times.fajr, 'ccw' );
            var sunrise = this.sunAngleTime( this.riseSetAngle(), times.sunrise, 'ccw' );
            var dhuhr = this.midDay( times.dhuhr );
            var asr = this.asrTime( this.asrFactor( params.asr ), times.asr );
            var sunset = this.sunAngleTime( this.riseSetAngle(), times.sunset );;
            var maghrib = this.sunAngleTime( this.eval( params.maghrib ), times.maghrib );
            var isha = this.sunAngleTime( this.eval( params.isha ), times.isha );

            return {
                imsak: imsak, fajr: fajr, sunrise: sunrise, dhuhr: dhuhr,
                asr: asr, sunset: sunset, maghrib: maghrib, isha: isha
            };
        },


        // compute prayer times 
        computeTimes: function () {
            // default times
            var times = {
                imsak: 5, fajr: 5, sunrise: 6, dhuhr: 12,
                asr: 13, sunset: 18, maghrib: 18, isha: 18
            };

            // main iterations
            for ( var i = 1 ; i <= numIterations ; i++ )
                times = this.computePrayerTimes( times );

            times = this.adjustTimes( times );

            // add midnight time
            times.midnight = ( setting.midnight == 'Jafari' ) ?
                    times.sunset + this.timeDiff( times.sunset, times.fajr ) / 2 :
                    times.sunset + this.timeDiff( times.sunset, times.sunrise ) / 2;

            times = this.tuneTimes( times );
            return this.modifyFormats( times );
        },


        // adjust times 
        adjustTimes: function ( times ) {
            var params = setting;
            for ( var i in times )
                times[i] += timeZone - lng / 15;

            if ( params.highLats != 'None' )
                times = this.adjustHighLats( times );

            if ( this.isMin( params.imsak ) )
                times.imsak = times.fajr - this.eval( params.imsak ) / 60;
            if ( this.isMin( params.maghrib ) )
                times.maghrib = times.sunset + this.eval( params.maghrib ) / 60;
            if ( this.isMin( params.isha ) )
                times.isha = times.maghrib + this.eval( params.isha ) / 60;
            times.dhuhr += this.eval( params.dhuhr ) / 60;

            return times;
        },


        // get asr shadow factor
        asrFactor: function ( asrParam ) {
            var factor = { Standard: 1, Hanafi: 2 }[asrParam];
            return factor || this.eval( asrParam );
        },


        // return sun angle for sunset/sunrise
        riseSetAngle: function () {
            //var earthRad = 6371009; // in meters
            //var angle = DMath.arccos(earthRad/(earthRad+ elv));
            var angle = 0.0347 * Math.sqrt( elv ); // an approximation
            return 0.833 + angle;
        },


        // apply offsets to the times
        tuneTimes: function ( times ) {
            for ( var i in times )
                times[i] += offset[i] / 60;
            return times;
        },


        // convert times to given time format
        modifyFormats: function ( times ) {
            for ( var i in times )
                times[i] = this.getFormattedTime( times[i], timeFormat );
            return times;
        },


        // adjust times for locations in higher latitudes
        adjustHighLats: function ( times ) {
            var params = setting;
            var nightTime = this.timeDiff( times.sunset, times.sunrise );

            times.imsak = this.adjustHLTime( times.imsak, times.sunrise, this.eval( params.imsak ), nightTime, 'ccw' );
            times.fajr = this.adjustHLTime( times.fajr, times.sunrise, this.eval( params.fajr ), nightTime, 'ccw' );
            times.isha = this.adjustHLTime( times.isha, times.sunset, this.eval( params.isha ), nightTime );
            times.maghrib = this.adjustHLTime( times.maghrib, times.sunset, this.eval( params.maghrib ), nightTime );

            return times;
        },


        // adjust a time for higher latitudes
        adjustHLTime: function ( time, base, angle, night, direction ) {
            var portion = this.nightPortion( angle, night );
            var timeDiff = ( direction == 'ccw' ) ?
                this.timeDiff( time, base ) :
                this.timeDiff( base, time );
            if ( isNaN( time ) || timeDiff > portion )
                time = base + ( direction == 'ccw' ? -portion : portion );
            return time;
        },


        // the night portion used for adjusting times in higher latitudes
        nightPortion: function ( angle, night ) {
            var method = setting.highLats;
            var portion = 1 / 2 // MidNight
            if ( method == 'AngleBased' )
                portion = 1 / 60 * angle;
            if ( method == 'OneSeventh' )
                portion = 1 / 7;
            return portion * night;
        },


        // convert hours to day portions 
        dayPortion: function ( times ) {
            for ( var i in times )
                times[i] /= 24;
            return times;
        },


        //---------------------- Time Zone Functions -----------------------


        // get local time zone
        getTimeZone: function ( date ) {
            var year = date[0];
            var t1 = this.gmtOffset( [year, 0, 1] );
            var t2 = this.gmtOffset( [year, 6, 1] );
            return Math.min( t1, t2 );
        },


        // get daylight saving for a given date
        getDst: function ( date ) {
            return 1 * ( this.gmtOffset( date ) != this.getTimeZone( date ) );
        },


        // GMT offset for a given date
        gmtOffset: function ( date ) {
            var localDate = new Date( date[0], date[1] - 1, date[2], 12, 0, 0, 0 );
            var GMTString = localDate.toGMTString();
            var GMTDate = new Date( GMTString.substring( 0, GMTString.lastIndexOf( ' ' ) - 1 ) );
            var hoursDiff = ( localDate - GMTDate ) / ( 1000 * 60 * 60 );
            return hoursDiff;
        },


        //---------------------- Misc Functions -----------------------

        // convert given string into a number
        eval: function ( str ) {
            return 1 * ( str + '' ).split( /[^0-9.+-]/ )[0];
        },


        // detect if input contains 'min'
        isMin: function ( arg ) {
            return ( arg + '' ).indexOf( 'min' ) != -1;
        },


        // compute the difference between two times 
        timeDiff: function ( time1, time2 ) {
            return DMath.fixHour( time2 - time1 );
        },


        // add a leading 0 if necessary
        twoDigitsFormat: function ( num ) {
            return ( num < 10 ) ? '0' + num : num;
        }

    }
}



//---------------------- Degree-Based Math Class -----------------------


var DMath = {

    dtr: function ( d ) { return ( d * Math.PI ) / 180.0; },
    rtd: function ( r ) { return ( r * 180.0 ) / Math.PI; },

    sin: function ( d ) { return Math.sin( this.dtr( d ) ); },
    cos: function ( d ) { return Math.cos( this.dtr( d ) ); },
    tan: function ( d ) { return Math.tan( this.dtr( d ) ); },

    arcsin: function ( d ) { return this.rtd( Math.asin( d ) ); },
    arccos: function ( d ) { return this.rtd( Math.acos( d ) ); },
    arctan: function ( d ) { return this.rtd( Math.atan( d ) ); },

    arccot: function ( x ) { return this.rtd( Math.atan( 1 / x ) ); },
    arctan2: function ( y, x ) { return this.rtd( Math.atan2( y, x ) ); },

    fixAngle: function ( a ) { return this.fix( a, 360 ); },
    fixHour: function ( a ) { return this.fix( a, 24 ); },

    fix: function ( a, b ) {
        a = a - b * ( Math.floor( a / b ) );
        return ( a < 0 ) ? a + b : a;
    }
}


//---------------------- Init Object -----------------------

} )();

( function () {
    // Extensions to the Math routines - Trig routines in degrees
    // JavaScript by Peter Hayes http://www.peter-hayes.freeserve.co.uk/
    // Copyright 2001-2002

    function rev( angle ) { return angle - Math.floor( angle / 360.0 ) * 360.0; }
    function sind( angle ) { return Math.sin(( angle * Math.PI ) / 180.0 ); }
    function cosd( angle ) { return Math.cos(( angle * Math.PI ) / 180.0 ); }
    function tand( angle ) { return Math.tan(( angle * Math.PI ) / 180.0 ); }
    function asind( c ) { return ( 180.0 / Math.PI ) * Math.asin( c ); }
    function acosd( c ) { return ( 180.0 / Math.PI ) * Math.acos( c ); }
    function atan2d( y, x ) { return ( 180.0 / Math.PI ) * Math.atan( y / x ) - 180.0 * ( x < 0 ); }

    function anglestring( a, circle ) {
        // returns a in degrees as a string degrees:minutes
        // circle is true for range between 0 and 360 and false for -90 to +90
        var ar = Math.round( a * 60 ) / 60;
        var deg = Math.abs( ar );
        var min = Math.round( 60.0 * ( deg - Math.floor( deg ) ) );
        if ( min >= 60 ) { deg += 1; min = 0; }
        var anglestr = "";
        if ( !circle ) anglestr += ( ar < 0 ? "-" : "+" );
        if ( circle ) anglestr += ( ( Math.floor( deg ) < 100 ) ? "0" : "" );
        anglestr += ( ( Math.floor( deg ) < 10 ) ? "0" : "" ) + Math.floor( deg );
        anglestr += ( ( min < 10 ) ? ":0" : ":" ) + ( min );
        return anglestr;
    }

    // Utility functions for astronomical programming.
    // JavaScript by Peter Hayes http://www.peter-hayes.freeserve.co.uk/
    // Copyright 2001-2002
    // This code is made freely available but please keep this notice.
    // I accept no liability for any errors in my coding but please
    // let me know of any errors you find. My address is on my home page.

    function dayno( year, month, day, hours ) {
        // Day number is a modified Julian date, day 0 is 2000 January 0.0
        // which corresponds to a Julian date of 2451543.5
        var d = 367 * year - Math.floor( 7 * ( year + Math.floor(( month + 9 ) / 12 ) ) / 4 )
               + Math.floor(( 275 * month ) / 9 ) + day - 730530 + hours / 24;
        return d;
    }

    function julian( year, month, day, hours ) {
        return dayno( year, month, day, hours ) + 2451543.5
    }

    function jdtocd( jd ) {
        // The calendar date from julian date
        // Returns year, month, day, day of week, hours, minutes, seconds
        var Z = Math.floor( jd + 0.5 );
        var F = jd + 0.5 - Z;
        if ( Z < 2299161 ) {
            var A = Z;
        } else {
            var alpha = Math.floor(( Z - 1867216.25 ) / 36524.25 );
            var A = Z + 1 + alpha - Math.floor( alpha / 4 );
        }
        var B = A + 1524;
        var C = Math.floor(( B - 122.1 ) / 365.25 );
        var D = Math.floor( 365.25 * C );
        var E = Math.floor(( B - D ) / 30.6001 );
        var d = B - D - Math.floor( 30.6001 * E ) + F;
        if ( E < 14 ) { var month = E - 1; } else { var month = E - 13; }
        if ( month > 2 ) { var year = C - 4716; } else { var year = C - 4715; }
        var day = Math.floor( d );
        var h = ( d - day ) * 24;
        var hours = Math.floor( h );
        var m = ( h - hours ) * 60;
        var minutes = Math.floor( m );
        var seconds = Math.round(( m - minutes ) * 60 );
        if ( seconds >= 60 ) { minutes = minutes + 1; seconds = seconds - 60; }
        if ( minutes >= 60 ) { hours = hours + 1; minutes = 0; }
        var dw = Math.floor( jd + 1.5 ) - 7 * Math.floor(( jd + 1.5 ) / 7 );
        return new Array( year, month, day, dw, hours, minutes, seconds );
    }

    function local_sidereal( year, month, day, hours, lon ) {
        // Compute local siderial time in degrees
        // year, month, day and hours are the Greenwich date and time
        // lon is the observers longitude
        var d = dayno( year, month, day, hours );
        var lst = ( 98.9818 + 0.985647352 * d + hours * 15 + lon );
        return rev( lst ) / 15;
    }

    function radtoaa( ra, dec, year, month, day, hours, lat, lon ) {
        // convert ra and dec to altitude and azimuth
        // year, month, day and hours are the Greenwich date and time
        // lat and lon are the observers latitude and longitude
        var lst = local_sidereal( year, month, day, hours, lon );
        var x = cosd( 15.0 * ( lst - ra ) ) * cosd( dec );
        var y = sind( 15.0 * ( lst - ra ) ) * cosd( dec );
        var z = sind( dec );
        // rotate so z is the local zenith
        var xhor = x * sind( lat ) - z * cosd( lat );
        var yhor = y;
        var zhor = x * cosd( lat ) + z * sind( lat );
        var azimuth = rev( atan2d( yhor, xhor ) + 180.0 ); // so 0 degrees is north
        var altitude = atan2d( zhor, Math.sqrt( xhor * xhor + yhor * yhor ) );
        return new Array( altitude, azimuth );
    }

    // JavaScript by Peter Hayes http://www.aphayes.pwp.blueyonder.co.uk/
    // Copyright 2001-2010
    // Unless otherwise stated this code is based on the methods in
    // Astronomical Algorithms, first edition, by Jean Meeus
    // Published by Willmann-Bell, Inc.
    // This code is made freely available but please keep this notice.
    // The calculations are approximate but should be good enough for general use,
    // I accept no responsibility for errors in astronomy or coding.

    // WARNING moonrise code changed on 6 May 2003 to correct a systematic error 
    // these are now local times NOT UTC as the original code did.

    // Meeus first edition table 45.A Longitude and distance of the moon

    var T45AD = new Array( 0, 2, 2, 0, 0, 0, 2, 2, 2, 2,
                          0, 1, 0, 2, 0, 0, 4, 0, 4, 2,
                          2, 1, 1, 2, 2, 4, 2, 0, 2, 2,
                          1, 2, 0, 0, 2, 2, 2, 4, 0, 3,
                          2, 4, 0, 2, 2, 2, 4, 0, 4, 1,
                          2, 0, 1, 3, 4, 2, 0, 1, 2, 2 );

    var T45AM = new Array( 0, 0, 0, 0, 1, 0, 0, -1, 0, -1,
                          1, 0, 1, 0, 0, 0, 0, 0, 0, 1,
                          1, 0, 1, -1, 0, 0, 0, 1, 0, -1,
                          0, -2, 1, 2, -2, 0, 0, -1, 0, 0,
                          1, -1, 2, 2, 1, -1, 0, 0, -1, 0,
                          1, 0, 1, 0, 0, -1, 2, 1, 0, 0 );

    var T45AMP = new Array( 1, -1, 0, 2, 0, 0, -2, -1, 1, 0,
                           -1, 0, 1, 0, 1, 1, -1, 3, -2, -1,
                            0, -1, 0, 1, 2, 0, -3, -2, -1, -2,
                            1, 0, 2, 0, -1, 1, 0, -1, 2, -1,
                            1, -2, -1, -1, -2, 0, 1, 4, 0, -2,
                            0, 2, 1, -2, -3, 2, 1, -1, 3, -1 );

    var T45AF = new Array( 0, 0, 0, 0, 0, 2, 0, 0, 0, 0,
                            0, 0, 0, -2, 2, -2, 0, 0, 0, 0,
                            0, 0, 0, 0, 0, 0, 0, 0, 2, 0,
                            0, 0, 0, 0, 0, -2, 2, 0, 2, 0,
                            0, 0, 0, 0, 0, -2, 0, 0, 0, 0,
                           -2, -2, 0, 0, 0, 0, 0, 0, 0, -2 );

    var T45AL = new Array( 6288774, 1274027, 658314, 213618, -185116,
                          -114332, 58793, 57066, 53322, 45758,
                           -40923, -34720, -30383, 15327, -12528,
                            10980, 10675, 10034, 8548, -7888,
                            -6766, -5163, 4987, 4036, 3994,
                             3861, 3665, -2689, -2602, 2390,
                            -2348, 2236, -2120, -2069, 2048,
                            -1773, -1595, 1215, -1110, -892,
                             -810, 759, -713, -700, 691,
                              596, 549, 537, 520, -487,
                             -399, -381, 351, -340, 330,
                              327, -323, 299, 294, 0 );

    var T45AR = new Array( -20905355, -3699111, -2955968, -569925, 48888,
                              -3149, 246158, -152138, -170733, -204586,
                            -129620, 108743, 104755, 10321, 0,
                              79661, -34782, -23210, -21636, 24208,
                              30824, -8379, -16675, -12831, -10445,
                             -11650, 14403, -7003, 0, 10056,
                               6322, -9884, 5751, 0, -4950,
                               4130, 0, -3958, 0, 3258,
                               2616, -1897, -2117, 2354, 0,
                                  0, -1423, -1117, -1571, -1739,
                                  0, -4421, 0, 0, 0,
                                  0, 1165, 0, 0, 8752 );

    // Meeus table 45B latitude of the moon

    var T45BD = new Array( 0, 0, 0, 2, 2, 2, 2, 0, 2, 0,
                          2, 2, 2, 2, 2, 2, 2, 0, 4, 0,
                          0, 0, 1, 0, 0, 0, 1, 0, 4, 4,
                          0, 4, 2, 2, 2, 2, 0, 2, 2, 2,
                          2, 4, 2, 2, 0, 2, 1, 1, 0, 2,
                          1, 2, 0, 4, 4, 1, 4, 1, 4, 2 );

    var T45BM = new Array( 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                          -1, 0, 0, 1, -1, -1, -1, 1, 0, 1,
                           0, 1, 0, 1, 1, 1, 0, 0, 0, 0,
                           0, 0, 0, 0, -1, 0, 0, 0, 0, 1,
                           1, 0, -1, -2, 0, 1, 1, 1, 1, 1,
                           0, -1, 1, 0, -1, 0, 0, 0, -1, -2 );

    var T45BMP = new Array( 0, 1, 1, 0, -1, -1, 0, 2, 1, 2,
                           0, -2, 1, 0, -1, 0, -1, -1, -1, 0,
                           0, -1, 0, 1, 1, 0, 0, 3, 0, -1,
                           1, -2, 0, 2, 1, -2, 3, 2, -3, -1,
                           0, 0, 1, 0, 1, 1, 0, 0, -2, -1,
                           1, -2, 2, -2, -1, 1, 1, -1, 0, 0 );

    var T45BF = new Array( 1, 1, -1, -1, 1, -1, 1, 1, -1, -1,
                          -1, -1, 1, -1, 1, 1, -1, -1, -1, 1,
                           3, 1, 1, 1, -1, -1, -1, 1, -1, 1,
                          -3, 1, -3, -1, -1, 1, -1, 1, -1, 1,
                           1, 1, 1, -1, 3, -1, -1, 1, -1, -1,
                           1, -1, 1, -1, -1, -1, -1, -1, -1, 1 );

    var T45BL = new Array( 5128122, 280602, 277693, 173237, 55413,
                            46271, 32573, 17198, 9266, 8822,
                             8216, 4324, 4200, -3359, 2463,
                             2211, 2065, -1870, 1828, -1794,
                            -1749, -1565, -1491, -1475, -1410,
                            -1344, -1335, 1107, 1021, 833,
                              777, 671, 607, 596, 491,
                             -451, 439, 422, 421, -366,
                             -351, 331, 315, 302, -283,
                             -229, 223, 223, -220, -220,
                             -185, 181, -177, 176, 166,
                             -164, 132, -119, 115, 107 );

    // MoonPos calculates the Moon position, based on Meeus chapter 45

    function MoonPos( year, month, day, hours ) {
        // julian date
        var jd = julian( year, month, day, hours );
        var T = ( jd - 2451545.0 ) / 36525;
        var T2 = T * T;
        var T3 = T2 * T;
        var T4 = T3 * T;
        // Moons mean longitude L'
        var LP = 218.3164477 + 481267.88123421 * T - 0.0015786 * T2 + T3 / 538841.0 - T4 / 65194000.0;
        // Moons mean elongation
        var D = 297.8501921 + 445267.1114034 * T - 0.0018819 * T2 + T3 / 545868.0 - T4 / 113065000.0;
        // Suns mean anomaly
        var M = 357.5291092 + 35999.0502909 * T - 0.0001536 * T2 + T3 / 24490000.0;
        // Moons mean anomaly M'
        var MP = 134.9633964 + 477198.8675055 * T + 0.0087414 * T2 + T3 / 69699.0 - T4 / 14712000.0;
        // Moons argument of latitude
        var F = 93.2720950 + 483202.0175233 * T - 0.0036539 * T2 - T3 / 3526000.0 + T4 / 863310000.0;

        // Additional arguments
        var A1 = 119.75 + 131.849 * T;
        var A2 = 53.09 + 479264.290 * T;
        var A3 = 313.45 + 481266.484 * T;
        var E = 1 - 0.002516 * T - 0.0000074 * T2;
        var E2 = E * E;
        // Sums of periodic terms from table 45.A and 45.B
        var Sl = 0.0;
        var Sr = 0.0;
        for ( var i = 0; i < 60; i++ ) {
            var Eterm = 1;
            if ( Math.abs( T45AM[i] ) == 1 ) Eterm = E;
            if ( Math.abs( T45AM[i] ) == 2 ) Eterm = E2;
            Sl += T45AL[i] * Eterm * sind( rev( T45AD[i] * D + T45AM[i] * M + T45AMP[i] * MP + T45AF[i] * F ) );
            Sr += T45AR[i] * Eterm * cosd( rev( T45AD[i] * D + T45AM[i] * M + T45AMP[i] * MP + T45AF[i] * F ) );
        }
        var Sb = 0.0;
        for ( var i = 0; i < 60; i++ ) {
            var Eterm = 1;
            if ( Math.abs( T45BM[i] ) == 1 ) Eterm = E;
            if ( Math.abs( T45BM[i] ) == 2 ) Eterm = E2;
            Sb += T45BL[i] * Eterm * sind( rev( T45BD[i] * D + T45BM[i] * M + T45BMP[i] * MP + T45BF[i] * F ) );
        }
        // Additional additive terms
        Sl = Sl + 3958 * sind( rev( A1 ) ) + 1962 * sind( rev( LP - F ) ) + 318 * sind( rev( A2 ) );
        Sb = Sb - 2235 * sind( rev( LP ) ) + 382 * sind( rev( A3 ) ) + 175 * sind( rev( A1 - F ) ) +
           175 * sind( rev( A1 + F ) ) + 127 * sind( rev( LP - MP ) ) - 115 * sind( rev( LP + MP ) );
        // geocentric longitude, latitude and distance
        var mglong = rev( LP + Sl / 1000000.0 );
        var mglat = rev( Sb / 1000000.0 );
        if ( mglat > 180.0 ) mglat = mglat - 360;
        var mr = Math.round( 385000.56 + Sr / 1000.0 );
        // Obliquity of Ecliptic
        var obl = 23.4393 - 3.563E-9 * ( jd - 2451543.5 );
        // RA and dec
        var ra = rev( atan2d( sind( mglong ) * cosd( obl ) - tand( mglat ) * sind( obl ),
                          cosd( mglong ) ) ) / 15.0;
        var dec = rev( asind( sind( mglat ) * cosd( obl ) + cosd( mglat ) * sind( obl ) * sind( mglong ) ) );
        if ( dec > 180.0 ) dec = dec - 360;
        return new Array( ra, dec, mr );
    }

    function MoonRise( year, month, day, TZ, latitude, longitude ) {
        // returns an array containing rise and set times or one of the
        // following codes.
        // -1 rise or set event not found and moon was down at 00:00
        // -2 rise or set event not found and moon was up   at 00:00
        // WARNING code changes on 6/7 May 2003 these are now local times
        // NOT UTC and rise/set not found codes changed.
        var hours = 0;
        var riseset = new Array();
        // elh is the elevation at the hour elhdone is true if elh calculated
        var elh = new Array();
        var elhdone = new Array();
        for ( var i = 0; i <= 24; i++ ) { elhdone[i] = false; }
        // Compute the moon elevation at start and end of day
        // store elevation at the hours in an array elh to save search time
        var rad = MoonPos( year, month, day, hours - TZ );
        var altaz = radtoaa( rad[0], rad[1], year, month, day, hours - TZ, latitude, longitude );
        elh[0] = altaz[0]; elhdone[0] = true;
        // set the return code to allow for always up or never rises
        if ( elh[0] > 0.0 ) {
            riseset = new Array( -2, -2 );
        } else {
            riseset = new Array( -1, -1 );
        }
        hours = 24;
        rad = MoonPos( year, month, day, hours - TZ );
        altaz = radtoaa( rad[0], rad[1], year, month, day, hours - TZ, latitude, longitude );
        elh[24] = altaz[0]; elhdone[24] = true;
        // search for moonrise and set
        for ( var rise = 0; rise < 2; rise++ ) {
            var found = false;
            var hfirst = 0;
            var hlast = 24;
            // Try a binary chop on the hours to speed the search
            while ( Math.ceil(( hlast - hfirst ) / 2 ) > 1 ) {
                hmid = hfirst + Math.round(( hlast - hfirst ) / 2 );
                if ( !elhdone[hmid] ) {
                    hours = hmid;
                    rad = MoonPos( year, month, day, hours - TZ );
                    altaz = radtoaa( rad[0], rad[1], year, month, day, hours - TZ, latitude, longitude );
                    elh[hmid] = altaz[0]; elhdone[hmid] = true;
                }
                if ( ( ( rise == 0 ) && ( elh[hfirst] <= 0.0 ) && ( elh[hmid] >= 0.0 ) ) ||
                    ( ( rise == 1 ) && ( elh[hfirst] >= 0.0 ) && ( elh[hmid] <= 0.0 ) ) ) {
                    hlast = hmid; found = true; continue;
                }
                if ( ( ( rise == 0 ) && ( elh[hmid] <= 0.0 ) && ( elh[hlast] >= 0.0 ) ) ||
                    ( ( rise == 1 ) && ( elh[hmid] >= 0.0 ) && ( elh[hlast] <= 0.0 ) ) ) {
                    hfirst = hmid; found = true; continue;
                }
                break;
            }
            // If the binary chop did not find a 1 hour interval
            if ( ( hlast - hfirst ) > 1 ) {
                for ( var i = hfirst; i < hlast; i++ ) {
                    found = false;
                    if ( !elhdone[i + 1] ) {
                        hours = i + 1;
                        rad = MoonPos( year, month, day, hours - TZ );
                        altaz = radtoaa( rad[0], rad[1], year, month, day, hours - TZ, latitude, longitude );
                        elh[hours] = altaz[0]; elhdone[hours] = true;
                    }
                    if ( ( ( rise == 0 ) && ( elh[i] <= 0.0 ) && ( elh[i + 1] >= 0.0 ) ) ||
                        ( ( rise == 1 ) && ( elh[i] >= 0.0 ) && ( elh[i + 1] <= 0.0 ) ) ) {
                        hfirst = i; hlast = i + 1; found = true; break;
                    }
                }
            }
            // simple linear interpolation for the minutes
            if ( found ) {
                var elfirst = elh[hfirst]; var ellast = elh[hlast];
                hours = hfirst + 0.5;
                rad = MoonPos( year, month, day, hours - TZ );
                altaz = radtoaa( rad[0], rad[1], year, month, day, hours - TZ, latitude, longitude );
                // alert("day ="+day+" hour ="+hours+" altaz="+altaz[0]+" "+altaz[1]);
                if ( ( rise == 0 ) && ( altaz[0] <= 0.0 ) ) { hfirst = hours; elfirst = altaz[0]; }
                if ( ( rise == 0 ) && ( altaz[0] > 0.0 ) ) { hlast = hours; ellast = altaz[0]; }
                if ( ( rise == 1 ) && ( altaz[0] <= 0.0 ) ) { hlast = hours; ellast = altaz[0]; }
                if ( ( rise == 1 ) && ( altaz[0] > 0.0 ) ) { hfirst = hours; elfirst = altaz[0]; }
                var eld = Math.abs( elfirst ) + Math.abs( ellast );
                riseset[rise] = hfirst + ( hlast - hfirst ) * Math.abs( elfirst ) / eld;
            }
        } // End of rise/set loop
        return ( riseset );
    }

    function MoonPhase( year, month, day, hours ) {
        // the illuminated percentage from Meeus chapter 46
        var j = dayno( year, month, day, hours ) + 2451543.5;
        var T = ( j - 2451545.0 ) / 36525;
        var T2 = T * T;
        var T3 = T2 * T;
        var T4 = T3 * T;
        // Moons mean elongation Meeus first edition
        // var D=297.8502042+445267.1115168*T-0.0016300*T2+T3/545868.0-T4/113065000.0;
        // Moons mean elongation Meeus second edition
        var D = 297.8501921 + 445267.1114034 * T - 0.0018819 * T2 + T3 / 545868.0 - T4 / 113065000.0;
        // Moons mean anomaly M' Meeus first edition
        // var MP=134.9634114+477198.8676313*T+0.0089970*T2+T3/69699.0-T4/14712000.0;
        // Moons mean anomaly M' Meeus second edition
        var MP = 134.9633964 + 477198.8675055 * T + 0.0087414 * T2 + T3 / 69699.0 - T4 / 14712000.0;
        // Suns mean anomaly
        var M = 357.5291092 + 35999.0502909 * T - 0.0001536 * T2 + T3 / 24490000.0;
        // phase angle
        var pa = 180.0 - D - 6.289 * sind( MP ) + 2.1 * sind( M ) - 1.274 * sind( 2 * D - MP )
               - 0.658 * sind( 2 * D ) - 0.214 * sind( 2 * MP ) - 0.11 * sind( D );
        return ( rev( pa ) );
    }

    function MoonQuarters( year, month, day ) {
        // returns an array of Julian Ephemeris Days (JDE) for 
        // new moon, first quarter, full moon and last quarter
        // Meeus first edition chapter 47 with only the most larger additional corrections
        // Meeus code calculate Terrestrial Dynamic Time
        // TDT = UTC + (number of leap seconds) + 32.184
        // At the end of June 2012 the 25th leap second was added
        // 
        var quarters = new Array();
        // k is an integer for new moon incremented by 0.25 for first quarter 0.5 for new etc.
        var k = Math.floor(( year + ( ( month - 1 ) + day / 30 ) / 12 - 2000 ) * 12.3685 );
        // Time in Julian centuries since 2000.0
        var T = k / 1236.85;
        // Sun's mean anomaly
        var M = rev( 2.5534 + 29.10535669 * k - 0.0000218 * T * T );
        // Moon's mean anomaly (M' in Meeus)
        var MP = rev( 201.5643 + 385.81693528 * k + 0.0107438 * T * T + 0.00001239 * T * T * T - 0.00000011 * T * T * T );
        var E = 1 - 0.002516 * T - 0.0000074 * T * T;
        // Moons argument of latitude
        var F = rev( 160.7108 + 390.67050274 * k - 0.0016341 * T * T - 0.00000227 * T * T * T + 0.000000011 * T * T * T * T );
        // Longitude of ascending node of lunar orbit
        var Omega = rev( 124.7746 - 1.56375580 * k + 0.0020691 * T * T + 0.00000215 * T * T * T );
        // The full planetary arguments include 14 terms, only used the 7 most significant
        var A = new Array();
        A[1] = rev( 299.77 + 0.107408 * k - 0.009173 * T * T );
        A[2] = rev( 251.88 + 0.016321 * k );
        A[3] = rev( 251.83 + 26.651886 * k );
        A[4] = rev( 349.42 + 36.412478 * k );
        A[5] = rev( 84.88 + 18.206239 * k );
        A[6] = rev( 141.74 + 53.303771 * k );
        A[7] = rev( 207.14 + 2.453732 * k );

        // New moon
        var JDE0 = 2451550.09765 + 29.530588853 * k + 0.0001337 * T * T - 0.000000150 * T * T * T + 0.00000000073 * T * T * T * T;
        // Correct for TDT since 1 July 2012
        JDE0 = JDE0 - 57.184 / ( 24 * 60 * 60 );
        var JDE = JDE0 - 0.40720 * sind( MP ) + 0.17241 * E * sind( M ) + 0.01608 * sind( 2 * MP ) + 0.01039 * sind( 2 * F )
                     + 0.00739 * E * sind( MP - M ) - 0.00514 * E * sind( MP + M ) + 0.00208 * E * E * sind( 2 * M ) - 0.00111 * sind( MP - 2 * F )
                     - 0.00057 * sind( MP + 2 * F ) + 0.00056 * E * sind( 2 * MP + M ) - 0.00042 * sind( 3 * MP ) + 0.00042 * E * sind( M + 2 * F )
                     + 0.00038 * E * sind( M - 2 * F ) - 0.00024 * E * sind( 2 * MP - M ) - 0.00017 * sind( Omega ) - 0.00007 * sind( MP + 2 * M );

        quarters[0] = JDE + 0.000325 * sind( A[1] ) + 0.000165 * sind( A[2] ) + 0.000164 * sind( A[3] ) + 0.000126 * sind( A[4] )
                       + 0.000110 * sind( A[5] ) + 0.000062 * sind( A[6] ) + 0.000060 * sind( A[7] );

        // The following code needs tidying up with a loop and conditionals for each quarter
        // First Quarter k=k+0.25
        JDE = JDE0 + 29.530588853 * 0.25;
        M = rev( M + 29.10535669 * 0.25 );
        MP = rev( MP + 385.81693528 * 0.25 );
        F = rev( F + 390.67050274 * 0.25 );
        Omega = rev( Omega - 1.56375580 * 0.25 );
        A[1] = rev( A[1] + 0.107408 * 0.25 ); A[2] = rev( A[2] + 0.016321 * 0.25 ); A[3] = rev( A[3] + 26.651886 * 0.25 );
        A[4] = rev( A[4] + 36.412478 * 0.25 ); A[5] = rev( A[5] + 18.206239 * 0.25 ); A[6] = rev( A[6] + 53.303771 * 0.25 );
        A[7] = rev( A[7] + 2.453732 * 0.25 );

        JDE = JDE - 0.62801 * sind( MP ) + 0.17172 * E * sind( M ) - 0.01183 * E * sind( MP + M ) + 0.00862 * sind( 2 * MP )
               + 0.00804 * sind( 2 * F ) + 0.00454 * E * sind( MP - M ) + 0.00204 * E * E * sind( 2 * M ) - 0.00180 * sind( MP - 2 * F )
               - 0.00070 * sind( MP + 2 * F ) - 0.00040 * sind( 3 * MP ) - 0.00034 * E * sind( 2 * MP - M ) + 0.00032 * E * sind( M + 2 * F )
               + 0.00032 * E * sind( M - 2 * F ) - 0.00028 * E * E * sind( MP + 2 * M ) + 0.00027 * E * sind( 2 * MP + M ) - 0.00017 * sind( Omega );
        // Next term is w add for first quarter & subtract for second
        JDE = JDE + ( 0.00306 - 0.00038 * E * cosd( M ) + 0.00026 * cosd( MP ) - 0.00002 * cosd( MP - M ) + 0.00002 * cosd( MP + M ) + 0.00002 * cosd( 2 * F ) );

        quarters[1] = JDE + 0.000325 * sind( A[1] ) + 0.000165 * sind( A[2] ) + 0.000164 * sind( A[3] ) + 0.000126 * sind( A[4] )
                       + 0.000110 * sind( A[5] ) + 0.000062 * sind( A[6] ) + 0.000060 * sind( A[7] );

        // Full moon k=k+0.5 
        JDE = JDE0 + 29.530588853 * 0.5;
        // Already added 0.25 for first quarter
        M = rev( M + 29.10535669 * 0.25 );
        MP = rev( MP + 385.81693528 * 0.25 );
        F = rev( F + 390.67050274 * 0.25 );
        Omega = rev( Omega - 1.56375580 * 0.25 );
        A[1] = rev( A[1] + 0.107408 * 0.25 ); A[2] = rev( A[2] + 0.016321 * 0.25 ); A[3] = rev( A[3] + 26.651886 * 0.25 );
        A[4] = rev( A[4] + 36.412478 * 0.25 ); A[5] = rev( A[5] + 18.206239 * 0.25 ); A[6] = rev( A[6] + 53.303771 * 0.25 );
        A[7] = rev( A[7] + 2.453732 * 0.25 );

        JDE = JDE - 0.40614 * sind( MP ) + 0.17302 * E * sind( M ) + 0.01614 * sind( 2 * MP ) + 0.01043 * sind( 2 * F )
               + 0.00734 * E * sind( MP - M ) - 0.00515 * E * sind( MP + M ) + 0.00209 * E * E * sind( 2 * M ) - 0.00111 * sind( MP - 2 * F )
               - 0.00057 * sind( MP + 2 * F ) + 0.00056 * E * sind( 2 * MP + M ) - 0.00042 * sind( 3 * MP ) + 0.00042 * E * sind( M + 2 * F )
               + 0.00038 * E * sind( M - 2 * F ) - 0.00024 * E * sind( 2 * MP - M ) - 0.00017 * sind( Omega ) - 0.00007 * sind( MP + 2 * M );

        quarters[2] = JDE + 0.000325 * sind( A[1] ) + 0.000165 * sind( A[2] ) + 0.000164 * sind( A[3] ) + 0.000126 * sind( A[4] )
                       + 0.000110 * sind( A[5] ) + 0.000062 * sind( A[6] ) + 0.000060 * sind( A[7] );

        // Last Quarter k=k+0.75
        JDE = JDE0 + 29.530588853 * 0.75;
        // Already added 0.5 for full moon
        M = rev( M + 29.10535669 * 0.25 );
        MP = rev( MP + 385.81693528 * 0.25 );
        F = rev( F + 390.67050274 * 0.25 );
        Omega = rev( Omega - 1.56375580 * 0.25 );
        A[1] = rev( A[1] + 0.107408 * 0.25 ); A[2] = rev( A[2] + 0.016321 * 0.25 ); A[3] = rev( A[3] + 26.651886 * 0.25 );
        A[4] = rev( A[4] + 36.412478 * 0.25 ); A[5] = rev( A[5] + 18.206239 * 0.25 ); A[6] = rev( A[6] + 53.303771 * 0.25 );
        A[7] = rev( A[7] + 2.453732 * 0.25 );

        JDE = JDE - 0.62801 * sind( MP ) + 0.17172 * E * sind( M ) - 0.01183 * E * sind( MP + M ) + 0.00862 * sind( 2 * MP )
               + 0.00804 * sind( 2 * F ) + 0.00454 * E * sind( MP - M ) + 0.00204 * E * E * sind( 2 * M ) - 0.00180 * sind( MP - 2 * F )
               - 0.00070 * sind( MP + 2 * F ) - 0.00040 * sind( 3 * MP ) - 0.00034 * E * sind( 2 * MP - M ) + 0.00032 * E * sind( M + 2 * F )
               + 0.00032 * E * sind( M - 2 * F ) - 0.00028 * E * E * sind( MP + 2 * M ) + 0.00027 * E * sind( 2 * MP + M ) - 0.00017 * sind( Omega );
        // Next term is w add for first quarter & subtract for second
        JDE = JDE - ( 0.00306 - 0.00038 * E * cosd( M ) + 0.00026 * cosd( MP ) - 0.00002 * cosd( MP - M ) + 0.00002 * cosd( MP + M ) + 0.00002 * cosd( 2 * F ) );

        quarters[3] = JDE + 0.000325 * sind( A[1] ) + 0.000165 * sind( A[2] ) + 0.000164 * sind( A[3] ) + 0.000126 * sind( A[4] )
                       + 0.000110 * sind( A[5] ) + 0.000062 * sind( A[6] ) + 0.000060 * sind( A[7] );

        return quarters;
    }

    jSignage.Astronomy.moonEquation = function ( year, month, day, latitude, longitude, timezone, tune ) {
        var midnightLocalTime = jSignage.timeLocal( new Date( year, month-1, day ), timezone || null );
        var TZ = - ( midnightLocalTime - Date.UTC( year, month-1, day ) ) / 3600000;
        var mrs = MoonRise( year, month, day, TZ, latitude, longitude );
        var tRise, tSet;    // In seconds since midnight local time
        if ( ( mrs[0] < 0 ) && ( mrs[1] < 0 ) ) {
            if ( mrs[0] == -1 ) {
                // Moon is down all day
                tSet = 0;
                tRise = 24;
            } else {
                // Moon is up all day
                tRise = 0;
                tSet = 24;
            }
        } else {
            tRise = mrs[0];
            tSet = mrs[1];
        }
        var phase = MoonPhase( year, month, day, 12 - TZ );
        var unicode_moon;
        if (phase <= 22.5 || phase > 337.5)
            unicode_moon = "\uD83C\uDF11";
        else if ( phase <= 67.5 )
            unicode_moon = "\uD83C\uDF12";
        else if (phase <= 112.5)
            unicode_moon = "\uD83C\uDF13";
        else if (phase <= 157.5)
            unicode_moon = "\uD83C\uDF14";
        else if (phase <= 202.5)
            unicode_moon = "\uD83C\uDF15";
        else if (phase <= 247.5)
            unicode_moon = "\uD83C\uDF16";
        else if (phase <= 292.5)
            unicode_moon = "\uD83C\uDF17";
        else
            unicode_moon = "\uD83C\uDF18";

        var JDE = MoonQuarters( year, month, day );
        var newMoon = jdtocd( JDE[0] );
        var firstQuarter = jdtocd( JDE[1] );
        var fullMoon = jdtocd( JDE[2] );
        var lastQuarter = jdtocd( JDE[3] );
        return {
            moonrise: jSignage.localTime( midnightLocalTime + tRise * 3600000 + ( tune && tune.monrise ? tune.moonrise * 60000 : 0 ), timezone || null ),
            moonset: jSignage.localTime( midnightLocalTime + tSet * 3600000 + ( tune && tune.moonset ? tune.moonset * 60000 : 0 ), timezone || null ),
            phase: Math.round( phase ),
            symbol: unicode_moon,
            illumination: Math.round( 100 * ( 1 + cosd(phase) ) / 2 ),
            newMoon: jSignage.localTime( Date.UTC( newMoon[0], newMoon[1] - 1, newMoon[2], newMoon[4], newMoon[5], newMoon[6] ), timezone || null ),
            firstQuarter: jSignage.localTime( Date.UTC( firstQuarter[0], firstQuarter[1] - 1, firstQuarter[2], firstQuarter[4], firstQuarter[5], firstQuarter[6] ), timezone || null ),
            fullMoon: jSignage.localTime( Date.UTC( fullMoon[0], fullMoon[1] - 1, fullMoon[2], fullMoon[4], fullMoon[5], fullMoon[6] ), timezone || null ),
            lastQuarter: jSignage.localTime( Date.UTC( lastQuarter[0], lastQuarter[1] - 1, lastQuarter[2], lastQuarter[4], lastQuarter[5], lastQuarter[6] ), timezone || null )
        };
    };

    jSignage.Astronomy.moonPath = function ( phase, latitude, invert, x, y, width, height ) {
        var mag, sweep;
        if ( phase <= 90 ) {
            sweep = [1, 0];
            mag = 1 - phase / 90;
        } else if ( phase <= 180 ) {
            sweep = [0, 0];
            mag = ( phase - 90 ) / 90;
        } else if ( phase <= 270 ) {
            sweep = [1, 1];
            mag = 1 - ( phase - 180 ) / 90;
        } else {
            sweep = [0, 1];
            mag = ( phase - 270 ) / 90;
        }
        if ( latitude < 0 ) {
            sweep[0] = 1 - sweep[0];
            sweep[1] = 1 - sweep[1];
        }
        if ( invert )
            sweep[1] = 1 - sweep[1];
        var r = Math.min( width, height ) / 2;
        var d = new jSignage.pathData;
        d.moveTo( x+ width / 2, y + height / 2 - r );
        d.arcTo( mag * r, r, 0, 1, 1 - sweep[0], x + width / 2, y + height / 2 + r );
        d.arcTo( r, r, 0, 1, sweep[1], x + width / 2, y + height / 2 - r );
        d.close();
        return d;
    };

} )();
