#!/bin/bash
#
# License: Copyright 2015 SpinetiX. This file is licensed
#          under the terms of the GNU General Public License version 2.
#          This program is licensed "as is" without any warranty of any
#          kind, whether express or implied.
#
### BEGIN INIT INFO
# Required-Start:
# Required-Stop:
# Should-Start:
# Should-Stop:
# Default-Start: 2 3 4 5
# Default-Stop:
# Short-Description: Does pending media-mount actions
# Description: Does pending media-mount actions
### END INIT INFO

# Init script information
NAME=media-mount.sh
DESC=""

# Load general helpers and config
. /etc/media-mount/media-mount-functions
load_sysconfig

# The hook directory. All executables in the hook dir will be
# run with the "init" argument before processing pending actions
# and with the "ready" argument after no pending actions remain
HOOKDIR=/etc/media-mount/startup.d

# The executable that handles the device events
DEVEXE=/etc/udev/scripts/media-mount

# Cleans all possible leftover media-mount entries in fstab.
# Must be called with the lock on fstab held.
clean_fstab() {
    local RET

    # clean fstab
    if ! mktmpfstab; then
        echo "ERROR: cannot create temporary fstab while cleaning fstab"
        return 1
    fi
    # remove all entries which are mounts under $MEDIADIR or bind mounts
    # with a source under $MEDIADIR
    grep -v "^[[:space:]]*\($MEDIADIR/\|[^#[:space:]][^[:space:]]*[[:space:]]\+$MEDIADIR/\)" /etc/fstab > "$TMPFSTAB"
    RET=$?
    if [ $RET -eq 0 ]; then
        savetmpfstab
        RET=$?
    elif [ $RET -eq 1 ]; then
        RET=0 # Nothing to clean in fstab
    fi

    cleantmpfstab

    if [ $RET -ne 0 ]; then
        echo "ERROR: failed cleaning fstab (RET = $RET)"
    fi

    return $RET
}

# Runs all the executable files in the hook directory passing the arguments;
# files are processed in alphabetical order.
run_hooks() {
    local f

    for f in "$HOOKDIR"/*; do
        [ -x "$f" ] && "$f" "$@"
    done
}

# Waits for a pending file to exist, up to the max delay which
# must be given as first argument (seconds); returns non-zero if
# pending file does not appear; zero if it does.
# This can be called without holding the state lock
wait_for_pending() {
    local n="$1"

    while [ ! -f "$PENDINGFILE" ]; do
        ((--n >= 0)) || return 1
        sleep 1
    done
    return 0
}

# Processes all pending actions, needs to be called with the state lock held
# The lock will be temporarily released during lengthy operations
do_pending_actions() {
    local oldifs="$IFS"
    local action
    local devname
    local devpath

    [ -f "$PENDINGFILE" ] || return 0

    while [ -f "$PENDINGFILE" ]; do
        if ! mv "$PENDINGFILE" "$PENDINGFILE".act; then
            echo "ERROR: failed to move pending actions file" >&2
            break
        fi
        # Process the previously pending actions without the lock
        # as the device handler invocation below may take long and
        # having the lock would prevent adding new device actions
        state_unlock
        IFS="${IFS}="
        while read key value; do
            case "$key" in
                --START--)
                    action=
                    devname=
                    devpath=
                    ;;
                --END--)
                    logger -p daemon.info \
                        -t "media-mount.sh [$$]" \
                        "Processing pending '$action' '$devname' at '$devpath'"
                    SUBSYSTEM="block" \
                        ACTION="$action" \
                        DEVNAME="$devname" \
                        DEVPATH="$devpath" \
                        MEDIAMOUNTREADY="yes" \
                        "$DEVEXE"
                    ;;
                ACTION)
                    action="$value"
                    ;;
                DEVNAME)
                    devname="$value"
                    ;;
                DEVPATH)
                    devpath="$value"
                    ;;
                *)
                    echo "WARNING: unrecognized key '$key' in pending actions" >&2
                    ;;
            esac
        done < "$PENDINGFILE".act
        IFS="$oldifs"
        rm -f "$PENDINGFILE".act
        # allow a few seconds for an action to be enqueued before we give up
        # as block device discovery may be a bit slow
        wait_for_pending 5
        # Re-obtain the lock to examine if any new actions have been
        # queued as pending
        state_lock || break
    done
}

set_ready() {
    echo -n > "$READYFILE"
    run_hooks ready
}

do_init() {
    # Make sure state dir exists
    [ -d "$STATEDIR" ] || mkdir -p "$STATEDIR"

    if ! state_lock; then
        touch "$ERRORFILE"
        return 1
    fi
    if fstab_lock; then
        rmdir "$MEDIADIR"/* 2>/dev/null # clean mount points
        clean_fstab || touch "$ERRORFILE"
        run_hooks init
        fstab_unlock
    else
        touch "$ERRORFILE"
    fi
    state_unlock

}

do_start() {
    local usbdevs

    echo "starting..."

    usbdevs=
    [ -f /sys/kernel/debug/usb/devices ] && usbdevs=/sys/kernel/debug/usb/devices || \
        { [ -f /proc/bus/usb/devices ] && usbdevs=/proc/bus/usb/devices; }

    # USB mass storage devices take long to be initialized,
    # so we should wait, but no need to delay stuff if there
    # are no mass storage USB devices connected
    if [ -n "$usbdevs" ] && grep -q '^I: .* Cls=0*8[^0-9]' "$usbdevs"; then
        # It takes a few seconds to load the driver, then the
        # driver waits about 5 seconds for device to settle
        wait_for_pending 15
    fi

    # Process all pending actions with state lock held
    if ! state_lock ; then
        # Cannot process stuff but at least let new actions run directly
        echo "ERROR: failed to process queued actions, aborting" >&2
        set_ready
        return
    fi

    do_pending_actions

    # From now on everything is setup for media-mount to do its own work
    set_ready

    state_unlock

    echo "done"
}

#
# Main
#

case "$1" in
    start)
        if [ -f "$READYFILE" ]; then
            echo "media-mount boot time processing already done"
            exit 1
        fi

        echo -n "Initializing media-mount..."
        do_init 2>&1 | \
            logger -s -p daemon.info -t "$NAME [$$]"
        echo ""

        echo -n "Processing pending media-mount actions in background..."

        do_start < /dev/null 2>&1 | \
            logger -p daemon.info -t "$NAME [$$]" &

        echo ""

        ;;
    stop)
        ;;
esac
