<?php
class WifiConfig extends CFormModel
{
    static $path = "/var/lib/iwd/";

    public $ssid;
    public $method;

    public $hidden;
    public $username;
    public $password;
    public $pem;

    public $file;
    public $fileBase64;

    /**
     * @return array validation rules for model attributes.
     */
    public function rules()
    {
        return array(
            array('ssid', 'safe'),
            array('hidden', 'safe'),
            array('username', 'safe'),
            array('password', 'safe'),
            array('pem', 'safe'),
            array('ssid', 'safe'),
            array('file', 'safe'),
            array('fileBase64', 'safe'),
            array('method', 'in', 'range' => array("open", "psk", "8021x")),
        );
    }

    public function attributeLabels()
    {
        return array();
    }
    public function createFile()
    {
        $file = "# Automatically generate by Control Center\n";
        $file = "[Settings]\n";
        $file .= "Hidden=" . ($this->hidden ? "true" : "false") . "\n";
        if ($this->method === "psk") {
            $file .= "[Security]\n";
            $file .= "Passphrase=" . $this->password . "\n";
        } elseif ($this->method === "8021x") {
            $file .= "[Security]\n";
            $file .= "EAP-Method=PEAP\n";
            $file .= "EAP-Identity=" . $this->username . "\n";
            $file .= "EAP-PEAP-Phase2-Method=MSCHAPV2\n";
            $file .= "EAP-PEAP-Phase2-Identity=" . $this->username . "\n";
            $file .= "EAP-PEAP-Phase2-Password=" . $this->password . "\n";
            if ($this->pem) {
                $file .= "[@pem@cacert]\n";
                $file .= $this->pem . "\n";
            }
        }
        return $file;
    }

    public function save()
    {
        if ($this->fileBase64) {
            $this->file = base64_decode($this->fileBase64);
        }
        if (!$this->file) {
            if ($this->method === "psk" && !$this->password) {
                return $this->addError(
                    "file",
                    "A configuration file is required"
                );
            }
            if (
                $this->method === "8021x" &&
                (!$this->password || !$this->username)
            ) {
                return $this->addError(
                    "file",
                    "A configuration file is required"
                );
            }
            $this->file = $this->createFile();
        }
        $name = WifiConfig::getFileName($this->ssid, $this->method);
        Tools::save_file( WifiConfig::$path . $name, $this->file );
        if (
            !Tools::setSafeModeData(
                "iwd/" . $name,
                WifiConfig::$path . $name,
                true
            )
        ) {
            $this->addError(
                'file',
                'Failed saving wifi configuration for recovery console, please try again.'
            );
            return false;
        }
    }
    public function load()
    {
        $this->file = WifiConfig::loadFile($this->ssid, $this->method);
    }

    public static function getFileName($ssid, $method)
    {
        if (preg_match("/[^\w\- ]/", $ssid)) {
            $encodedSsid = "=";
            for ($pos = 0; $pos < strlen($ssid); $pos++) {
                $byte = substr($ssid, $pos);
                $encodedSsid .= dechex(ord($byte));
            }
            $ssid = $encodedSsid;
        }
        return $ssid . "." . $method;
    }
    public static function checkFileName($name)
    {
        return substr($name, -4) === "open" ||
            substr($name, -3) === "psk" ||
            substr($name, -5) === "8021x";
    }
    public static function decodeFileName($name)
    {
        $data = explode(".", $name);
        if (substr($data[0], 0, 1) === "=") {
            $ssid = "";
            for ($pos = 1; $pos < strlen($data[0]); $pos += 2) {
                $byte = substr($data[0], $pos, 2);
                $ssid .= chr(hexdec($byte));
            }
            $data[0] = $ssid;
        }
        return $data;
    }

    public static function saveFile($ssid, $method, $file)
    {
        $name = WifiConfig::getFileName($ssid, $method);
        Tools::save_file( WifiConfig::$path . $name, $file );        
        return $name;
    }
    public static function loadFile($ssid, $method)
    {
        $name = WifiConfig::getFileName($ssid, $method);
        if (!file_exists(WifiConfig::$path . $name)) {
            return null;
        }

        return file_get_contents(WifiConfig::$path . $name);
    }
    public static function clearAll()
    {
        $dir = scandir(WifiConfig::$path);

        foreach ($dir as $file) {
            if (!WifiConfig::checkFileName($file)) {
                continue;
            }
            unlink(WifiConfig::$path . $file);
        }

        Tools::setSafeModeData("iwd", null);
    }
    public static function listAll()
    {
        $list = array();
        $dir = scandir(WifiConfig::$path);

        foreach ($dir as $file) {
            if (!WifiConfig::checkFileName($file)) {
                continue;
            }
            $data = WifiConfig::decodeFileName($file);
            $conf = new WifiConfig();
            $conf->ssid = $data[0];
            $conf->method = $data[1];
            $list[] = $conf;
        }
        return $list;
    }
}
