<?php

if (!extension_loaded('json')) {
    echo "ERROR: json extension is not loaded\n";
    trigger_error("json extension is not loaded", E_USER_ERROR);
}

require_once './protected/utils/init.php';

setTimeZone();

require_once './protected/utils/JSONRPCServer.php';
require_once './protected/utils/WebStorageAPI.php';
require_once './protected/utils/PullMode.php';

require_once './protected/utils/Updater.php'; // for the update process
require_once './protected/utils/MaintenanceJobs.php'; // for the reboot and other processes

require_once './protected/utils/MainInfo.php'; // for the info

function check($options, $name)
{
    if (isset($options['all']) && $options['all'] !== false) {
        return true;
    }
    return isset($options[$name]) && $options[$name] !== false;
}
class RPC
{
    public $needreload = false;
    public $callId = null;
    /**
     * @var array Additinal users autorized for a specific method
     */
    public $methods_groups = array(
        'get_info' => array('monitoring'),
        'wifi_get_info' => array('monitoring'),
        'wifi_scan' => array('monitoring'),
        'get_config' => array('monitoring'),
        'webstorage_list' => array('content'),
        'webstorage_get' => array('content'),
        'webstorage_set' => array('content'),
        'webstorage_remove' => array('content'),
        'webstorage_cmpxchg' => array('content'),
    );
    /**
     * @var array Methods that are safe to run while the firmware is being
     * updated.
     */
    public $safe_methods = array(
        'firmware_update_status',
        'get_config',
        'get_info',
        'get_license_device_info',
        'set_rpc_poll_fallback_times',
        'wifi_connect',
        'wifi_disconnect',
        'wifi_get_info',
        'wifi_scan'
    );

    private function isLoggedUser()
    {
        if (!isset($_COOKIE["uisess"])) {
            return false;
        }
        $id = $_COOKIE["uisess"];
        $autentication_path = ini_get('session.save_path');
        return file_exists($autentication_path . "/auth_" . $id);
    }
    private function isARYAEnabled()
    {
        if (!$this->checkLocalhostCall()) {
            return false;
        }
        return getAryaStatus() === "yes";
    }
    private function checkWebstorage()
    {
        $brandingInfo = new BrandingInfo();
        if ($brandingInfo->hasRight('netAPI')) {
            return true;
        }
        if ($this->isLoggedUser()) {
            return true;
        }
        if ($this->checkLocalhostCall()) {
            return true;
        }
        throw new Exception("Not supported by this model");
    }
    private function checkPullMode($type)
    {
        $brandingInfo = new BrandingInfo();
        if ($brandingInfo->hasRight('pullMode')) {
            return true;
        }
        if ($type == "upload") {
            //everybody can upload logs or other info
            return true;
        }
        if ($this->isARYAEnabled()) {
            return true;
        }
        throw new Exception("Not supported by this model");
    }
    private function checkLocalhostCall()
    {
        $localhost = array('127.0.0.1', 'localhost', '::1');
        return in_array($_SERVER['REMOTE_ADDR'], $localhost);
    }

    /*
	restart
	
	void restart();
	
	Immediately initiates a clean restart of the device. If a restart has already been initiated the call is ignored.
*/
    public function restart()
    {
        $result = array();
        $delay =
            $this->checkLocalhostCall() && $this->callId !== null ? 1 : false;
        $ret = MaintenanceJobs::shutdown(
            "user initiated shutdown : rpc restart() call",
            false,
            $delay
        );

        $result['success'] = $ret === true;
        $result['reason'] = $ret === true ? '' : $ret;

        return $result;
    }

    public function shutdown()
    {
        $result = array();
        $delay =
            $this->checkLocalhostCall() && $this->callId !== null ? 1 : false;
        $ret = MaintenanceJobs::shutdown(
            "user initiated shutdown with power-off : rpc shutdown() call",
            false,
            $delay,
            true
        );

        $result['success'] = $ret === true;
        $result['reason'] = $ret === true ? '' : $ret;

        return $result;
    }

    public function get_info($options = array())
    {
        $serial = "[undefined]";
        $model = "[undefined]";
        $hardware = "[undefined]";
        $firmware = "[undefined]";
        $build = "[undefined]";
        if (!is_array($options)) {
            $options = array();
        }
        if (check($options, 'all')) {
            if (!isset($options['temperature'])) {
                $options['temperature'] = $options['all'];
            }
            if (!isset($options['stats'])) {
                $options['stats'] = $options['all'];
            }
            if (!isset($options['errors'])) {
                $options['errors'] = $options['all'];
            }
        }
        $result = array();
        $info = getInfoSerial($serial, $model, $hardware);
        $result['serial'] = $info['altSerial']['service'];
        $result["altSerial"] = $info['altSerial'];

        //$result['version'] = $hardware;

        // name
        $conf = new DOMDocument();
        if ($conf->load('/etc/raperca/spxconfig.xml')) {
            $xp = new DOMXPath($conf);
            $xp->registerNamespace(
                'd',
                'http://www.spinetix.com/namespace/1.0/spxconf'
            );
            $nl = $xp->query("//d:player/d:reporting/@deviceName");
            if ($nl->length) {
                $result['name'] = $nl->item(0)->value;
            }
        }
        if (!isset($result['name']) || empty($result['name'])) {
            $result['name'] = $result['serial'];
        }

        // model
        $brandingInfo = new BrandingInfo();
        $result['model'] = $brandingInfo->product;
        // running mode
        $result['mode'] = getInfoSafeMode() ? 'safe' : 'normal';

        // firmware
        $result['firmware'] = array();
        // firmware - version
        getInfoFirmware($firmware, $build);
        $result['firmware']['version'] = $firmware . "-" . $build;
        // firmware - status
        $result['firmware']['status'] = getInfoFirmwareCorrupted()
            ? 'corrupted'
            : 'normal';

        if ($model !== 'ikebana' && check($options, 'license')) {
            $result['license'] = $brandingInfo->getLicenseData();
        }

        $result['uptime'] = getInfoUpTime();
        $result['bootid'] = MaintenanceJobs::bootID();

        $parser = new ParseLogs();

        if ($model != "Bonsai" && check($options, 'temperature')) {
            // temperature
            $t = getInfoTemperature();
            $result['temperature']['C'] = $t['temp'];
            $result['temperature']['F'] = (9 / 5) * $t['temp'] + 32;
            if (is_nan($t['temp'])) {
                $result['temperature']['C'] = '[not found]';
                $result['temperature']['F'] = '[not found]';
            }
            if ($t['crit']) {
                $result['temperature']['alarm'] = 'critical';
            } elseif ($t['max']) {
                $result['temperature']['alarm'] = 'maximum';
            } else {
                $result['temperature']['alarm'] = 'none';
            }

            if ($options['temperature'] !== true) {
                $result['temperature']['history'] = $parser->readTemperature(
                    $options['temperature']
                );
            }
        }

        if ($model !== 'ikebana' && check($options, 'videoConnectors')) {
            // display
            $result['videoConnectors'] = array();
            $screeninfo = new ScreenInfoDRM();
            $screeninfo->load($brandingInfo->videoOutputs);
            foreach ($screeninfo->screens as $screen) {
                if ($screen['displayName'] === false) {
                    continue;
                }

                $sc = array(
                    'name' => $screen['displayName'],
                    'id' => $screen['name'],
                    'type' => $screen['type'],
                    'status' => $screen['status'],
                    'enabled' => $screen['enabled'],
                );
                if (isset($screen['modes'])) {
                    $sc["modes"] = $screen['modes'];
                }
                $result['videoConnectors'][] = $sc;
            }
        }
        if ($model === 'ikebana' && check($options, 'videoConnectors')) {
            // display
            $result['videoConnectors'] = array();
            $screeninfo = new ScreenInfoBasics();
            $screeninfo->load();
            if ($screeninfo->monitors) {
                $screen = $screeninfo->monitors[0];
                $modes = array();
                foreach (ScreenInfoBasics::getModes() as $mode) {
                    $modes["{$mode['resolution']}@{$mode['hz']}"] = true;
                }
                $sc = array(
                    'name' => isset($screen['name'])
                        ? $screen['name']
                        : $screen['type'],
                    'type' => $screen['type'],
                    'status' => isset($screen['powered'])
                        ? "connected"
                        : "disconnected",
                    'enabled' => true,
                    "modes" => array_keys($modes)
                );
                $result['videoConnectors'][] = $sc;
            }
        }
        if ($model !== 'ikebana' && check($options, 'audioConnectors')) {
            // audio
            $result['audioConnectors'] = ScreenInfoDRM::getAudioInfo(
                $brandingInfo->audioOutputs,
                false
            );
        }
        if (check($options, 'display')) {
            // display

            $screeninfo = new ScreenInfoBasics();
            $screeninfo->load();
            $result['display'] = array();
            foreach ($screeninfo->monitors as $screen) {
                if (isset($screen['powered'])) {
                    $screen['power'] = $screen['powered'];
                    unset($screen['powered']);
                } else {
                    $screen['power'] = 'gone';
                }
                $result['display'][] = $screen;
            }
        }

        if (check($options, 'storage')) {
            // storage
            $result['storage'] = array();
            $infoSystem = getDiskInfo("/");
            $infoSystem['type'] = 'system';
            $result['storage'][] = $infoSystem;

            $infoContent = getDiskInfo("/srv/raperca/content");
            $infoContent['type'] = 'content';
            $result['storage'][] = $infoContent;
        }
        if (check($options, 'status')) {
            $result['status'] = array(
                "config" => getWizardStatus(),
                "aryaStatus" => getAryaStatus(),
            );
        }

        if (check($options, 'stats')) {
            $result['stats'] = $parser->readStats($options['stats']);
        }
        if (check($options, 'errors')) {
            $result['errors'] = $parser->readErrors($options['errors']);
        }
        if (check($options, 'reason')) {
            $result['reason'] = ParseSyslog::getRebootReason();
        }
        if (check($options, 'storageHealth')) {
            $result['storageHealth'] = getStorageHealth();
        }
        if (check($options, 'network')) {
            $ip_addrs = getInfoIp($ip, $prefixLength, $ifaceType);
            $ipv6 = getInfoIpv6('eth0', true);

            $ipconfig = array(
                "hostname" => getInfoHostname(),
                "ipv4" => array(),
                "ipv6" => array(),
            );

            $addr = reset($ip_addrs);
            while ($addr) {
                $ipconfig['ipv4'][] = array(
                    "addr" => $addr[0],
                    "prefixLength" => $addr[1],
                    "type" => $addr[2],
                    "interface" => $addr[3],
                );
                $addr = next($ip_addrs);
            }
            foreach ($ipv6 as $ipv6addr) {
                $ipconfig['ipv6'][] = $ipv6addr;
            }
            $result['network'] = $ipconfig;
        }
        if (check($options, 'enrollement')) {
            $result['enrollement'] = MaintenanceJobs::queryEnrollment();
        }
        if (hasPowerMCU() && check($options, 'power')) {
            $powerInfo = getPowerInfo();
            $result['power'] = array(
                'source' => getPowerSource($powerInfo),
                'watt' => $powerInfo['power']
            );
        }
        return $result;
    }
    public function wifi_scan()
    {
        MaintenanceJobs::scanWifi();
        return array("success" => true);
    }

    public function wifi_connect($name = "")
    {
        if ($name === "") {
            throw new Exception("Name is required");
        }
        $data = MaintenanceJobs::scanDBUS();
        $network = null;
        $netpath = null;
        foreach ($data['Network'] as $key => $info) {
            if ($info['Name'] === $name) {
                $network = $info['proxy'];
                $netpath = $key;
                break;
            }
        }
        if (!$network) {
            throw new Exception("Network '" . $name . "' not found");
        }

        $retry = false;
        do {
            try {
                syslog(
                    LOG_DEBUG,
                    "RPC wifi_connect to network '" .
                        $name .
                        "' with path '" .
                        $netpath .
                        "'"
                );
                $network->Connect();
                $retry = false;
            } catch (Exception $e) {
                syslog(
                    LOG_DEBUG,
                    "RPC wifi_connect, connect exception: " . $e->getMessage()
                );
                if (
                    !$retry &&
                    strpos($e->getMessage(), "net.connman.iwd.NoAgent") !==
                        false
                ) {
                    // try again
                    sleep(1);
                    $retry = true;
                } else {
                    throw new Exception("Connection failed, check password");
                }
            }
        } while ($retry);

        $max_wait = 200;
        do {
            $data = MaintenanceJobs::scanDBUS();

            if (!isset($data['Station']['State'])) {
                throw new Exception(
                    "Failed to query connection state, missing station object"
                );
            }
            $state = $data['Station']['State'];
            if ($state == 'connected') {
                if (!isset($data['Station']['ConnectedNetwork'])) {
                    syslog(
                        LOG_DEBUG,
                        "RPC wifi_connect, connected but no ConnectedNetwork!"
                    );
                    throw new Exception(
                        "Failed to verify connection state, missing connected network object"
                    );
                }
                if ($data['Station']['ConnectedNetwork'] != $netpath) {
                    // associated (i.e. connected or connecting) to another network!
                    syslog(
                        LOG_DEBUG,
                        "RPC wifi_connect, associated to unexpected network with path '" .
                            $data['Station']['ConnectedNetwork'] .
                            "'"
                    );
                    throw new Exception("Connection failed, check password");
                }

                syslog(
                    LOG_DEBUG,
                    "RPC wifi_connect, connected to network '" . $name . "'"
                );
                break;
            } elseif ($state != 'connecting') {
                syslog(
                    LOG_DEBUG,
                    "RPC wifi_connect, connection to '" .
                        $name .
                        "' failed with state '" .
                        $state .
                        "'"
                );
                throw new Exception("Connection failed, check password");
            }

            if ($max_wait-- > 0) {
                syslog(
                    LOG_DEBUG,
                    "RPC wifi_connect, still connecting to '" .
                        $name .
                        "', waiting"
                );
                // polling period should be relatively short as iwd will automatically retry the
                // connection and transition back to 'connecting' after a connection failure
                usleep(50000);
            } else {
                syslog(
                    LOG_DEBUG,
                    "RPC wifi_connect, timeout connecting to '" . $name . "'"
                );
                throw new Exception("Connection timeout, check password");
            }
        } while (true);

        if (!isset($data['Network'][$netpath])) {
            throw new Exception(
                "Failed to query connection state, missing network object"
            );
        }
        $info = $data['Network'][$netpath];
        $result = array(
            'state' => $data['Station']['State'],
            'name' => $info['Name'],
            'type' => $info['Type'],
        );
        return $result;
    }

    public function wifi_disconnect()
    {
        $data = MaintenanceJobs::scanDBUS();
        if (!isset($data['Station'])) {
            throw new Exception("Station not found");
        }
        try {
            $data['Station']['proxy']->Disconnect();
        } catch (Exception $e) {
            throw new Exception("Disconnect failed");
        }
        return array("success" => true);
    }

    public function wifi_get_info($options = array())
    {
        return MaintenanceJobs::getWifiInfo();
    }

    /*
	firmware_update
	
	string handle firmware_update(object options);
	
	Starts the firmware update process in the background. 
	A handle is returned that should be used to poll for completion using the firmware_update_status() call.
	object options {
		"repo_id"	: [ string .. ],					// optional
		"repo_uri"	: [ { "id" : string, "uri" : string } ... ],	// optional
		"max_wait"	: int,							// optional
		"check_only"	: bool,						// optional
	};
	repo_id	
		an array specifying the repository to use for updating, currently only one can be specified; this argument is optional, 
		if not present the configured default is used. The recognized repository IDs are "base" and "local-usb".
	repo_uri	
		an array of pairs of repository IDs and URIs, used to override the URI of a repository without requiring a configuration change; 
		this argument is optional, if not present the configured default is used. The URIs must be percent encoded as specified in RFC 3986. 
		Only http, https and local file URI schemes are supported.
	max_wait	
		the maximum wait, in seconds, before the updates server is contacted, the actual wait is a random time between 0 and the maximum; 
		this argument is optional, if not present the default of 600 seconds is used. 
		The purpose of this argument is to reduce load on servers, a value of 0 disables the wait.
	check_only	
		if true only a check is performed against the updates server, no updates are downloaded nor applied; 
		defaults to false if not present.
	handle	
		an opaque string which should be passed to firmware_update_status() to poll for completion.
	
	Note: performing a check is resource intensive as all firmware components are evaluated, 
		  therefore this call should not be made frequently or the player performance of the HMP may be degraded.
*/
    public function firmware_update($options = array())
    {
        $brandingInfo = new BrandingInfo();
        $product = $brandingInfo->product;
        $email = $brandingInfo->email;

        $updater = new Updater();
        $updater->product = $product;
        $updater->email = $email;

        $source = 'default';
        $id = "base";
        if (isset($options['repo_id']) && isset($options['repo_id'][0])) {
            $id = $options['repo_id'][0];
        }
        if ($id == "local-usb") {
            $source = "usb";
        }
        if (isset($options['repo_uri']) && is_array($options['repo_uri'])) {
            foreach ($options['repo_uri'] as $repo) {
                if ($repo['id'] == $id) {
                    $source = $repo['uri'];
                }
            }
        }

        $max_wait = -1;
        if (isset($options['max_wait'])) {
            $max_wait = (int) $options['max_wait'];
        }

        $check_only = false;
        if (isset($options['check_only']) && $options['check_only'] == 'true') {
            $check_only = true;
        }

        $test = false;
        if (isset($options['test']) && $options['test'] == 'true') {
            $test = true;
        }
        if (
            $updater->startUpdate($check_only, $source, $max_wait, false, $test)
        ) {
            return $updater->update_id;
        } else {
            throw new Exception($updater->errorStr);
        }
    }

    /*
	firmware_update_status
	
	object status firmware_update_status(string handle, object options);
	
	Returns the status of a previously started firmware update. 
	When a firmware update is done further calls to this method with the same handle will return an error.
	
	object options {
		"include_log"	: string,	// optional
	};
	
	object status {
		"exit"		: string,
		"mode"		: string,
		"type"		: string,
		"version"	: string,
		"packages"	: int,
		"complete"	: bool,
		"applied_on_reboot"	: bool,
		"done"		: bool,
		"log"		: string,
	};
	handle	
		the opaque string returned by the firmware_update() call, used to identify the call for which the call is done.
	
	include_log	
		specifies if the message log should be included in the response; 
		it should be one of 
			"no" (do not send log, default), 
			"yes" (send it), 
			"on_done" (send it only when done is true, i.e. at the end of the run); 
		if not present "no" is assumed.
	
	exit	
		the exit status of the update process, one of 
			"ok" (completed without error), 
			"error" (an error occurred that prevented the update), 
			"fatal" (a fatal error occurred that left the firmware corrupted); 
		only present when the update process has exited, although it may not be completely done yet.
	mode	
		the running mode, one of 
			"check" (only checking for updates), 
			"update" (updating), 
			"test" (testing updates, for debugging only); 
		not present if not yet known. 
	type	
		the type of update available or being done, one of 
			"firmware" (new firmware), 
			"updater" (firmware updater component only), 
			"minor" (only minor components) or 
			"none" (if no updates available); 
		not present if not yet known.
	version	
		the version of the firmware that is available, if type is "firmware", 
		or version of updater component that is available, if type is "updater"; 
		not present if not applicable.
	packages	
		the number of packages available or being updated; 
		not present if not yet known.
	complete	
		true if the update process will be complete when done, 
		false if the update process should be restarted to get a complete update 
		(e.g., the updater needed to be updated before the complete firmware); 
		not present if not yet known.
	applied_on_reboot
		true if the update will actually applied during reboot and thus reboot
		can take long and the device may reboot multiple times before completing
		the update; not present if not known or if the update does is applied
		before the reboot starts.
	done	
		if true the firmware_update() call and thus the complete update process, is done,
		no further calls to this method can be done with this handle.
	log	
		the complete message log of the firmware update process; not present if not requested.
	
	Note: when a firmware update process exits successfully the HMP automatically restarts, so it will be unreachable until it restarts; 
		  there is however, a 1 minute delay between the end of the firmware update and the start of the shutdown procedure.
*/
    public function firmware_update_status($handle = "", $options = array())
    {
        $result = array();

        if (empty($handle)) {
            $handle = Updater::getRunningId();
            $result['handle'] = $handle;
        }
        $updater = new Updater();

        $logs = $updater->getLog($handle);
        if ($logs === false) {
            throw new Exception("Updater not running");
        }

        $updater->parseLog($logs);

        $done = false;
        $exited = ! $updater->isUpdateRunning();
        if ( $exited ) {
        	$status = $updater->getStatus();
        	// if the updater is no longer running but we are waiting for the
        	// reboot to finish the update process then we are not done yet
        	$done = ($status != "REBOOTING");
        	if ($updater->updater_status === false) {
                // update not running but log is not complete
        		if ($status == "READY" || $status == "REBOOTING") {
                    $result['exit'] = 'ok';
                } elseif ($status == "CORRUPTED") {
                    $result['exit'] = 'fatal';
                } else {
                    $result['exit'] = 'error';
                }
            } else {
                if ($updater->updater_status == "OK") {
                    $result['exit'] = 'ok';
                } else {
                    $result['exit'] = 'error';
                }
            }
        }
        if ( $done ) {
        	$updater->cleanUp($handle);
        }

        if ($updater->updater_mode !== false) {
            $result['mode'] = $updater->updater_mode;
        }
        if ($updater->update_info !== false) {
            if ($updater->update_info['type'] == "FIRMWARE") {
                $result['type'] = "firmware";
            } elseif ($updater->update_info['type'] == "UPDATER") {
                $result['type'] = "updater";
            } elseif ($updater->nb_updates != 0) {
                $result['type'] = "minor";
            } else {
                $result['type'] = "none";
            }
            $result['version'] =
                $updater->update_info['ver'] .
                "-" .
                $updater->update_info['rel'];
        }
        if ($updater->nb_updates !== false) {
            $result['packages'] = intval($updater->nb_updates);
            if ($updater->nb_updates == 0) {
                $result['type'] = "none";
            } elseif ($exited && !isset($result['type'])) {
                $result['type'] = "minor";
            }
        }
        if ($updater->applied_on_reboot) {
        	$result['applied_on_reboot'] = true;
        }
        if ($exited) {
            $result['complete'] = $updater->updater_fallback ? false : true;
        }

        $include_log = false;
        if (isset($options['include_log'])) {
            if ($done && $options['include_log'] == 'on_done') {
                $include_log = true;
            } elseif ($options['include_log'] == 'yes') {
                $include_log = true;
            }
        }

        $result['done'] = $done;
        if ($include_log) {
            $result['log'] = implode("\n", $logs);
        }
        return $result;
    }
    /* set_password
    
    void set_password(object pwinfo);

    Removes or sets a new password for a user on the HMP. Passwords can be set by sending them in clear or by sending the digest.
    Note that setting a password for users others than admin and setting no password for admin offers no protection as the passwords 
    can be changed by anyone having network access to the HMP.
    object pwinfo {
        "type"		: string,
        "user"		: string,
        "realm"	: [ string ...],	// only for type digest
        "password"	: [ string ...],	// only for type other than none
    };
    type	
        The type of password, either none (removes password for user), cleartext (new password sent in clear) or digest (password digest sent for each realm).
    user	
        The user for which to set the new password, either admin, content or monitoring.
    realm	
        The realms used to encode the password digests; only used for type digest. The realms to be used for each user are specified below.
    password	
        For type cleartext this array should contain a single element, which is the new password in clear. 
        For type digest it should be the array of digests of the password obtained for each realm.
    */
    public function set_password($pwinfo = null)
    {
        if ($pwinfo === null) {
            throw new Exception("Missing pwinfo");
        }

        if (!isset($pwinfo['type'])) {
            throw new Exception("Type is required");
        }

        if (!isset($pwinfo['user'])) {
            throw new Exception("User is required");
        }

        if ($pwinfo['type'] == 'digest') {
            throw new Exception("Digest no longer supported");
        }

        if (isset($pwinfo['password']) && is_array($pwinfo['password'])) {
            if (count($pwinfo['password']) > 1) {
                throw new Exception(
                    "Password array can contain only a single entry"
                );
            } else {
                $pwinfo['password'] = $pwinfo['password'][0];
            }
        }

        if ($pwinfo['type'] != 'none' && !isset($pwinfo['password'])) {
            throw new Exception(
                "Password is required for type=" . $pwinfo['type']
            );
        }

        createYiiWebApplication();

        $user = User::model()->findByAttributes(array(
            'username' => $pwinfo['user'],
        ));

        if (!$user) {
            throw new Exception("User not found");
        }

        switch ($pwinfo['type']) {
            case 'none':
                if (!$user) {
                    throw new Exception("User not found");
                }
                if ($user->id <= 1) {
                    throw new Exception("Cannot delete user admin");
                }
                // clearing the user
                $transaction = $user->getDbConnection()->beginTransaction();
                try {
                    if (!$user->delete()) {
                        throw new Exception("Cannot delete user");
                    } else {
                        if (!$user->updateHttpdSecurity()) {
                            throw new Exception("update user failed");
                        }
                    }
                    $transaction->commit();
                } catch (Exception $e) {
                    // an exception is raised if a query fails
                    $transaction->rollback();
                    throw $e;
                }
                break;
            case 'cleartext':
            case 'crypt':
                // updating an user

                $transaction = $user->getDbConnection()->beginTransaction();
                try {
                    $user->password = $pwinfo['password'];
                    $user->verifyPassword = $pwinfo['password'];
                    $user->lowSecurityPassword = $pwinfo['type'] == "cleartext";
                    $user->crypted = $pwinfo['type'] == "crypt";
                    if ($user->save()) {
                        if (!$user->updateHttpdSecurity()) {
                            throw new Exception("Update user failed");
                        }
                    } else {
                        throw new Exception("Save user failed");
                    }
                    $transaction->commit();
                } catch (Exception $e) {
                    // an exception is raised if a query fails
                    $transaction->rollback();
                    throw $e;
                }
                break;
            default:
                throw new Exception(
                    "Type must be one of the following: none, cleartext or crypt"
                );
        }
    }

    public function set_config($config = null)
    {
        if ($config === null) {
            throw new Exception("Missing config");
        }

        if (!isset($config['xmlconfig']) || $config['xmlconfig'] == "") {
            throw new Exception("xmlconfig is required");
        }

        createYiiWebApplication();

        $result = array();
        $backup = new ConfigBackup();
        $backup->shutdownDelay =
            $this->checkLocalhostCall() && $this->callId !== null ? 1 : false;

        $backup->apply($config['xmlconfig']);

        if ($backup->hasErrors()) {
            $errs = array();
            foreach ($backup->getErrors() as $attr => $desc) {
                if (is_array($desc)) {
                    foreach ($desc as $d) {
                        $errs[] = array(
                            'element' => $attr,
                            'description' => $d,
                        );
                    }
                } else {
                    $errs[] = array('element' => $attr, 'description' => $desc);
                }
            }
            $result['errors'] = $errs;
        }

        $result['success'] = !$backup->hasErrors();
        $result['reboot_pending'] = Tools::checkInstallationmode();
        $result['reboot'] =
            Tools::doShutdown() && !Tools::checkInstallationmode();
        $this->needreload = $backup->needreload;
        return $result;
    }

    public function get_config()
    {
        createYiiWebApplication();

        //Yii::setPathOfAlias('application', dirname(__FILE__).DIRECTORY_SEPARATOR.'protected' );
        //Yii::import('application.models.*');
        // Need users

        $backup = new ConfigBackup();

        $result = array();

        $config = $backup->create();
        $result['xmlconfig'] = $config;

        return $result;
    }

    public function add_pull_action($action = null)
    {
        if ($action === null) {
            throw new Exception("Missing action");
        }

        if (!isset($action['type'])) {
            throw new Exception("type is required");
        }

        if (!isset($action['uri'])) {
            throw new Exception("uri is required");
        }

        if (
            $action['type'] != "publish" &&
            $action['type'] != "upload" &&
            $action['type'] != "rpc" &&
            $action['type'] != "aws-rpc"
        ) {
            throw new Exception(
                "type must be one of the following: publish or upload"
            );
        }

        $this->checkPullMode($action['type']);

        if (!isset($action['max_retry'])) {
            $action['max_retry'] = 3600;
        }

        $pull = new PullMode();
        return $pull->createEvent(
            $action['type'],
            $action['uri'],
            $action['max_retry'],
            isset($action['options']) ? $action['options'] : null
        );
    }

    public function trigger_enrollment($options = null)
    {
        $result = array();

        $result['success'] = MaintenanceJobs::triggerEnrollment();

        return $result;
    }

    public function set_rpc_poll_fallback_times($options = null)
    {
        $result = array();
        if ($options === null) {
            throw new Exception("Missing options");
        }
        if (!isset($options['service'])) {
            throw new Exception("service is required");
        }
        if (!isset($options['concentratorId'])) {
            throw new Exception("concentratorId is required");
        }
        if (!isset($options['interval'])) {
            $options['interval'] = 0;
        }
        if (!isset($options['startDelay'])) {
            $options['startDelay'] = 0;
        }
        if (!isset($options['duration'])) {
            $options['duration'] = 0;
        }

        $result['success'] = MaintenanceJobs::setRpcPollFallbackTimes(
            $options['service'],
            $options['concentratorId'],
            $options['interval'],
            $options['startDelay'],
            $options['duration']
        );

        return $result;
    }

    public function reset($options = null)
    {
        if ($options === null) {
            throw new Exception("Missing options");
        }
        if (isset($options['web-page-data']) && $options['web-page-data'] !== 
            false) {
            $brandingInfo = new BrandingInfo();
            if (!$brandingInfo->hasRight('HTML')) {
                throw new Exception('Parameter web-page-data not ' . 
                    'supported by this model');
            }
            $clearWebPageData = true;
        }

        $result = array();

        $jobs = new MaintenanceJobs();
        $restart = false;
        $success = true;

        if (isset($options['arya']) && $options['arya'] !== false) {
            $success &= $jobs->clearConfigured();
            $success &= $jobs->resetContent();
            $success &= $jobs->clearWebstorage();

            $success &= MaintenanceJobs::enableARYAMode();
            $success &= MaintenanceJobs::enableCloudConnection();

            $restart = $success;
        }
        if (
            isset($options['published-content']) &&
            $options['published-content'] !== false
        ) {
            $success &= $jobs->resetContent();
        }
        if (isset($options['content']) && $options['content'] !== false) {
            $success &= $jobs->resetContent();
            $success &= $jobs->clearInterfaceContent();
            $success &= $jobs->clearWebstorage();
            $restart = $success;
        }
        if (isset($options['webstorage']) && $options['webstorage'] !== false) {
            $success &= $jobs->clearWebstorage();
        }
        if (isset($options['log']) && $options['log'] !== false) {
            $success &= $jobs->clearLogs();
        }
        if (isset($options['cache']) && $options['cache'] !== false) {
            $success &= $jobs->clearCache();
        }
        if (isset($options['ntp']) && $options['ntp'] !== false) {
            $success &= $jobs->clearNtp();
            $restart = $success;
        }
        if (isset($options['factory']) && $options['factory'] !== false) {
            $success &= $jobs->resetFactory();
            $restart = $success;
        }
        if (!empty($clearWebPageData)) {
            $success &= $jobs->clearWebPageData();
            $restart = $success;
        }

        if ($restart && $success) {
            $delay =
                $this->checkLocalhostCall() && $this->callId !== null
                    ? 1
                    : false;
            $ret = MaintenanceJobs::shutdown(
                "user initiated shutdown : rpc reset() call",
                false,
                $delay
            );
        }
        $result['success'] = !!$success;
        $result['reboot'] = !!$restart;

        return $result;
    }

    public function webstorage_list()
    {
        $this->checkWebstorage();

        $store = new WebStorageAPI();

        $ret = $store->getList();

        if ($ret === false) {
            throw new Exception("Internal error");
        }

        $store->close();
        return $ret;
    }
    public function webstorage_get($names = null)
    {
        $this->checkWebstorage();

        if ($names === null) {
            throw new Exception("Missing names");
        }

        if (isset($names['debug'])) {
            $names = $names['debug'];
        }
        if (!is_array($names) || isset($names['name'])) {
            throw new Exception("expect array as argument");
        }
        $store = new WebStorageAPI();
        $ret = array();
        foreach ($names as $name) {
            $val = $store->get($name);
            if ($val === false) {
                throw new Exception("Internal error");
            }
            $ret[] = $val;
        }
        $store->close();
        return $ret;
    }
    public function webstorage_set($variables = null)
    {
        $this->checkWebstorage();

        if ($variables === null) {
            throw new Exception("Missing variables");
        }

        if (isset($variables['debug'])) {
            $variables = $variables['debug'];
        }
        if (!is_array($variables) || isset($variables['name'])) {
            throw new Exception("expect array as argument");
        }
        foreach ($variables as $name => $variable) {
            if (!is_array($variable)) {
                throw new Exception("name and value required");
                //$variables[] = array('value' => $variable, 'name' => $name );
                //unset( $variables[$name]);
                //continue;
            }
            if (!isset($variable['name'])) {
                throw new Exception("name is required");
            }
            if (!isset($variable['value'])) {
                throw new Exception("value is required");
            }
        }

        $store = new WebStorageAPI();
        $ret = array();
        foreach ($variables as $variable) {
            $timestamp = "+0";
            if (isset($variable['offset'])) {
                $timestamp = "+" . $variable['offset'];
            }
            if (isset($variable['timestamp'])) {
                $timestamp = $variable['timestamp'];
            }
            $val = $store->set(
                $variable['name'],
                $variable['value'],
                $timestamp
            );
            if ($val === false) {
                $store->close();
                throw new Exception("Internal error");
            }
            $ret[] = $val;
        }
        $store->close();
        return $ret;
    }
    public function webstorage_remove($names = null)
    {
        $this->checkWebstorage();

        if ($names === null) {
            throw new Exception("Missing names");
        }

        if (isset($names['debug'])) {
            $names = $names['debug'];
        }
        if (!is_array($names) || isset($names['name'])) {
            throw new Exception("expect array as argument");
        }
        $store = new WebStorageAPI();
        $ret = array();
        foreach ($names as $name) {
            $val = $store->remove($name);
            if ($val === false) {
                throw new Exception("Internal error");
            }
            $ret[] = $val;
        }
        $store->close();
        return $ret;
    }
    public function webstorage_cmpxchg($variables = null)
    {
        $this->checkWebstorage();

        if ($variables === null) {
            throw new Exception("Missing variables");
        }

        if (isset($variables['debug'])) {
            $variables = $variables['debug'];
        }
        if (!is_array($variables) || isset($variables['name'])) {
            throw new Exception("expect array as argument");
        }
        foreach ($variables as $variable) {
            if (!isset($variable['name'])) {
                throw new Exception("name is required");
            }
            if (!isset($variable['value'])) {
                throw new Exception("value is required");
            }
        }
        $store = new WebStorageAPI();
        $ret = array();
        foreach ($variables as $variable) {
            $timestamp = "+0";
            if (isset($variable['offset'])) {
                $timestamp = "+" . $variable['offset'];
            }
            if (isset($variable['timestamp'])) {
                $timestamp = $variable['timestamp'];
            }

            if (!isset($variable['expect'])) {
                $val = $store->create(
                    $variable['name'],
                    $variable['value'],
                    $timestamp
                );
            } else {
                $val = $store->cmpxchg(
                    $variable['name'],
                    $variable['expect'],
                    $variable['value'],
                    $timestamp
                );
            }
            if ($val === false) {
                throw new Exception("Internal error");
            }
            $ret[] = $val;
        }
        $store->close();
        return $ret;
    }

    public function get_license_device_info($context = null)
    {
        return MaintenanceJobs::licensecheck($context);
    }

    public function revoke_license($counter, $context = null)
    {
        if ($counter === null) {
            throw new Exception("Missing counter");
        }

        $check = MaintenanceJobs::licensecheck($context);

        if (
            !isset($check["tpm2"]) ||
            !isset($check["tpm2"]["licenseCounter"]) ||
            !isset($check["tpm2"]["licenseCounter"]["val"])
        ) {
            throw new Exception("Internal error: Missing counter");
        }

        if ($check["tpm2"]["licenseCounter"]["val"] != $counter) {
            throw new Exception("Counter mismatch");
        }

        if (!MaintenanceJobs::licenseupdate()) {
            throw new Exception("Internal error: License update failed");
        }

        return MaintenanceJobs::licensecheck($context, true);
    }
}

$obj = new RPC();
$rpc = new JSONRPCServer($obj);

if ($rpc->run() !== null) {
    if ($obj->needreload) {
        MaintenanceJobs::reloadHttpd();
    }
    exit(0);
}
header("HTTP/1.0 400 Bad Request");
header('Content-Type: text/html; charset=UTF-8');
?>
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN"
   "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">
<html>
<head>
<title>Invalid RPC call</title>
<meta http-equiv="Content-Type" content="text/html; charset=UTF-8"/>
</head>
<body>
<h1>Invalid RPC call</h1>
<p>This is a JSON-RPC server, only the <?php echo htmlspecialchars(
    JSONRPCServer::CONTENT_TYPE
); ?> 
content-type
is accepted via the POST method with properly formed JSON-RPC 1.0 requests.</p>
</body>
</html>
