#!/bin/bash
#
# License: Copyright 2015 SpinetiX. This file is licensed
#          under the terms of the GNU General Public License version 2.
#          This program is licensed "as is" without any warranty of any
#          kind, whether express or implied.
#
# Copyright 1999-2003 MontaVista Software, Inc.
# Copyright 2002, 2003, 2004 Sony Corporation
# Copyright 2002, 2003, 2004 Matsushita Electric Industrial Co., Ltd.
#
### BEGIN INIT INFO
# Required-Start:
# Required-Stop:
# Should-Start:
# Should-Stop:
# Default-Start: S
# Default-Stop: 0 1 2 3 4 5 6
# Short-Description: Storage partitioning and formatting.
# Description: Storage partitioning and formatting.
### END INIT INFO

# Init script information
NAME=spxformat.sh
DESC=""

# The file that permanently stores configuration changes so as to be
# robust against reboots while resetting.
BOOTCONFIG=/boot-config

# The partition type to use
PTYPE=83

# The directory where the filesystem is normally mounted
MOUNTDIR=/srv

# function to load configuration according to hardware type
function load_config
{
    local tag sep val
    local hw
    while read tag sep val; do
        if [ "$tag" = "Hardware" ]; then
            hw="$val"
            break
        fi
    done < /proc/cpuinfo

    if [ -z "$hw" ]; then
        echo "ERROR: could not find hardware type" >&2
        return 1
    fi

    #
    # MAINDEV = whole block device
    # PARTPREFIX = prefix name to form partitions
    # PARTNUM = storage partition number
    # MINPSTART = minimum sector for partition start
    # MINPSIZE = minimum size of partition, in sectors
    # MINENDFREE = minimum free space at end, in sectors
    # FSTYPE = type of filesystem (blkid, mkfs, etc.)
    # MKFSBLKSZ = fs block size, in bytes
    # MKFSOPTS = extra options to pass to mkfs
    # FWPARTNUM = firmware package partition number (0 if none)
    # PARTDEV = storage partition block device
    #
    case "$hw" in
        "Bonsai")
            MAINDEV=/dev/mmcblk0
            PARTPREFIX="$MAINDEV"p
            PARTNUM=2
            MINPSTART=8192 # 4 MiB aligned
            MINPSIZE=524288
            MINENDFREE=0
            FSTYPE=ext3
            MKFSBLKSZ=4096
            MKFSOPTS="-m 1"
            FWPARTNUM=4
            ;;
        "Sakura")
            MAINDEV=/dev/hda
            PARTPREFIX="$MAINDEV"
            PARTNUM=1
            MINPSTART=8192 # 4 MiB aligned
            MINPSIZE=524288
            MINENDFREE=0
            FSTYPE=ext3
            MKFSBLKSZ=4096
            MKFSOPTS="-m 1"
            FWPARTNUM=4
            ;;
        "ikebana")
            MAINDEV=/dev/mmcblk0
            PARTPREFIX="$MAINDEV"p
            PARTNUM=3
            MINPSTART=16384 # 8 MiB aligned
            MINPSIZE=1048576
            MINENDFREE=256 # leave space for GPT backup
            FSTYPE=ext4
            MKFSBLKSZ=4096
            MKFSOPTS="-m 1"
            FWPARTNUM=0
            ;;
        *)
            echo "ERROR: unknown hardware type '$hw'" >&2
            return 1
    esac

    PARTDEV="$PARTPREFIX""$PARTNUM"

    return 0
}

function check_bdev() {
    local dev="$1"
    local n
    local r=1
    for n in 1 2 3 4 5 6; do
        if [ ! -b "$dev" ]; then
            sleep 5
        else
            sleep 1
            if [ -b "$dev" ]; then
                r=0
                break
            fi
        fi
    done
    return $r
}

function format_extN() {
    local bdev="$1"

    # the optimal I/O size is the preferred sequential I/O size;
    # attempt to obtain it from the kernel, but if the default of 0 or
    # 512 is present then set it to 16 KiB, which is a usual and
    # reasonable super page size.
    local io_size_opt=
    [ -f /sys/block/${MAINDEV##*/}/queue/optimal_io_size ] && \
        io_size_opt="$(< /sys/block/${MAINDEV##*/}/queue/optimal_io_size)"
    [ -z "$io_size_opt" ] && io_size_opt=0
    [ "$io_size_opt" -le 512 ] && io_size_opt=16384

    # the minimal I/O size if the preferred random I/O size; attempt
    # to obtain it from the kernel, but if the default of 0 or 512 is
    # present then set it to 16 KiB, which is a usual and reasonable
    # super page size.
    local io_size_min=
    [ -f /sys/block/${MAINDEV##*/}/queue/minimum_io_size ] && \
        io_size_min="$(< /sys/block/${MAINDEV##*/}/queue/minimum_io_size)"
    [ -z "$io_size_min" ] && io_size_min=0
    [ "$io_size_min" -le 512 ] && io_size_min=16384

    if [ ! -b "$bdev" ]; then
        echo "ERROR: partition block device not present" >&2
        return 1
    fi
    mkfs."$FSTYPE" -q -L sys-data -b "$MKFSBLKSZ" \
        -E stride=$((io_size_min/MKFSBLKSZ)),stripe_width=$((io_size_opt/MKFSBLKSZ)) \
        $MKFSOPTS "$bdev"
}

function kill_fs() {
    local bdev="$1"
    if [ ! -b "$bdev" ]; then
        echo "ERROR: partition block device not present" >&2
        return 1
    fi
    dd if=/dev/zero bs=4096 count=1 of="$bdev" 2>/dev/null
}

function check_fs_present() {
    # For now we just test that the filesystem is of the expected type
    [ "$(blkid -c /dev/null -w /dev/null -o value -s TYPE "$PARTDEV")" = "$FSTYPE" ]
}

function check_partition() {
    sfdisk -d $MAINDEV | \
        awk -F '[ =,:]+' -v p=$PARTDEV -v minst=$MINPSTART \
        -v minsz=$MINPSIZE -v id=$PTYPE \
        'BEGIN { r = 1 } END { exit r } $1 == p { if ($3 >= minst && $5 >= minsz && $7 == id) {r = 0; exit} }'
}

# partitions the storage device
#
# WARNING: this only really works on machines which do not have the
# system (root) partition on the device to be partitioned, but should
# anyhow never be called otherwise as the system partition needs to be
# present to execute this script and fails because the device is in
# use and cannot be repartitioned.
function make_partition() {
    local nrsects nrblocks
    local pdata
    local pstart psize pend pid
    local minsect maxsect maxend
    local start size
    local fwstart fwsize
    local i

    # Check that we can change the partition table
    if ! blockdev --rereadpt "$MAINDEV" 2>/dev/null; then
        echo "ERROR: internal storage device in use, cannot repartition" >&2
        return 1
    fi

    # currently only machines that use the first partition are handled correctly
    if [ "$PARTUM" != 1 ]; then
        echo "ERROR: partitioning is not supported for this machine's layout" >&2
        return 1
    fi

    # Get the size of the device
    nrsects="$(blockdev --getsz "$MAINDEV")"
    if [ -z "$nrsects" ]; then
        echo "ERROR: failed getting size of internal storage" >&2
        return 1
    fi
    nrblocks=$((nrsects / 2))

    # Get the partition data into pdata array for partitions 1 to 4,
    # for each partition we get the start sector, size (in sectors) and id
    # unassigned partitions get all zero fields
    pdata=($(sfdisk -d "$MAINDEV" | awk -F '[ =,:]+' -v d="$PARTPREFIX"  '$1 ~ "^"d"[1-4]" && $2 == "start" { print $3, $5, $7 }'))

    # Extract the start, size, end and id for each partition (one based arrays)
    for i in 1 2 3 4; do pstart[$i]=${pdata[$(((i-1)*3))]}; done
    for i in 1 2 3 4; do psize[$i]=${pdata[$(((i-1)*3+1))]}; done
    for i in 1 2 3 4; do pend[$i]=$(( ${pstart[$i]} + ${psize[$i]} )); done
    for i in 1 2 3 4; do pid[$i]=${pdata[$(((i-1)*3+2))]}; done

    # Get the overall start and end of partitions
    minsect=-1
    maxsect=0
    for i in 1 2 3 4; do
        [ "${psize[$i]}" -eq 0 ] && continue
        [ "$minsect" -eq -1 -o "$minsect" -gt "${pstart[$i]}" ] && minsect="${pstart[$i]}"
        [ "$maxsect" -lt "${pend[$i]}" ] && maxsect="${pend[$i]}"
    done

    # Fill in the main partition definition
    start="$MINPSTART"
    [ "$start" -lt "$minsect" ] && start="$minsect"
    maxend=$(( nrblocks * 2 - MINENDFREE ))
    [ "$maxsect" -eq 0 -o "$maxsect" -gt "$maxend" ] && maxsect="$maxend"

    if [ "$FWPARTNUM" != 0 -a "${pid[$FWPARTNUM]}" = "$PTYPE" ]; then
        # firmware package partition to preserve
        size=$(( pstart[$FWPARTNUM] - start ))
        fwstart="${pstart[$FWPARTNUM]}"
        fwsize="${psize[$FWPARTNUM]}"
    else
        # no partitions to preserve, just use till end of device
        size=$(( maxsect - start ))
        fwstart=0
        fwsize=0
    fi

    # Fill in all partition definitions
    for (( i = 1; i <= 4; i++ )) ; do
        if [ "$i" -eq "$PARTNUM" ]; then
            # main partition
            pdef[$i]="$start,$size,$PTYPE,*"
        elif [ "$i" -eq "$FWPARTNUM" -a "$fwsize" -ne 0 ]; then
            # firmware package partition
            pdef[$i]="$fwstart,$fwsize,$PTYPE"
        else
            # unallocated partition
            pdef[$i]=",0"
        fi
    done

    # Now create the partitions
    echo -e "${pdef[1]}\n${pdef[2]}\n${pdef[3]}\n${pdef[4]}\n" | \
        sfdisk -f "$MAINDEV" > /dev/null

    # And finally wait until the device node appears
    if ! check_bdev "$PARTDEV"; then
        echo "ERROR: timed out waiting for block device $MAINDEV" >&2
        return 1
    fi

}

# creates the necessary directories in the newly created fs so that
# all bind mounts in fstab will succeed.
function populate_fs() {
    local bdev="$1"
    local ret=0
    local dev mpt type opts junk

    mount -t "$FSTYPE" "$bdev" "$MOUNTDIR"
    if [ $? -ne 0 ]; then
        echo "ERROR: failed to mount newly created fs" >&2
        return 1
    fi
    while read dev mpt type opts junk; do
        [ "$opts" = bind ] || continue
        [ "${dev#$MOUNTDIR/}" != "$dev" ] || continue
        if [ ! -d "$mpt" ]; then
            echo "ERROR: bind mount new dir '$mpt' is not a directory" >&2
            ret=1
            break
        fi
        mkdir -p "$dev" && chmod --reference="$mpt" "$dev"
        if [ $? -ne 0 ]; then
            echo "ERROR: could not create bind mount source dir '$dev'" >&2
            ret=1
            break
        fi
    done < /etc/fstab
    umount "$MOUNTDIR"
    if [ $? -ne 0 ]; then
        echo "ERROR: failed to unmount newly created fs" >&2
        ret=1
    fi
    return $ret
}

function is_root_ro() {
    local dev mnt type opts dump pass junk
    local ret=1

    # a mountpoint may appear multiple times, last one wins
    while read dev mnt type opts dump pass junk; do
        [ "$mnt" = "/" ] || continue
        case "$opts" in
            ro|ro,*|*,ro|*,ro,*)
                ret=0
                ;;
            *)
                ret=1
        esac
    done < /proc/mounts

    return $ret
}

# Does the internal storage format, if the first argument is -f the format
# is forced, otherwise it is done only if not yet formatted
function do_format() {
    local isforced
    local rootmode

    [ "$1" = "-f" ] && isforced=yes && shift

    # Skip this on NFS root systems, unless forced, as we can otherwise
    # break havoc on special configurations and such
    if [ -z "$isforced" ] && grep -q '\bnfsroot=' /proc/cmdline; then
        echo "NFS root system, skipping internal storage formatting"
        return 0
    fi

    if ! load_config; then
        echo "ERROR: failed to load internal storage configuration" >&2
        return 1
    fi

    if [ ! -b $MAINDEV ]; then
        echo "ERROR: internal storage device not present" >&2
        return 1
    fi

    if [ -n "$isforced" ]; then
        echo -n "Internal storage format forced: "
    elif ! check_fs_present; then
        echo -n "Internal storage needs format: "
    else
        return 0
    fi

    # we need to format, if not already there write the config order
    # to stable storage (atomically) or a power cut during the format
    # will leave us with a half initialized filesystem; if we are in
    # forced mode then the config order is already on stable storage

    if [ -z "$isforced" ]; then
        if [ -f "$BOOTCONFIG" ]; then
            cat "$BOOTCONFIG" > "$BOOTCONFIG".new
        else
            echo -n > "$BOOTCONFIG".new
        fi
        [ $? -eq 0 ] && echo "format" >> "$BOOTCONFIG".new && sync && \
            mv -f "$BOOTCONFIG".new "$BOOTCONFIG" && sync
        if [ $? -ne 0 ]; then
            echo "ERROR: failed to save format config order" >&2
            return 1
        fi
    fi

    if ! check_partition; then
        # partition is not there or not right, attempt to make it
        echo -n "making partition... "
        make_partition
        if [ $? -ne 0 ]; then
            echo "ERROR: main partition not present and failed creating it" >&2
            return 1
        fi
    fi

    echo -n "formatting... "
    format_extN "$PARTDEV"
    if [ $? -ne 0 ]; then
        echo "ERROR: failed to format internal storage" >&2
        kill_fs "$PARTDEV"
        return 1
    fi

    echo -n "populating... "
    populate_fs "$PARTDEV"
    if [ $? -ne 0 ]; then
        echo "ERROR: failed populating newly created fs" >&2
        kill_fs "$PARTDEV"
        return 1
    fi

    # succeeded, remove the format config order atomically
    is_root_ro && rootmode=ro || rootmode=rw
    [ "$rootmode" = "ro" ] && mount -o remount,rw /
    sed -e '/^format\($\|=\)/d' "$BOOTCONFIG" > "$BOOTCONFIG".new && sync
    if [ $? -eq 0 ]; then
        if [ -s "$BOOTCONFIG".new ]; then
            mv -f "$BOOTCONFIG".new "$BOOTCONFIG" && sync
        else
            rm -f "$BOOTCONFIG" "$BOOTCONFIG".new && sync
        fi
    else
        rm -f "$BOOTCONFIG".new
        [ -n "" ] # just make a non-zero exit code to trigger message below
    fi
    if [ $? -ne 0 ]; then
        echo "ERROR: failed to remove format config order" >&2
    fi
    [ "$rootmode" = "ro" ] && mount -o remount,"$rootmode" /

    echo "done"
    return 0
}

case "$1" in
    start)
        if [ -f "$BOOTCONFIG" ] && grep -q '^format\($\|=\)' "$BOOTCONFIG"; then
            do_format -f # do a forced format
        else
            do_format # do a format if required only
        fi
        ;;
    stop)
        ;;
esac
