<?php

require_once dirname(__FILE__).'/../utils/MainInfo.php';

class ScreenInfo {
	
    public $resolution;
    public $refresh;
    public $aspect;
    public $monitors = [];
    public $powersaveSupported = true;
    public $cecSupported = true;
    public $settingsStr;

    public function load()
    {
        $this->getMonitorsInfo();

        if ($this->monitors) {
            $this->getRenderInfo();
            $this->getConnectionInfo();
        }
    }

    public function getMonitorState()
    {
        if (count($this->monitors) > 1) {
            return array('text' => 'Multi-output', 'alert' => false);
        } elseif ($this->monitors) {
            return $this->monitors[0];
        } else {
            return self::NOT_FOUND;
        }
    }

    private const NOT_FOUND = array('text' => 'NOT FOUND', 'alert' => true);

    private static function getOutputLabel(&$monitor, $labels)
    {
        $id = $monitor['id'];
        $monitor['outputLabel'] = !empty($labels[$id]) ? $labels[$id] : $id;
    }

    private static function getPosition(&$monitor) 
    {
        $region = $monitor['mode']['region'];
        $monitor['x'] = min($region['x1'], $region['x2']);
        $monitor['y'] = min($region['y1'], $region['y2']);
    }

    private static function getSettings(&$monitor)
    {
        $mode = $monitor['mode'];
        $monitor['settings'] = self::stringify(
            $mode['width'],
            $mode['height'],
            $mode['refresh'],
            $monitor['aspectRatio'] ?? null
        );
        if (isset($mode['hdr'])) {
            $hdr = $mode['hdr'];
            $monitor['settings'] .= " (HDR {$hdr['transfer']} " .
                "{$hdr['bpc']} bits {$mode['colorspace']})";
        } elseif (isset($mode['colorspace'])) {
            $monitor['settings'] .= " ({$mode['colorspace']})";
        }
    }

    private static function stringify(
        $width,
        $height,
        $refresh,
        $aspectRatio
    ) {
        $refresh = round($refresh, 2);
        $settings = "{$width}x{$height}@{$refresh}Hz";
        $ratio = explode(':', $aspectRatio);
        if (count($ratio) == 2 && $ratio[0] != 0) {
            $settings .= " ({$aspectRatio})";
        }
        return $settings;
    }

    private static function getState(&$monitor)
    {
        if (isset($monitor['powered'])) {
            $powerStates = array(
                'on' => 'ON',
                'off' => 'OFF',
                'unknown' => 'CONNECTED'
            );
            $monitor['outputLabel'] .= ' (connected)';
            $monitor['text'] = $powerStates[$monitor['powered']];
            $monitor['alert'] = $monitor['powered'] === 'off';
        } else {
            $monitor += self::NOT_FOUND;
        }
    }

    private static function getSync(&$monitor)
    {
        if (isset($monitor['vsync'])) {
            $vsync = $monitor['vsync'];
            $monitor['sync'] = number_format($vsync['diff'], 2, '.', '') .
                ' ms (' . number_format($vsync['ppm'], 1, '.', '') . ' ppm)';
        } else {
            $monitor['sync'] = 'Not Active';
        }
    }

    private function getMonitorsInfo()
    {
        $info = new ScreenInfoBasics();
        $info->load();
        $this->monitors = $info->monitors;

        $outputLabels = Yii::app()->branding->videoOutputs;
        foreach ($this->monitors as &$monitor) {
            self::getOutputLabel($monitor, $outputLabels);
            self::getPosition($monitor);
            self::getSettings($monitor);
            self::getState($monitor);
            self::getSync($monitor);
        }
    }

    private function getRenderInfo()
    {   
        $mode = $this->monitors[0]['mode'];
        $region = new Region();
        $region->setCoordinates($mode['region']);
        $refresh = $mode['refresh'];

        foreach(array_slice($this->monitors, 1) as $monitor) {
            $mode = $monitor['mode'];
            $region->encompassCoordinates($mode['region']);
            $refresh = max($mode['refresh'], $refresh);
        }

        $this->resolution = ScreenInfoBasics::getResolution(
            $region->width(),
            $region->height()
        );
        $this->refresh = $refresh;
        if (
            count($this->monitors) === 1 && 
            isset($this->monitors[0]['aspectRatio'])
        ) {
            $this->aspect = $this->monitors[0]['aspectRatio'];
        }
        $this->settingsStr = self::stringify(
            $region->width(),
            $region->height(),
            $refresh,
            $this->aspect
        );
    }

    private function getConnectionInfo()
    {
        foreach($this->monitors as $monitor) {
            if (isset($monitor['powered'])) {
                // Monitor is connected
                if (!$monitor['pm_supported']) {
                    $this->powersaveSupported = false;
                    if (!$monitor['cec_capable']) {
                        $this->cecSupported = false;
                    }
                }
            }
        }
    }
}
