<?php

$spxsNS = 'http://www.spinetix.com/namespace/1.0/spxstatus';
$spxsP = 'spxs';

require_once "BrandingInfo.php"; // branding info

// Serial number
function getInfoSerial(&$serial, &$model, &$hardware)
{
    $serviceSerial = "[undefined]";
    $cpuinfo = file_get_contents("/proc/cpuinfo");
    $identifiers_path = "/etc/spinetix/identifiers";
    if (file_exists($identifiers_path)) {
        $identifiers = file_get_contents($identifiers_path);
    } else {
        $identifiers = "";
    }

    if (preg_match("/Serial\s+:\s+(\S+)/", $cpuinfo, $match)) {
        $serviceSerial = $match[1];
    } elseif (
        preg_match(
            '/^\s*SPX_SN=(\'|"|)((?![\'"]).*?)\1\s*$/m',
            $identifiers,
            $match
        )
    ) {
        $serviceSerial = $match[2];
    }

    if (
        preg_match(
            '/^\s*SPX_USER_SN=(\'|"|)((?![\'"]).*?)\1\s*$/m',
            $identifiers,
            $match
        )
    ) {
        $userSerial = $match[2];
    } else {
        $userSerial = $serviceSerial;
    }

    $model = "[undefined]";
    if (preg_match("/Hardware\W+:\s+(\w+)/", $cpuinfo, $match)) {
        $model = $match[1];
    } elseif (
        preg_match(
            '/^\s*SPX_HW_TYPE=(\'|"|)((?![\'"]).*?)\1\s*$/m',
            $identifiers,
            $match
        )
    ) {
        $model = $match[2];
    }

    $hardware = "0.0";
    if (preg_match("/Revision\W+:\s+(\w+)/", $cpuinfo, $match)) {
        if ($model == 'Bonsai') {
            $val = hexdec($match[1]);
            $valc = floor($val / 65536);
            $core = floor($valc / 256) . "." . $valc % 256;
            $valc = $val % 65536;
            $carier = floor($valc / 256) . "." . $valc % 256;
            $hardware = $core . "/" . $carier;
        } else {
            $val = hexdec($match[1]);
            $valc = $val % 65536;
            $carier = floor($valc / 256) . "." . $valc % 256;
            $hardware = $carier;
        }
    }
    $serial = $userSerial;
    $ret = array(
        "serial" => $serial,
        "model" => $model,
        "hardware" => $hardware,
        "hasSoftLicense" => $model != "ikebana",
        "hasDrmVideoConnectors" => $model != "ikebana",
        "hasDynamicAudioConnectors" => $model != "ikebana",
        "altSerial" => array(
            "service" => $serviceSerial,
        ),
    );
    if ($userSerial !== $serviceSerial) {
        $ret["altSerial"]["user"] = $userSerial;
    }

    if (
        preg_match(
            '/^\s*SPX_BOARD_SN=(\'|"|)((?![\'"]).*?)\1\s*$/m',
            $identifiers,
            $match
        )
    ) {
        $ret["altSerial"]["board"] = $match[2];
    }
    if (
        preg_match(
            '/^\s*SPX_SYSTEM_SN=(\'|"|)((?![\'"]).*?)\1\s*$/m',
            $identifiers,
            $match
        )
    ) {
        $ret["altSerial"]["system"] = $match[2];
    }
    if (
        preg_match(
            '/^\s*SPX_CHASSIS_SN=(\'|"|)((?![\'"]).*?)\1\s*$/m',
            $identifiers,
            $match
        )
    ) {
        $ret["altSerial"]["chassis"] = $match[2];
    }

    return $ret;
}
function getInfoModel()
{
    $brandingInfo = new BrandingInfo();
    return $brandingInfo->product;
}

function getStorageSizes($model)
{
    if ($model == "Bonsai" || $model == "ikebana") {
        $name = "mmcblk0";
    } else {
        $name = "hda";
    }

    if (!file_exists("/proc/partitions")) {
        return array();
    }
    $disk = file_get_contents("/proc/partitions");
    $devs = explode("\n", $disk);

    $sizes = array();
    foreach ($devs as $dev) {
        $data = preg_split("/[\s]+/", $dev);
        if (count($data) == 5 && $data[4] == $name) {
            $sdata = array(
                "location" => "internal",
                "size" => $data[3],
                "unit" => "KiB",
            );
            $sizes[] = $sdata;
        }
    }
    return $sizes;
}

function getInfoSafeMode()
{
    return file_exists("/var/run/raperca/safe-mode");
}

function getInfoUpTime()
{
    $uptime = file_get_contents("/proc/uptime");
    $uptime = explode(" ", $uptime);
    return $uptime[0];
}

function getInfoIpv6($iface = "eth0", $advanced = false, $locallink = false)
{
    if (PHP_OS == "WINNT") {
        if ($locallink) {
            $ipcmd =
                "2: eth0    inet6 fe80::cf9b:1069:ac49:eaf4/64 scope link noprefixroute \       valid_lft forever preferred_lft forever";
        } else {
            $ipcmd =
                "2: eth0    inet6 fd35:299:14bc:0:21d:50ff:fe20:15/64 scope global dynamic \       valid_lft 2591983sec preferred_lft 604783sec\n2: eth0    inet6 fe80::21d:50ff:fe20:15/64 scope link \       valid_lft forever preferred_lft forever";
        }
    } else {
        $ipcmd = shell_exec('ip -o -f inet6 addr show dev ' . escapeshellarg($iface) );
    }
    if (strncmp($iface, 'eth', 3) == 0) {
        $type = 'ethernet';
    } elseif (strncmp($iface, 'wlan', 4) == 0) {
        $type = 'wlan';
    } elseif (strncmp($iface, 'ppp', 3) == 0) {
        $type = 'modem';
    } elseif ($iface == 'lo') {
        $type = 'loopback';
    } else {
        $type = '';
    }
    preg_match_all(
        "@^\d+:\s*\S+\s+.*\binet6\s+([0-9a-fA-F:]+)\/([0-9]+)@m",
        $ipcmd,
        $matches,
        PREG_SET_ORDER
    );
    $addrs = array();
    foreach ($matches as $idx => $value) {
        if ($advanced) {
            $addrs[] = array(
                "addr" => $value[1],
                "prefixLength" => $value[2],
                "interface" => $iface,
                "type" => $type,
            );
        } else {
            $addrs[] = $value[1];
        }
    }
    return $addrs;
}

function getInfoIp(&$ip, &$ipprefix, &$iptype)
{
    $netconf = new ipconfigBasics();
    $netconf->load();

    $ipcmd = shell_exec('ip -o -f inet addr show');
    preg_match_all(
        "@^\d+:\s*(\S+)\s+.*\binet\s+([\d.]+)(?:/(\d+)|\s+)@m",
        $ipcmd,
        $matches,
        PREG_SET_ORDER
    );

    # Multidimensional array. Each IP
    # address gets a row, and each row has three entries: the IP
    # address, prefix length, interface type (e.g., 'ethernet',
    # 'modem') and the interface name (e.g., 'eth0', 'ppp0'). The "main"
    # address is returned in the first row. The loopback interface
    # is not returned.
    $addrs = array();
    foreach ($matches as $idx => $value) {
        $iface = $value[1];
        $addr = $value[2];
        $plen = $value[3];

        if (!$plen) {
            $plen = 32;
        }

        $order = 0; // to order the addresses by decreasing importance

        // add n in [0-99] to ordering to keep ordering if all other
        // ordering criteria are equal
        $order += $idx;

        if (strncmp($iface, 'eth', 3) == 0) {
            $type = 'ethernet';
        } elseif (strncmp($iface, 'wlan', 4) == 0) {
            $type = 'wlan';
        } elseif (strncmp($iface, 'ppp', 3) == 0) {
            $type = 'modem';
        } elseif ($iface == 'lo') {
            $type = 'loopback';
        }
        if ($iface == 'lo') {
            continue;
        }

        // add n*100 to order for interface type (n in [0,9])
        if ($type == 'modem') {
            $order += 100;
        } elseif ($netconf->ifname === $iface) {
            $order += 200;
        } elseif ($type == 'wlan') {
            $order += 700;
        } elseif ($type == 'ethernet') {
            $order += 800;
        } else {
            $order += 900;
        }

        // add n*1000 to order for address class (n in [0,9])
        if (strncmp($addr, '169.254', 7) == 0) {
            $order += 8000;
        } // link-local
        if (strncmp($addr, '127.', 4) == 0) {
            $order += 9000;
        } // host-local

        $addrs[$order] = array($addr, $plen, $type, $iface);
    }

    // apply the ordering
    ksort($addrs);
    $addr = reset($addrs);
    if ($addr) {
        $ip = $addr[0];
        $ipprefix = $addr[1];
        $iptype = $addr[2] === "wlan" ? "WLAN" : ucfirst($addr[2]);
    } elseif (PHP_OS == "WINNT") {
        //$ip = "169.254.0.xxx";
        $ip = "192.168.0.12";
        $ipprefix = 32;
        $iptype = "Ethernet";
    } else {
        $ip = "[not found]";
        $ipprefix = "";
        $iptype = "Unknown";
    }
    return $addrs;
}
function getInfoMac()
{
    $addresses = array();
    if (($identifiers = file_get_contents('/etc/spinetix/identifiers')) 
        && preg_match(
            '/^\s*SPX_MAIN_IFACE_NAME=(\'|")((?![\'"]).*?)\1\s*$/m', 
            $identifiers, $nameMatches
        )
        && preg_match(
            '/^\s*SPX_MAIN_IFACE_HWADDR=(\'|")((?![\'"]).*?)\1\s*$/m', 
            $identifiers, $addressMatches
        )
    ) {
        $addresses[$nameMatches[2]] = strtoupper($addressMatches[2]);
    }
    if ($address = @file_get_contents('/sys/class/net/wlan0/address')) {
        $addresses['wlan0'] = strtoupper($address);
    }
    return $addresses;
}
function getInfoHostname()
{
    return trim(file_get_contents("/etc/hostname"));
}
function getUBootVersion()
{
    return shell_exec(
        "LC_ALL=C tr -c '[ -~]' '\n' < /dev/mtd0 | grep 'U-Boot [0-9]' | head -n 1"
    );
}

function getInfoFirmware(&$firmware, &$build)
{
    $release = file_get_contents('/etc/spinetix-release');
    $vals = explode(" ", $release);
    if (count($vals) == 6) {
        $firmware = trim($vals[3]);
        $build = trim($vals[5]);
    } else {
        $firmware = "[not found]";
        $build = "";
    }
}

function getInfoFirmwareCorrupted()
{
    $file = "/var/lib/updater/status";
    if (file_exists($file)) {
        return trim(file_get_contents($file)) === "CORRUPTED";
    }
    return false;
}

function getInfoTemperature()
{
    $brandingInfo = new BrandingInfo();
    $hwTempPath = glob($brandingInfo->hwTempPath);
    $hwTempFileIndex = $brandingInfo->hwTempFileIndex;
    $ret = array( 'temp' => NAN, 'max' => false, 'crit' => false);
    $tempInputFile = "";
    $tempMaxAlarmFile = "";
    $tempCritAlarmFile = "";

    if ($hwTempPath && $hwTempFileIndex ) {
        $tempBase = $hwTempPath[0] . "/temp" . $hwTempFileIndex;
        $tempInputFile = $tempBase . "_input";
        $tempMaxAlarmFile = $tempBase .  "_max_alarm";
        $tempCritAlarmFile = $tempBase . "_crit_alarm";
    }

    if (($temp = @file_get_contents($tempInputFile))) {
        $ret['temp'] = intval($temp) / 1000;
    }

    if (($temp = @file_get_contents($tempMaxAlarmFile))) {
        if (trim($temp) === "1") {
            $ret['max'] = true;
        }
    }

    if (($temp = @file_get_contents($tempCritAlarmFile))) {
        if (trim($temp) === "1") {
            $ret['crit'] = true;
        }
    }
    return $ret;
}
function getStorageHealth()
{
    $file = "/var/lib/hwwatchdog/mmcblk0.info";
    $result = array();
    if (file_exists($file)) {
        $data = json_decode(file_get_contents($file), true);
        $result[] = $data;
    }
    return $result;
}
function getDiskInfo($path)
{
    $data = exec('df -B 1048576 -P ' . escapeshellarg($path));

    $spaces = preg_split("@\s+@", $data);

    $info = array();
    if (count($spaces) >= 4) {
        $info['free'] = $spaces[3];
        $info['used'] = $spaces[2];
        $info['tot'] = $spaces[1];
    } elseif (file_exists($path)) {
        $info['free'] = floor(disk_free_space($path) / 1024 / 1024);
        $info['tot'] = floor(disk_total_space($path) / 1024 / 1024);

        $info['used'] = $info['tot'] - $info['free'];
    } else {
        $info['free'] = "unknown";
        $info['used'] = "unknown";
        $info['tot'] = "unknown";
    }
    $info["unit"] = "Mbytes";
    return $info;
}

function getWizardStatus()
{
    if (file_exists("/etc/spxmanage")) {
        $file = "/etc/spxmanage/configured";
    } else {
        $file = "/etc/spinetix/configured";
    }

    if (!file_exists($file)) {
        return "unconfigured";
    }
    $config = file_get_contents($file);

    if (preg_match("/^\s*WIZARD=(.*)/m", $config, $matches)) {
        return $matches[1] != 0 ? "wizard" : "normal";
    }
    return "normal";
}

function getAryaStatus()
{
    $file = "/etc/spxmanage/arya-settings";
    if (!file_exists($file)) {
        return "unknown";
    }
    $config = file_get_contents($file);
    if (preg_match("/^\s*ENABLED=(.*)/m", $config, $matches)) {
        return trim($matches[1]);
    }

    return "unknown";
}

class ipconfigBasics
{
    const NET_ETHERNET = "ethernet";
    const NET_WIFI = "wlan";
    var $ifname = "";
    var $netiface = self::NET_ETHERNET;
    var $dhcp = 1;
    var $address = "";
    var $netmask = "";
    var $gateway = "";
    var $hostname = "";

    var $startTag = "# START-spxsysconfig DO NOT REMOVE THE START AND END TAGS";
    var $endTag = "# END-spxsysconfig";
    var $fname = "/etc/network/interfaces";
    function mask($adr, $msk)
    {
        $a = ip2long($adr);
        $m = ip2long($msk);
        return long2ip($a & $m);
    }
    function broad($adr, $msk)
    {
        $a = ip2long($adr);
        $m = ip2long($msk);

        return long2ip(($a & $m) + ~$m);
    }

    function load()
    {
        // get the config
        $interface = file_get_contents($this->fname);
        if (!isset($interface) && $interface) {
            return "Warning: cannot read configuration file";
        }

        $start = strpos($interface, $this->startTag);
        $end = strpos($interface, $this->endTag);
        if ($start === false || $end === false) {
            return "Warning: configuration file is incomplete";
        }

        // should look for first auto interface to be
        // consistent with ifwatchdog
        $str1 = "auto ";
        $iface = strpos($interface, $str1, $start);
        $iface += strlen($str1);
        $line_end = strpos($interface, "\n", $iface);
        $this->ifname = substr($interface, $iface, $line_end - $iface);
        if ($this->ifname == "wlan0") {
            $this->netiface = self::NET_WIFI;
        } else {
            $this->netiface = self::NET_ETHERNET;
        }

        $str1 = "iface " . $this->ifname . " inet ";
        $iface = strpos($interface, $str1, $start);
        $iface += strlen($str1);
        $conf = substr($interface, $iface, $end - $iface);
        $vals = explode("\n", $conf);

        if (trim($vals[0]) == "dhcp") {
            $this->dhcp = 1;
            foreach ($vals as $name) {
                if (!(($n = strpos($name, "hostname ")) === false)) {
                    $this->hostname = trim(substr($name, strlen("hostname ")));
                }
            }
        } elseif (trim($vals[0]) == "static") {
            $this->dhcp = 0;
            foreach ($vals as $name) {
                if (!(($n = strpos($name, "address ")) === false)) {
                    $this->address = trim(
                        substr($name, strlen("address") + $n)
                    );
                } elseif (!(($n = strpos($name, "netmask ")) === false)) {
                    $this->netmask = trim(
                        substr($name, strlen("netmask") + $n)
                    );
                } elseif (!(($n = strpos($name, "gateway ")) === false)) {
                    $this->gateway = trim(
                        substr($name, strlen("gateway") + $n)
                    );
                }
            }
        } elseif (trim($vals[0]) == "ppp") {
            // nothing else to read, we always use the same ppp config
        } else {
            return "Warning: unknown inet state: '" . $vals[0] . "'";
        }
        return true;
    }
}

class dnsconfigBasics
{
    var $nameserver1 = "";
    var $nameserver2 = "";
    var $nameserver3 = "";
    var $search = "";
    var $domain = "";
    var $fname = "/etc/network/resolv.conf.static";
    function load()
    {
        $dns = file_get_contents($this->fname);
        $tmp = $dns;
        $i = 1;
        $pos = strpos($tmp, "nameserver");
        while (!($pos === false)) {
            $pos += strlen("nameserver");
            $name = "nameserver" . $i;
            $i++;
            $this->$name = trim(
                substr($tmp, $pos, strpos($tmp, "\n", $pos) - $pos)
            );
            $tmp = substr($tmp, strpos($tmp, "\n", $pos));
            $pos = strpos($tmp, "nameserver");
        }

        if (!($pos === false)) {
            $pos = strpos($dns, "search ");
        }
        if (!($pos === false)) {
            $pos += strlen("search ");
            $this->search = trim(
                substr($dns, $pos, strpos($dns, "\n", $pos) - $pos)
            );
        }
        $pos = strpos($dns, "domain ");
        if (!($pos === false)) {
            $pos += strlen("domain ");
            $this->domain = trim(
                substr($dns, $pos, strpos($dns, "\n", $pos) - $pos)
            );
        }
    }
}

class ScreenInfoDRM
{
    var $screens = array();

    public function getDisplayNames()
    {
        $file = "/usr/share/resources/default/interface/output-display.json";
        if (!file_exists($file)) {
            return array();
        }

        return json_decode(file_get_contents($file), true);
    }

    public static function selectModel($names)
    {
        $serial = "";
        $model = "";
        $hardware = "";
        getInfoSerial($serial, $model, $hardware);
        $model = explode("|", $model);
        $model = count($model) >= 2 ? $model[1] : $model[0];

        $selectorMap = array();
        foreach ($names as $prefix => $selector) {
            if (preg_match($prefix, $model)) {
                $selectorMap = $selector;
            }
        }

        return $selectorMap;
    }

    public static function getAudioInfo($surround)
    {
        $file = "/usr/share/resources/default/interface/output-audio.json";
        if (file_exists($file)) {
            $audioNames = json_decode(file_get_contents($file), true);
            $audioSelectorMap = ScreenInfoDRM::selectModel($audioNames);
        } else {
            $audioSelectorMap = array();
        }

        if ($surround) {
            $extendedAudioSelectorMap = array();
            foreach ($audioSelectorMap as $id => $name) {
                $extendedAudioSelectorMap[$id] = $name;
                if (strpos($id, "hdmi-stereo") !== false) {
                    $extendedAudioSelectorMap[
                        str_replace("stereo", "surround", $id)
                    ] = $name . " Surround 5.1";
                    $extendedAudioSelectorMap[
                        str_replace("stereo", "surround71", $id)
                    ] = $name . " Surround 7.1";
                }
            }
            $audioSelectorMap = $extendedAudioSelectorMap;
        }
        $output = array();
        $ret = 10;
        exec("spxaudiocfg -l -j", $output, $ret);
        if ($ret) {
            syslog(
                LOG_NOTICE,
                'no audio detected: spxaudiocfg returned ' . $ret
            );
            return array();
        }
        $info = json_decode(implode("\n", $output), true);
        if (!$info) {
            syslog(
                LOG_NOTICE,
                'no audio detected: file not json: ' . implode("\n", $output)
            );
            return array();
        }
        $audioInfo = array();
        if (count($audioSelectorMap) === 0) {
            foreach ($info as &$audio) {
                if (!isset($audio["id"]) || !isset($audio["desc"])) {
                    continue;
                }
                $audio['name'] = $audio["desc"];
            }
        } else {
            foreach ($info as &$audio) {
                if (!isset($audio["id"]) || !isset($audio["desc"])) {
                    continue;
                }
                $audioId = ScreenInfoDRM::findAudioId($audioSelectorMap, $audio["id"]);
                if (substr($audioId, 0, 13) === "alsa_card.usb") {
                    if (!$audio["default"]) {
                        continue;
                    }
                    if (!isset($audioSelectorMap[$audioId])) {
                        $audio['name'] = $audio["desc"];
                    } else {
                        $audio['name'] = ScreenInfoDRM::getAudioName($audioSelectorMap, $audioId);
                    }
                } elseif (substr($audioId, 0, 13) === "alsa_card.pci") {
                    if (!isset($audioSelectorMap[$audioId])) {
                        continue;
                    }
                    $audio['name'] = ScreenInfoDRM::getAudioName($audioSelectorMap, $audioId);
                } else {
                    $audio['name'] = $audio["desc"];
                }
            }
            // Alway add the audio from the branding
            foreach ($audioSelectorMap as $id => $name) {
                $matchingAudioOutputs = array_filter($info, function($element) use ($id) {
                    return strpos($element["id"], $id) === 0;
                });
                if (empty($matchingAudioOutputs)) {
                    if (is_array($name)) {
                        $info[] = array(
                            "id" => $id,
                            "name" => $name['name'],
                            "desc" => isset($name['desc'])
                                ? $name['desc']
                                : $name['name'],
                        );
                    } else {
                        $info[] = array(
                            "id" => $id,
                            "name" => $name,
                            "desc" => $name,
                        );
                    }
                }
            }
        }
        return $info;
    }

    public function loadSimple()
    {
        $dir = '/sys/class/drm';
        if (!file_exists($dir)) {
            return;
        }
        $files = scandir($dir);
        $this->screens = array();

        $type2Name = array(
            "DP" => "DisplayPort / HDMI",
            "HDMI-A" => "HDMI",
        );

        foreach ($files as $file) {
            if (
                !is_dir($dir . "/" . $file) ||
                !preg_match("/^card\d+-([-\w]+)-\d+$/", $file, $matches)
            ) {
                continue;
            }
            $screen = array("name" => $file);

            $screen["type"] = $matches[1];
            $screen["status"] = trim(
                @file_get_contents($dir . "/" . $file . "/status")
            );
            $screen["enabled"] =
                trim(@file_get_contents($dir . "/" . $file . "/enabled")) ===
                "enabled";

            $modesStr = @file_get_contents($dir . "/" . $file . "/modes");
            $screen["modesStr"] = $modesStr;
            if ($modesStr !== "") {
                $modes = explode("\n", trim($modesStr));
                $res2 = array();
                foreach ($modes as $mode) {
                    if (!strpos($mode, "i")) {
                        $res2[$mode] = true;
                    }
                }
                $modes = array_keys($res2);
                $screen["modes"] = $modes;
            }
            if (isset($type2Name[$screen['type']])) {
                $screen['displayName'] = $type2Name[$screen['type']];
            } else {
                $screen['displayName'] = $screen['type'];
            }
            $this->screens[] = $screen;
        }
    }
    public function load()
    {
        $this->loadSimple();
        $existingName = array();

        $displayNames = $this->getDisplayNames();
        $outputSelectorMap = ScreenInfoDRM::selectModel($displayNames);

        foreach ($this->screens as $i => &$screen) {
            $displayName = $screen['displayName'];
            if (isset($outputSelectorMap[$screen['name']])) {
                $displayName = $outputSelectorMap[$screen['name']];
            }
            if ($displayName === false) {
                $screen['displayName'] = $displayName;
                continue;
            }

            $screen['displayName'] = $displayName;
            $existingName[$displayName][] = $i;
        }
        foreach ($existingName as $name => $indices) {
            if (count($indices) > 1) {
                $idx = 1;
                foreach ($indices as $i) {
                    $this->screens[$i]['displayName'] = $name  . " - " . $idx++;
                }
            }
        }
    }

    private static function findAudioId($audioSelectorMap, $configId) {
        foreach (array_keys($audioSelectorMap) as $brandingId) {
            if (strpos($configId, $brandingId) === 0) {
                return $brandingId;
            }
        }
        return $configId;
    }

    private static function getAudioName($audioSelectorMap, $audioId) {
        if (is_array($audioSelectorMap[$audioId])) {
            return $audioSelectorMap[$audioId]['name'];
        } else {
            return $audioSelectorMap[$audioId];
        }
    }
}

class ScreenInfoBasics
{
    var $width = "unknown";
    var $height = "unknown";
    var $resolution = "unknown";
    var $refresh = "unknown";
    var $aspect = "unknown";
    var $monitors = array();
    var $powersave = "off";
    protected $ikebanDisplayInfoFile = "/var/run/raperca/display-info";

    protected $timingsFile = '/sys/devices/platform/vpss/display0/timings';
    # Returns the location of the video out sysfs, with a trailing slash
    static function sysfs_vout_device()
    {
        if (file_exists('/sys/devices/platform/vpss/display0')) {
            return '/sys/devices/platform/vpss/display0/';
        } elseif (file_exists('/sys/class/x-display/disp0/device')) {
            return '/sys/class/x-display/disp0/device/';
        } else {
            return '/sys/class/graphics/fb0/device/';
        }
    }

    static function dbstr2array($str)
    {
        $res = array();
        if (preg_match("/(\S*)\s*=\s*(.*\s+p)\s+(\S+)/", $str, $match)) {
            $desc = $match[1];
            $values = explode(" ", $match[2]);
            if (count($values) >= 11) {
                // only progresiye settings
                $res['hz'] = intval($values[2]);
                $res['resolution'] = $values[0] . "x" . $values[1];
                $res['width'] = intval($values[0]);
                $res['height'] = intval($values[1]);
                $res['ar'] = $match[3];
                $res['pcf'] =
                    ($res['hz'] *
                        ($res['width'] +
                            intval($values[3]) +
                            intval($values[4]) +
                            intval($values[7])) *
                        ($res['height'] +
                            intval($values[5]) +
                            intval($values[6]) +
                            intval($values[7]))) /
                    1000000;
                $res['settings'] = $match[2];
                return $res;
            }
            return false;
        }
        return false;
    }

    public function load()
    {
        $dir = self::sysfs_vout_device();

        if (file_exists($dir . "vid_mode")) {
            $vidmod = file_get_contents($dir . "vid_mode");
            $sets = explode("=", $vidmod);
            if (count($sets) == 2) {
                $vals = explode(" ", trim($sets[1]));
                if (count($vals) >= 11) {
                    if ($vals[11] == "0:0") {
                        $vals[11] = "[auto]";
                    }
                    if ($vals[11] == "x:x") {
                        $vals[11] = "[unknown]";
                    }
                    $this->width = htmlspecialchars(trim($vals[0]));
                    $this->height = htmlspecialchars(trim($vals[1]));
                    $this->resolution = $this->width . "x" . $this->height;
                    $this->refresh = htmlspecialchars($vals[2]);
                    $this->aspect = htmlspecialchars($vals[11]);
                }
            }
        } elseif (file_exists($this->timingsFile)) {
            $timings = file_get_contents($this->timingsFile);
            $sets = explode(",", trim($timings));
            if (count($sets) == 4) {
                $horizontal = explode("/", trim($sets[1]));
                $this->width = htmlspecialchars(trim($horizontal[0]));
                $realwidth = array_sum($horizontal);
                $vertical = explode("/", trim($sets[2]));
                $this->height = htmlspecialchars(trim($vertical[0]));
                $realheight = array_sum($vertical);
                $this->refresh =
                    round(($sets[0] / $realheight / $realwidth) * 1000 * 100) /
                    100;

                $this->resolution = $this->width . "x" . $this->height;
                if ($this->height / $this->width == 0.5625) {
                    $this->aspect = "16:9";
                } else {
                    $this->aspect = "unknown";
                }
            }
        }

        if (file_exists($dir . "dpm_state")) {
            $dpm_state = file_get_contents($dir . "dpm_state");
            $this->powersave = $dpm_state[0] == "0" ? "on" : "off";
        }

        if (file_exists($dir)) {
            $content = scandir($dir);
            foreach ($content as $path) {
                if (
                    strncmp($path, "out", 3) == 0 &&
                    file_exists("$dir$path/name")
                ) {
                    $data = array(
                        "mon_powered",
                        "mon_manufacturer",
                        "mon_prod_id",
                        "mon_name",
                        "mon_serial_no",
                        "mon_width",
                        "mon_height",
                        "mon_underscan",
                        "mon_audio",
                        "mon_pict_ar",
                        "link_type",
                    );
                    if (file_exists("$dir$path/name")) {
                        $type = trim(file_get_contents("$dir$path/name"));
                    } else {
                        $type = "unknonw";
                    }
                    if (file_exists("$dir$path/mon_present")) {
                        $mon_present = file_get_contents(
                            "$dir$path/mon_present"
                        );
                    } else {
                        $mon_present = 0;
                    }
                    $scinfo = array('type' => $type);
                    if ($mon_present == 1) {
                        $result = array();
                        foreach ($data as $d) {
                            $file = $dir . $path . "/$d";
                            if (file_exists($file)) {
                                $result[$d] = trim(file_get_contents($file));
                            } else {
                                $result[$d] = "";
                            }
                        }
                        if (
                            $result['mon_manufacturer'] != "" ||
                            $result['mon_name'] != ""
                        ) {
                            $scinfo['manufacturer'] =
                                $result['mon_manufacturer'];
                            $scinfo['productID'] = $result['mon_prod_id'];
                            $scinfo['name'] = $result['mon_name'];
                            $scinfo['serial'] = $result['mon_serial_no'];
                        }
                        if ($result['mon_powered'] != "") {
                            $scinfo['powered'] =
                                $result['mon_powered'] == 0 ? "off" : "on";
                        } else {
                            $scinfo['powered'] = "unknown";
                        }

                        if (
                            $result['mon_pict_ar'] == "" ||
                            $result['mon_pict_ar'] == "x:x"
                        ) {
                            $result['mon_pict_ar'] = "unknown";
                        }
                        if (
                            $result['mon_width'] != "" &&
                            $result['mon_height'] != ""
                        ) {
                            $scinfo['width'] = $result['mon_width'];
                            $scinfo['height'] = $result['mon_height'];
                            $scinfo['aspectRatio'] = $result['mon_pict_ar'];
                        }
                        $scinfo['link'] = $result['link_type'];

                        if ($result['mon_underscan'] === "0") {
                            $scinfo['underscan'] = "unsupported";
                        } else {
                            $scinfo['underscan'] = "supported";
                        }
                        $scinfo['pm_supported'] = 1;
                    }
                    $this->monitors[] = $scinfo;
                }
            }
        }
        if (
            !count($this->monitors) &&
            file_exists($this->ikebanDisplayInfoFile)
        ) {
            $desc = json_decode(
                file_get_contents($this->ikebanDisplayInfoFile),
                true
            );
            foreach ($desc as $result) {
                $this->powersave =
                    $result['mon_power_save'] == 0 ? "off" : "on";

                $scinfo = array(
                    'id' => $result['name'],
                    'type' => $result['link_type'],
                );
                $mon_present = $result['mon_present'];
                if ($mon_present) {
                    if (isset($result['mon_powered'])) {
                        $scinfo['powered'] =
                            $result['mon_powered'] == 0 ? "off" : "on";
                    } else {
                        $scinfo['powered'] = "unknown";
                    }
                    if (isset($result['mon_pm_supported'])) {
                        $scinfo['pm_supported'] = $result['mon_pm_supported'];
                        if (isset($result['cec_capable'])) {
                            $scinfo['cec_capable'] = $result['cec_capable'];
                        }
                    } else {
                        $scinfo['pm_supported'] = 1;
                    }
                }
                $this->monitors[] = $scinfo;
            }
        }
        if (!count($this->monitors)) {
            $screeninfo = new ScreenInfoDRM();
            $screeninfo->load();

            foreach ($screeninfo->screens as $screen) {
                $displayName = $screen['displayName'];
                if ($displayName === false) {
                    continue;
                }

                $scinfo = array(
                    'id' => $screen['name'],
                    'type' => $screen['displayName'],
                    'name' => $screen['type'],
                );
                if ($screen["status"] === "connected") {
                    $scinfo['powered'] = "unknown";
                    $scinfo['pm_supported'] = 0; // no power mode for now, this may be added in the future.
                }

                $this->monitors[] = $scinfo;
            }
        }
    }
}

function net_mask_to_prefix($netmask)
{
    $length = 0;
    $vals = explode('.', $netmask);
    if (count($vals) != 4) {
        return false;
    }
    $expect_zero = false;
    foreach ($vals as $val) {
        if (!is_numeric($val) || $val < 0 || $val > 255) {
            return false;
        }
        if ($expect_zero) {
            if ($val != 0) {
                return false;
            } else {
                continue;
            }
        }
        switch ($val) {
            case 255:
                $length += 8;
                break;
            case 254:
                $length += 7;
                break;
            case 252:
                $length += 6;
                break;
            case 248:
                $length += 5;
                break;
            case 240:
                $length += 4;
                break;
            case 224:
                $length += 3;
                break;
            case 192:
                $length += 2;
                break;
            case 128:
                $length += 1;
                break;
            case 0:
                $length += 0;
                break;
            default:
                return false;
        }
        if ($val != 255) {
            $expect_zero = true;
        }
    }
    return $length;
}
class Parselogfile
{
    protected $file;
    protected $leftover = "";
    protected $lines = array();

    function open($logFile)
    {
        if (file_exists($logFile)) {
            $this->file = fopen($logFile, 'r');
            // go to the end of the file
            fseek($this->file, 0, SEEK_END);
        }
    }
    function close()
    {
        if ($this->file) {
            fclose($this->file);
        }
    }
    function readLine($targets = array(), $blockSize = 1024)
    {
        if (!$this->file) {
            return false;
        }
        $data = "";

        do {
            // need to know whether we can actually go back
            // $block_size bytes
            $canRead = $blockSize;
            if (ftell($this->file) < $blockSize) {
                $canRead = ftell($this->file);
            }

            // go back as many bytes as we can
            // read them to $data and then move the file pointer
            // back to where we were.
            fseek($this->file, -$canRead, SEEK_CUR);
            $data = fread($this->file, $canRead);
            $data .= $this->leftover;
            fseek($this->file, -$canRead, SEEK_CUR);

            // split lines by \n. Then reverse them,
            // now the last line is most likely not a complete
            // line which is why we do not directly add it, but
            // append it to the data read the next time.
            $splitData = array_reverse(explode("\n", $data));
            if ($canRead == $blockSize) {
                $this->leftover = $splitData[count($splitData) - 1];
                $newLines = array_slice($splitData, 0, -1);
            } else {
                $newLines[] = $this->leftover;
                $this->leftover = "";
            }
            $this->lines = array_merge($this->lines, $newLines);

            //print_r( $targets );
            // check if the lines contains the target
            foreach ($this->lines as $idx => $line) {
                unset($this->lines[$idx]);
                if (count($targets) == 0) {
                    return $line;
                } else {
                    foreach ($targets as $target) {
                        if (strpos($line, $target) !== false) {
                            return $line;
                        }
                    }
                }
            }
        } while ($canRead == $blockSize);

        return false;
    }
}

class ParseSyslog extends Parselogfile
{
    protected $logFile = "/var/log/syslog";

    var $readSocType = false;
    function open($logFile = null)
    {
        parent::open($this->logFile);
    }

    function readReboot()
    {
        $reason = "";
        $details = "";

        $SOCType = "SoC reset type";
        $watchdog = "shutting down the system because of error";
        $user = "init: Switching to runlevel:";

        $content = "file /var/run/raperca/progress was not changed";
        $crash = "pinging process";
        $exit = "cannot open /var/run";

        $interface = "user initiated shutdown";
        $button = "spx-gpio-btn: starting shutdown from push button";
        $updater = "rebooting for upgrades to take full effect";
        $safemode = "raperca-safe-mode: safe mode rebooting automatically";

        if ($this->readSocType === false) {
            $resetType = $this->readLine(array($SOCType));
        } else {
            $resetType = $this->readSocType;
            $this->readSocType = false;
        }
        $typeMatch = array();
        if (
            $resetType !== false &&
            preg_match(
                "/SoC reset type\s+0x[0-9a-fA-F]+\s+\((\w+).*-\s+(\w+)/",
                $resetType,
                $typeMatch
            )
        ) {
            // we found a reason
            $reason = $typeMatch[1];
            $details = $typeMatch[2];
        }

        if ($reason == "") {
            // we should open the rotated backup, but for the time being, we just say we don't know
            return "unknown";
        }
        if ($details == "watchdog") {
            return "crash";
        }
        // find the cause of the reboot
        $reasonLine = $this->readLine(array(
            $SOCType,
            $user,
            $watchdog,
            $interface,
        ));
        if ($reasonLine === false || strpos($reasonLine, $SOCType) !== false) {
            $this->readSocType = $reasonLine;
            return $reason; // no more info
        }

        if (strpos($reasonLine, $watchdog) !== false) {
            // reboot because of the watchdog
            $watchdogLine = $this->readLine(array(
                $content,
                $crash,
                $exit,
                $SOCType,
            ));
            if (
                $watchdogLine === false ||
                strpos($watchdogLine, $SOCType) !== false
            ) {
                $this->readSocType = $watchdogLine;
                return $reason; // no more info
            }

            if (strpos($watchdogLine, "hwwatchdog") !== false) {
                // the hwwatchdog is gone, this normally means that the temperature is too high
                // we may go further to check using the hwwatchdog logs
                return "temperature";
            } elseif (
                strpos($watchdogLine, $crash) !== false ||
                strpos($watchdogLine, $exit) !== false
            ) {
                return "crash";
            } elseif (strpos($watchdogLine, $content) !== false) {
                return "content";
            } else {
                return $reason;
            }
        } elseif (strpos($reasonLine, $user) !== false) {
            // reboot following user action
            $userLine = $this->readLine(array(
                $interface,
                $button,
                $updater,
                $safemode,
                $SOCType,
            ));
            if ($userLine === false || strpos($userLine, $SOCType) !== false) {
                $this->readSocType = $userLine;
                return $reason; // no more info
            }

            if (strpos($userLine, $updater) !== false) {
                return "updater";
            } elseif (strpos($userLine, $button) !== false) {
                return "button";
            } elseif (strpos($userLine, $safemode) !== false) {
                return "safemode";
            } else {
                if (strpos($userLine, "rpc") !== false) {
                    return "rpc";
                } else {
                    return "interface";
                }
            }
        } elseif (strpos($reasonLine, $interface) !== false) {
            if (strpos($reasonLine, "rpc") !== false) {
                return "rpc";
            } else {
                return "interface";
            }
        }

        return $reason; // no more info
    }
    static function getRebootReason()
    {
        $syslog = new ParseSyslog();
        $syslog->open();
        $reason = $syslog->readReboot();
        $syslog->close();
        return $reason;
    }
}

class ParseLogs extends Parselogfile
{
    protected $logFile = "/srv/raperca/log/player.log";
    protected $syslogFile = "/var/log/syslog";

    protected $logger;

    function open($logger)
    {
        if ($logger === "temperature") {
            parent::open($this->syslogFile);
            $this->logger = $logger;
        } else {
            parent::open($this->logFile);
            $this->logger = $logger;
        }
    }
    function readLines($time, $blockSize = 512)
    {
        $lastTime = strtotime(date("d-m-Y H:i:s"));
        if ($time !== true) {
            if (substr($time, 0, 1) == "-") {
                $offset = explode(":", substr($time, 1));

                $mintime = $lastTime - 3600 * floatval($offset[0]);
                if (count($offset) > 1) {
                    $mintime -= 60 * floatval($offset[1]);
                }
                if (count($offset) > 2) {
                    $mintime -= floatval($offset[2]);
                }
            } else {
                $mintime = strtotime($time);
            }
        }

        $retLines = array();
        do {
            $line = parent::readLine(array($this->logger), $blockSize);

            if ($line === false) {
                return $retLines;
            }
            if ($time === true) {
                $retLines[] = $line;
                break;
            } else {
                $date = explode(" ", $line, 4);
                if (strlen($date[0]) <= 6) {
                    // syslog format for date
                    $lineTime = strtotime($date[2]);
                }
                // log4cxx format for date
                else {
                    $lineTime = strtotime($date[1]);
                }
                if ($lineTime > $lastTime) {
                    break;
                } // time is going backward

                if ($lineTime < $mintime) {
                    break;
                } else {
                    $retLines[] = $line;
                }

                $lastTime = $lineTime;
            }
        } while (count($retLines) < 100);

        return $retLines;
    }

    function readStats($time = false)
    {
        $this->open("Period: 60s");
        $lines = $this->readLines($time);
        $stats = array();

        foreach ($lines as $line) {
            $pattern =
                "/(\S+\s\S+)\s+INFO\s+\S+\s+-\s+(\d+:\d+:\d+) " .
                "Period: 60s Pic:\s*(\d+) Usage:\s+([\d\.]+)% \(([\d\s\|\.)]*)\) " .
                "Peak:\s+(\d+)ms Buffers:\s*([\d\.]+)ms \(min\s*(\d+)ms\) " .
                "FPS Drop:\s*(\d+)% Peak drop:\s*(\d+)%/";
            $matches = array();
            if (preg_match($pattern, $line, $matches)) {
                $date = explode("-", $matches[1]);
                $time =
                    substr($date[2], 0, 4) .
                    "-" .
                    $date[1] .
                    "-" .
                    $date[0] .
                    "T" .
                    substr($date[2], 5);
                $cpu = explode("|", $matches[5]);
                $result = array(
                    'time' => $time,
                    'render' => $matches[2],
                    'pictures' => intval($matches[3]),
                    'usage' => array(
                        'average' => intval($matches[4]),
                        'arm' => floatval($cpu[0]),
                        'dsp' => floatval($cpu[1]),
                        'max' => array(0, 0, 0),
                    ),
                    'peak' => intval($matches[6]),
                    'buffers' => array(
                        'average' => floatval($matches[7]),
                        'min' => intval($matches[8]),
                    ),
                    'dropFPS' => array(
                        'average' => intval($matches[9]),
                        'max' => array(0, 0, 0),
                    ),
                    'dropPeak' => array(
                        'average' => intval($matches[10]),
                        'max' => array(0, 0, 0),
                    ),
                );
                $stats[] = $result;
            }
        }
        $this->close();
        return $stats;
    }
    function readErrors($time = false)
    {
        $this->open("spx.content");
        $lines = $this->readLines($time);
        $stats = array();
        $names = array(
            "101" => "JavaScript Exception",
            "102" => "Local file not found",
            "103" => "Rendering error",
            "104" => "Media out of specs",
            "105" => "Parsing/Decoding error",
            "106" => "Network error",
            "999" => "Others",
        );
        if (!file_exists('/srv/raperca/interface')) {
            $filterOut["File not found : '/srv/raperca/interface/public/index.svg'"] = true;
        }
        foreach ($lines as $line) {
            $pattern = "/(\S+\s\S+)\s+WARN\s+\S+\s+-\s*(\d+)\s(\S*)\s(.*)/";
            $matches = array();
            if (preg_match($pattern, $line, $matches)) {
                $message = trim($matches[4]);
                if (isset($filterOut[$message])) {
                    continue;
                }
                $date = explode("-", $matches[1]);
                $time =
                    substr($date[2], 0, 4) .
                    "-" .
                    $date[1] .
                    "-" .
                    $date[0] .
                    "T" .
                    substr($date[2], 5);
                $result = array(
                    'time' => $time,
                    'code' => intval($matches[2]),
                    'source' => $matches[3],
                    'custom' => $message,
                );
                if (isset($names[$matches[2]])) {
                    $result['desc'] = $names[$matches[2]];
                }
                $stats[] = $result;
            }
        }
        $this->close();
        return $stats;
    }

    function readTemperature($time = false)
    {
        $this->open("temperature");
        $lines = $this->readLines($time);
        $temps = array();
        foreach ($lines as $line) {
            $pattern =
                "/(\S+\s\S+\s\S+)\s+\S+\shwwatchdog\[\d+\]:\s+temperature\s+=\s+([\d\.]+)/";
            $matches = array();
            if (preg_match($pattern, $line, $matches)) {
                $timestamp = strtotime($matches[1]);
                $time = date("Y-m-d\TH:i:s", $timestamp);
                $result = array(
                    'time' => $time,
                    'C' => floatval($matches[2]),
                    'F' => (9 / 5) * floatval($matches[2]) + 32,
                );
                $temps[] = $result;
            }
        }
        $this->close();
        return $temps;
    }
}

function hasPowerMCU() {
    return file_exists('/run/spxucd/spxucd.pid');
}

function getPowerInfo() {
    $source = '[undefined]';
    $power = 0;
    if (hasPowerMCU()) {
        try {
            $dbus = new Dbus(Dbus::BUS_SYSTEM, false);
            $spxucd = $dbus->createProxy(
                'com.spinetix.spxucd', '/com/spinetix/SpxPlayer/PowerMCU', 
                'org.freedesktop.DBus.Properties');
            $powerInfo = $spxucd->GetAll(
                'com.spinetix.SpxPlayer.PowerMCU1.MainPower')->getData();
            //syslog(LOG_DEBUG, 'powerInfo: ' . print_r($powerInfo, true));
            $source = 0;
            if ($powerInfo['HasPoE']->getData()) {
                $source |= 1;
            }
            if ($powerInfo['HasPD']->getData()) {
                $source |= 2;
            }
            //syslog(LOG_DEBUG, 'Power source: ' . print_r($source, true));
            $power = round($powerInfo['Voltage']->getData() * 
                $powerInfo['Current']->getData(), 1);
            //syslog(LOG_DEBUG, 'Power: ' . print_r($power, true));
        } catch (Exception $e) {
            syslog(LOG_ERR, $e->getMessage());
        }
    }
    return array('source' => $source, 'power' => $power);
}

function getPowerSource($powerInfo) {
    switch ($powerInfo['source']) {
        case 1:
            return 'poe';
        case 2:
            return 'usb';
        case 3:
            return 'poe+usb';
        default:
            return '[undefined]';
    }
}

function getWiredAuthInfo()
{
    if (!file_exists('/var/lib/ead/default.8021x')) {
        return array();
    }

    $interfaces = array();

    try {
        $dbus = new Dbus(Dbus::BUS_SYSTEM, false);

        $ead = $dbus->createProxy(
            'net.connman.ead',
            '/',
            'org.freedesktop.DBus.ObjectManager'
        );

        foreach ($ead->GetManagedObjects()->getData() as $path => $object) {
            //syslog(LOG_DEBUG, $path . ": " . print_r($object, true));

            if (strpos($path, '/net/connman/ead/adapter/') === 0) {
                $objectInterfaces = $object->getData();
                $properties =
                    $objectInterfaces['net.connman.ead.Adapter']->getData();

                $interfaces[$properties['Name']->getData()] = array(
                    'address' => strtoupper($properties['Address']->getData()),
                    'connected' => $properties['Connected']->getData(),
                    'authenticated' => $properties['Authenticated']->getData()
                );
            }
        }
    } catch (Exception $e) {
        syslog(LOG_ERR, $e->getMessage());
    }

    //syslog(LOG_DEBUG, print_r($interfaces, true));

    return $interfaces;
}
