<?php

class ServerCertificates extends CFormModel {
	
    static $certFolder = "/etc/pki/spxmanage/server/";
    static $keyFolder =  "/etc/pki/spxmanage/private/";
    static $activeLink = "/etc/apache2/server.crt";
    static $activeKey = "/etc/apache2/server.key";
    static $activeLinkFolder ="/etc/pki/spxmanage/certs/";
    
    static $rootCertificateBundle = "/etc/ssl/certs/ca-certificates.crt";
    
    static $autoCertificateId = "_auto_self_signed";
    static $autoCertificateCert = "httpd-auto-ss.crt";
    static $autoCertificateKey = "httpd-auto-ss.key";
    
    static $prefix = "http-imported-";
    static $certName = "{id}.crt";
    static $keyName = "{id}.key";
    static $chainName = "{id}-trust-{idx}.crt";
    static $chainPemName = "{id}-trust.pem";
    
    static $certExt = ".crt";
    
    
    // Upload type,
    var $type;
    var $typeValues = array( 
        'pem' => 'PEM files', 
        'pfx' => 'PCKS#12 file'
    );
    
    // PCKS#12 file (.pfx or .p12) plus its encryption passphrase. The PCKS#12
    // file contains all the elements and can be converted to PEM format by OpenSSL
    // utilities.
	var $pfxFile;
    
    // The certificate itself
    var $certificateFile; 
    // The certificates private key
    var $keyFile; 
    // The bundle of certificates in the certificate chain, that is all the 
    // certificates in the chain from the certificate itself up to the root CA
    var $chainFile; 
    
    // for backup text is used instead of files
    var $pfxText;
    var $certificateText; 
    var $keyText; 
    var $chainText; 
    var $active = false ;
    
    var $passphrase; 
    
    var $uid;
    
    private $exe;
    public function init()
	{
        parent::init();
        $this->uid = uniqid();
        
        $this->exe = "openssl";
        if ( PHP_OS == "WINNT" && file_exists( "c:\OpenSSL-Win32\bin\openssl.exe" ) ) {
            $this->exe = "c:\OpenSSL-Win32\bin\openssl";
        }
	}
	public function rules() {
		$res = array(
            array( 'uid', 'match', 'pattern' => "/^[0-9a-zA-Z_.@]{4,32}$/" ),            
            array( 'type', 'in', 'range'=>array('pfx','pem'),'allowEmpty'=>false, 'except'=>'backup,delete,activate'),            
            array( 'pfxFile', 'file', 'types'=>'pfx,p12', 'allowEmpty'=> false, 'on'=>'pfx' ),
            array( 'certificateFile', 'file', 'types'=>'pem,crt,cer', 'allowEmpty'=> false, 'on'=>'pem' ),
            array( 'keyFile', 'file', 'types'=>'pem,key', 'allowEmpty'=> false, 'on'=>'pem' ),
            array( 'chainFile', 'file', 'types'=>'pem', 'allowEmpty'=> true, 'on'=>'pem' ),
            array( 'passphrase', 'safe' ),
            array( 'uid', 'validateUID', 'on'=>'delete,activate,backup' ),
            array( 'pfxText,certificateText,keyText,chainText, active', 'safe', 'on'=>'backup' ),
            
    	);
        return $res;
	}
    public function attributeLabels()
	{
		return array(
            'uid'=>'Name',
			'type'=>'Certificate format',
            'pfxFile'=>'PCKS#12 file (PFX, P12)',
            'certificateFile'=>'Certificate (PEM, CRT)',
            'keyFile'=>'Certificate private key (PEM, KEY)',
            'chainFile'=>'Certificate chain (PEM)',
            'passphrase' => "Passphrase (optional)"
		);
	}
    
    public function validateUID( ){
        if (!empty($this->keyText ) || !empty($this->pfxText)){
            // new certificated, no additinal check to be done
            return true; 
        }
        if ( $this->uid == self::$autoCertificateId ){
            return true;
        }
        $certName = strtr( self::$certFolder . self::$prefix. self::$certName, array( "{id}" => $this->uid ));        		
		if ( !file_exists( $certName ) ){
            $this->addError('uid', 'Certificate not found');
			return false;
		}
        return true;
        
    }
    
    public function handleFiles( $runValidation=true, $attributes=null )
    {
        if( $runValidation && !$this->validate(array('type'))){
            return false;
        }
        $this->setScenario( $this->type );        
        if ( $this->scenario == 'pfx'){
            $this->pfxFile = CUploadedFile::getInstance($this,'pfxFile');  
            return $this->savePCKSFile($runValidation, $attributes);
        } else {
            $this->certificateFile = CUploadedFile::getInstance($this,'certificateFile');  
            $this->keyFile = CUploadedFile::getInstance($this,'keyFile');
            $this->chainFile = CUploadedFile::getInstance($this,'chainFile');
            
            return $this->saveFiles($runValidation, $attributes);
        }
        
    }
    public function saveCertificate( $cert, $attribute )
    {
        // Check that this is an X.509 certificate
		if ( !preg_match("/^-----BEGIN (X509 |TRUSTED |)CERTIFICATE-----/", $cert ) ) {
			$this->addError($attribute, 'Only X509 certificates are supported');
			return false;
		}
        $resource = openssl_pkey_get_public( $cert );
        $data = openssl_pkey_get_details ( $resource );
        if ( $data['type'] !== OPENSSL_KEYTYPE_RSA ){
            $this->addError($attribute, 'Only certificates using RSA are supported');
			return false;
        }
        if ( $data['bits'] < 1024 ){
            $this->addError($attribute, 'Only certificates using RSA with at least 1024 bits are supported');
			return false;
        }
        $certName = strtr( self::$certFolder . self::$prefix. self::$certName, array( "{id}" => $this->uid ));        
        file_put_contents(  $certName, $cert );        
        
        
        exec(escapeshellarg($this->exe) . " pkcs12 -export -passout pass: -out /dev/null -nokeys -in " . escapeshellarg($certName), $output, $ret);
        if ( $ret!=0 ) {
			// something wrong with the certificate
			$this->addError($attribute, 'Certificate not valid');
			unlink( $certName );
			return false;
		}
            
		// Recover the subject hash and certificate fingerprint 
		exec(escapeshellarg($this->exe) . " x509 -noout -hash -fingerprint -in " . escapeshellarg($certName), $output, $ret);
		if ( $ret!=0 ) {
			// something wrong
			$this->addError($attribute, 'Certificate not valid');
			unlink( $certName );
			return false;
		}
		/*
		$md5 = $output[1];
        
		$files = scandir( self::$certFolder );        
        foreach ($files as $file) {
            $matches = array();
			if ( !preg_match("|^".strtr( self::$prefix. self::$certName, array( "." => "\\.", "{id}" => "([0-9a-zA-Z_.@]{4,32})" ))."$|", $file, $matches) ){
                continue;
            }
            if ( $matches[1] == $this->uid ){
                continue;
            }
            $output2 = array();
            exec(escapeshellarg($this->exe) . " x509 -noout -hash -fingerprint -in " . escapeshellarg(self::$certFolder.$file), $output2, $ret);            
			if ( $output2[1]==$md5 ){
				// this is the same certificate
				$this->addError($attribute, 'Certificate already uploaded');
                unlink( $certName );
				return false;
			} 
		}
		*/
        
        return $certName;
    }
    public function saveKey( $key, $attribute )
    {
        // Check that this is an X.509 key
		if ( !preg_match("/^-----BEGIN (RSA |ENCRYPTED |)PRIVATE KEY-----/", $key ) ) {
			$this->addError($attribute, 'Only X509 keys are supported');
			return false;
		}
        
        $resource = openssl_pkey_get_private( $key );
        $data = openssl_pkey_get_details ( $resource );
        if ( $data['bits'] < 1024 ){
            $this->addError($attribute, 'Minimum bits for RSA key is 1024 ');
			return false;
        }
        
        $keyName = strtr( self::$keyFolder . self::$prefix.self::$keyName, array( "{id}" => $this->uid ));
        
        file_put_contents( $keyName, $key );
            
        return $keyName;
    }
    public function saveChain( $chain, $attribute )
    {
        $certName = strtr( self::$certFolder . self::$prefix. self::$certName, array( "{id}" => $this->uid ));        
        // Check if teh chain is needed or not (self signed certificates)
        exec(escapeshellarg($this->exe) . " verify -CAfile " . escapeshellarg($certName) . " " . escapeshellarg($certName), $output2, $ret);        
        if ( count($output2) < 1 || $output2[0] === "$certName: OK"){
            // certificate is self signed, drop the chain            
			return array();        
        }
        
        $idx = 0;
        $files = array();
        $pem = "";
        foreach( $chain as $cert ) {
            if ( !preg_match("/^-----BEGIN (X509 |TRUSTED |)CERTIFICATE-----/", $cert ) ) {
                $this->addError($attribute, 'Only X509 certificates are supported');
                foreach( $files as $file ){
                    unlink( $file );
                }
                return false;
            }

            $certName2 = strtr( self::$certFolder . self::$prefix.self::$chainName, array( "{id}" => $this->uid, "{idx}" => $idx ));
            $idx++;
            $files[] = $certName2;
            file_put_contents( $certName2, $cert ); 
            $pem .= $cert . "\n" ;
        }
        if ( $idx > 0 ){
            $pemName = strtr( self::$certFolder . self::$prefix. self::$chainPemName, array( "{id}" => $this->uid ));        
            file_put_contents( $pemName, $pem );    
            $files[] = $pemName;
            
            // validate against pem file 
            exec(escapeshellarg($this->exe) . " verify -CAfile " . escapeshellarg($pemName) . " " . escapeshellarg($certName), $output2, $ret);
            if ( count($output2) < 1 || $output2[count($output2)-1] !== "$certName: OK"){
                foreach( $files as $file ){
                    unlink( $file );
                }                
                $this->addError($attribute, 'Certificate cannot be verified');
                return false;        
            }
        } else {
            // missing the certificate chain for non self signed certificates. 
            $this->addError($attribute, 'Certificate cannot be verified or is not a self-signed certificate');
            return false;        
        }
        
        /*
        $data = openssl_x509_parse( $cert );  
        
        // check data        
        while(openssl_error_string() !== false);
        if ( openssl_x509_checkpurpose ( $cert, X509_PURPOSE_SSL_SERVER, array( self::$certFolder )) !== true ) {
            $this->addError($attribute, "Certificate cannot be used  for the server side of an SSL connection");
            while ($msg = openssl_error_string()){                
                $this->addError($attribute, $msg);
            }
            return false;
        }
        */
        
        return $files;
    }
    public function saveAll( $cert, $key, $chain, $attributes )
    {
        if ( !is_array($attributes)){
            $attributes= array( $attributes, $attributes, $attributes);
        }
        
        // check that the key matches the certificates
        if ( !openssl_x509_check_private_key ( $cert , $key ) ){
            $this->addError($attributes[1], 'Key do not match certificates');
            return false;
        }
        
        // save all
        
        $certFile = $this->saveCertificate( $cert, $attributes[0] );
        if ( $certFile === false ){            
            return false;
        }
        $keyFile = $this->saveKey( $key, $attributes[1] );
        if ( $keyFile === false ){
            unlink( $certFile );
            return false;
        }
        $chainFiles  = $this->saveChain( $chain, $attributes[2] );        
        if ( $chainFiles === false ){
            unlink( $certFile );
            unlink( $keyFile );
            return false;
        }
        
        exec('sync');
        
        return true;
    }
    public function savePCKS( $pkcs12, $attribute )
    {
        // clear up possible left over SSL errors
        while(openssl_error_string() !== false);
        
        $certs = array();
        
        if ( !openssl_pkcs12_read ( $pkcs12, $certs, $this->passphrase ) ){            
            $this->addError('pfxFile', strtr('{pfxFile} incorect', array("{pfxFile}"=> $this->getAttributeLabel('pfxFile'))));
            while ($msg = openssl_error_string()){
                if (strpos($msg, 'mac verify failure')){
                    $this->addError('passphrase', strtr('{passphrase} incorect', array("{passphrase}"=> $this->getAttributeLabel('passphrase'))));
                } else {
                    $this->addError('pfxFile', $msg);
                }
            }            
            return false;
        }
        if ( !isset($certs['extracerts']) ){
            $certs['extracerts'] = array();
        }
        return $this->saveAll( $certs['cert'], $certs['pkey'], $certs['extracerts'], $attribute );        
    }
    
    public function savePCKSFile( $runValidation=true, $attributes=null )
    {
        if( $runValidation && !$this->validate($attributes)){
            return false;
        }
        
        $pkcs12 = file_get_contents( $this->pfxFile->getTempName() );
        $ret = $this->savePCKS( $pkcs12, 'pfxFile' );
        
        $this->pfxFile = null;
        
        return $ret;
    }
    
    public function saveFiles( $runValidation=true, $attributes=null )
    {
        if( $runValidation && !$this->validate($attributes)){
            return false;
        }
                
        $cert = file_get_contents( $this->certificateFile->getTempName() );
        
        $key = file_get_contents( $this->keyFile->getTempName() );      
        
        // Decrypth the key if needed        
        $resource = openssl_pkey_get_private( $key, $this->passphrase );
        if ( $resource === false ){
            $this->addError('passphrase', strtr('{passphrase} incorect', array("{passphrase}"=> $this->getAttributeLabel('passphrase')))); 
            return false;
        }        
        if ( !empty($this->passphrase)){
            openssl_pkey_export( $resource, $key );
        }
        $chain = array();
        if ( $this->chainFile ){
            $pemChain = file_get_contents( $this->chainFile->getTempName() );      
            $matches = array();
            preg_match_all('/(-----BEGIN .*?CERTIFICATE-----.*?-----END .*?CERTIFICATE-----)/si', $pemChain, $matches);        
            $chain = $matches[0];
        }
        $this->saveAll( $cert, $key, $chain, array('certificateFile', 'keyFile', 'chainFile') );
        
        
        $this->certificateFile = null;
        $this->keyFile = null;
        $this->chainFile = null;
    }
    public function restoreBackup( $runValidation=true, $attributes=null ){
        if( $runValidation && !$this->validate($attributes)){
            return false;
        }
        if ( !empty( $this->pfxText ) ){
            $ret = $this->savePCKS( base64_decode($this->pfxText), 'pfxText' );
            $this->pfxText = "";
        } else if ( !empty( $this->certificateText ) && !empty( $this->keyText ) ){
            if ( !empty($this->chainText)){
                $matches = array();
                preg_match_all('/(-----BEGIN .*?CERTIFICATE-----.*?-----END .*?CERTIFICATE-----)/si', $this->chainText, $matches);        
                $chain = $matches[0];            
            } else {
                $chain = array();
            }
            // Decrypth the key if needed        
            $resource = openssl_pkey_get_private( $this->keyText, $this->passphrase );
            if ( $resource === false ){
                $this->addError('passphrase', strtr('{passphrase} incorect', array("{passphrase}"=> $this->getAttributeLabel('passphrase')))); 
                return false;
            }        
            if ( !empty($this->passphrase)){
                openssl_pkey_export( $resource, $this->keyText );
            }
            $ret =  $this->saveAll( $this->certificateText, $this->keyText, $chain, array('certificateText', 'keyText', 'chainText') );
            $this->keyText = "";
        } else {
            return true; // nothing to do
        }
        if ( !$ret ){
            return $ret;
        }
        if ( $this->active ){
            $this->scenario = 'activate';            
            return $this->activate();
        }        
        return true;
    }
    public function loadForBackup( $uid, $includeKey ){
        $this->certificateText = "";
        $this->keyText = "";
        $this->chainText = "";
        
        $this->uid = $uid;
        // load all the files in text attributes:
        if ($this->uid == self::$autoCertificateId ){
            $includeKey = false;
            $certName = self::$certFolder . self::$autoCertificateCert;                
        } else {
            $certName = strtr( self::$certFolder . self::$prefix. self::$certName, array( "{id}" => $this->uid ));                
        }
        $this->certificateText = file_get_contents( $certName );
        if ( $includeKey ){
            $keyName = strtr( self::$keyFolder . self::$prefix.self::$keyName, array( "{id}" => $this->uid ));
            $this->keyText = file_get_contents( $keyName );
        }
        $pemName = strtr( self::$certFolder . self::$prefix. self::$chainPemName, array( "{id}" => $this->uid ));  
        if (file_exists( $pemName )){
            $this->chainText = file_get_contents( $pemName );
        }
        return true;
    }
    function dump( $uid  ) {
        $this->loadForBackup( $uid, false );
        
        $name = rawurlencode($uid . ".crt");
        header("Content-type: application/x-x509-user-cert");
        header('Content-Disposition: attachment; filename=\'$name\'; filename*=utf-8\'\''.$name);
        
        echo $this->certificateText;
        return true;
	}
    static function getActiveHost() {
        $activeFile = self::getActiveFilename( );
        $cert = file_get_contents( $activeFile );
        $info = SSLConfig::certificateInfo( $cert );
        
        return $info['name'];
    }
    static function getActiveFilename( ) {
        if ( PHP_OS != "WINNT" ) {
            $activeFile = readlink( self::$activeLink );
        } else {
            if (file_exists( self::$certFolder ."active.lnk")){
                $activeFile = file_get_contents( self::$certFolder ."active.lnk" );
            } else {
                $activeFile = self::$certFolder . self::$autoCertificateCert;
            }
        }
        return $activeFile;
    }
    function getCertificatesList( $allInfo=true ) {
		$filelist=array( );
		if ( !file_exists(self::$certFolder ) ){
            return $filelist;
        }
		
        $activeFile = self::getActiveFilename( );
        
        if ( $allInfo ){
            $cert = file_get_contents( self::$certFolder . self::$autoCertificateCert );
            $info = SSLConfig::certificateInfo( $cert );
            $dest = self::$certFolder . self::$autoCertificateCert;
            exec(escapeshellarg($this->exe) . " x509 -in device.crt -text -noout -in " . escapeshellarg($dest), $output, $ret);
            $info['text'] = implode( "\n",$output );
        } else { 
            $info = array();
        }
        $info['uid'] = self::$autoCertificateId;
        $info['activeCertificates'] = self::$certFolder . self::$autoCertificateCert == $activeFile ;
        $info['filename'] = self::$autoCertificateCert;   
        $info['uploaded'] = false;
        $filelist[] = $info;
        
        $filelist1=scandir(self::$certFolder);
        foreach ($filelist1 as $file){
            $matches = array();
			if ( !preg_match("|^".strtr( self::$prefix. self::$certName, array( "." => "\\.", "{id}" => "([0-9a-zA-Z_.@]{4,32})" ))."$|", $file, $matches) ){
                continue;
            }
            
            if ( $allInfo ){
                $output = array();
                $cert = file_get_contents( self::$certFolder . $file );
                $info = SSLConfig::certificateInfo( $cert );
                
                $dest = self::$certFolder . $file;
                exec(escapeshellarg($this->exe) . " x509 -text -noout -in " . escapeshellarg($dest), $output, $ret);
                $info['text'] = implode( "\n",$output );
                // is there a bundle ?
                $info['bundle'] = array();
                $filelist2=scandir(self::$certFolder);
                foreach ($filelist2 as $file2){    
                    if ( strpos( $file2, $matches[1] ."-trust-") === false ){
                        continue;
                    }            
                    $output = array();
                    $dest = self::$certFolder . $file2;
                    exec(escapeshellarg($this->exe) . " x509 -text -noout -in " . escapeshellarg($dest), $output, $ret);
                    $info['bundle'][] = implode( "\n",$output );
                } 
            } else {
                $info = array();
            }            
            $info['uid'] = $matches[1];
            
            $info['activeCertificates'] = self::$certFolder . $file == $activeFile ;            
            $info['filename'] = $file;   
            $info['uploaded'] = true;
            $filelist[] = $info;
		}
		return $filelist;
	}
    
    function delete( $runValidation=true, $attributes=null, $sync =  true, $deleteActive = false) {
        if( $runValidation && !$this->validate( $attributes ) ){
            return false;
        }
		$activeFile = self::getActiveFilename( );
        $certName = strtr( self::$certFolder . self::$prefix. self::$certName, array( "{id}" => $this->uid ));
        if ( $certName == $activeFile && !$deleteActive ) {
            $this->addError('uid', 'Cannot remove active certificate');
            return false;
        }     
        $filelist1=scandir(self::$certFolder);
        foreach ($filelist1 as $file){    
            if ( strpos( $file, $this->uid) === false ){
                continue;
            }            
            unlink( self::$certFolder . $file );
		}     
        if ( $sync ){
            exec('sync');        
        }
        return true;
    }
    function activate( $runValidation=true, $attributes=null ) {
        if( $runValidation && !$this->validate( $attributes ) ){
            return false;
        }
        
        if ( $this->uid === self::$autoCertificateId){
            $certFilename = self::$autoCertificateCert;
            $keyFilename = self::$autoCertificateKey;
        } else {
            $certFilename = strtr( self::$prefix. self::$certName, array( "{id}" => $this->uid ));
            $keyFilename = strtr( self::$prefix. self::$keyName, array( "{id}" => $this->uid ));            
        }        
        $certName = self::$certFolder . $certFilename;

        if ( $certName === self::getActiveFilename( )){
            // nothing to do
            return true;
        }
		
        if( PHP_OS == "WINNT"){
            file_put_contents(self::$certFolder ."active.lnk", $certName);
        } else {
            unlink(self::$activeLink);
            unlink(self::$activeKey);
            exec('sync');
            
            $filelist1=scandir(self::$activeLinkFolder);
            foreach ($filelist1 as $file){    
                if (is_link( self::$activeLinkFolder . $file)){
                    unlink( self::$activeLinkFolder . $file );
                }                
            }  
            // is certificate self-signed
            exec(escapeshellarg($this->exe) . " verify -CAfile " . escapeshellarg($certName) . " " . escapeshellarg($certName), $output2, $ret);        
            if ( count($output2) < 1 || $output2[0] === "$certName: OK"){
                // certificate is self signed        
                Tools::save_symlink( $certName, self::$activeLinkFolder . $certFilename);
            }
            // is there a bundle ?
            $filelist2=scandir(self::$certFolder);
            foreach ($filelist2 as $file){    
                if ( strpos( $file, $this->uid ."-trust-") === false ){
                    continue;
                }            
                Tools::save_symlink( self::$certFolder . $file, self::$activeLinkFolder . $file);
            } 
            // update the set of default trusted roots (ca-certificates.crt) 
            exec("update-ca-certificates -f", $output3, $ret);
            
            //copy the crt, key and updated ca-certificates.crt to the failsafe-date of the
            //recovery console as httpd-cert.crt, httpd-cert.key and cert.pem, respectively
            Tools::setSafeModeDataExt( array(
                array( 'dst'=>'httpd-cert.crt', 'src'=>$certName ),
                array( 'dst'=>'httpd-cert.key', 'src'=> self::$keyFolder . $keyFilename ),
                array( 'dst'=>'cert.pem', 'src'=>self::$rootCertificateBundle ),
            ));
            
            //create links /etc/apache2/server.crt and /etc/apache2/server.key
            Tools::save_symlink( $certName, self::$activeLink);
            Tools::save_symlink( self::$keyFolder . $keyFilename, self::$activeKey);
            exec('sync');            
        }
        // must update the redirect in server security
        $serverSecurity = new ServerSecurity;
        $serverSecurity->load();
        $serverSecurity->save();

        Tools::$reload = true;
        Tools::addReason("server certificate activated");
        return true;
    }
    function clearCertificates() {
        $this->uid = self::$autoCertificateId;
        $this->activate( false );
        
        $filelist1=scandir(self::$certFolder);        
        foreach ($filelist1 as $file){
            $matches = array();
			if ( !preg_match("|^".strtr( self::$prefix. self::$certName, array( "." => "\\.", "{id}" => "([0-9a-zA-Z_.@]{4,32})" ))."$|", $file, $matches) ){
                continue;
            }
            $this->uid = $matches[1];
            $this->delete( false, null, false, true);           
        }
        exec('sync'); 
    }
    
}
