<?php

require_once 'protected/utils/PullMode.php';

class PullModeConfig extends CFormModel {
	
	const PULL_DISABLED  	= "disabled";
	const PULL_STATIC     	= "static"; // backward compatibilty only
	const PULL_FILE     	= "file";
	const PULL_REMOTE     	= "remote";
	
    protected $config_name='/etc/raperca/uploaderconf.xml';
    protected $uploadercal="/etc/raperca/uploader.ics";
    
    protected $logconf="/etc/raperca/uploaderlog.xml";

    protected $_networkLog = null;
    protected $_cookieLog = null;
    protected $_cacheLog = null;

    var $selectMode=null;
    
    var $syncSources = array();
    
    var $enable_log= false;
    var $log_source="";
    var $hourLog = 0;
    var $minLog = 30;
    var $log_type='alllogs';
    
	var $enable_rpc=0;
    var $concentrator="";
	var $polling="60s";
	var $notification_only=false;
	
    var $icalfile="";
    var $icsfile;
    
    var $backup = false;
    
    var $uri="";
    
    var $checktime = "24h";
    
    var $priority=null;
    
    var $https_useDefaultCA=null;
   
    var $cockpitUri = null;
    var $cockpitOptions = "-polling 60";
    var $cockpitTime = 0;
    var $priorityList = array(  'trace'=>"Trace",
                                'debug'=>"Debug",
                                'info'=>"Info",
                                'warn'=>"Warning",
                                'error'=>"Error",
                                'fatal'=>"Fatal",);
    
    public function rules() {
		$res = array(
            array( "syncSources", 'validateSource' ),
            array( "enable_log, timeLog", 'safe', 'on'=>'manual' ),
            array( "hourLog", 'numerical', 'min'=>0, 'max'=>23, 
                'integerOnly'=>true, 'allowEmpty'=>false,
                'on'=>'manual' ),
            array( "minLog", 'numerical', 'min'=>0, 'max'=>59, 
                'integerOnly'=>true, 'allowEmpty'=>false,
                'on'=>'manual' ),
            array( "log_type", 'in', 'range'=>array( "accounting", "alllogs" ),
                'allowEmpty'=>false,
                'on'=>'manual' ),
			array( "enable_rpc, notification_only", 'safe', 'on'=>'manual' ),
            array( "polling", 'checkTime', 'on'=>'manual' ),
			
			array( "log_source, concentrator", 'emptyselect', 'on'=>'manual' ),
            array( "log_source, concentrator", 'SimpleURLValidator', 'on'=>'manual' ),
			
            array( "icalfile", 'file', 'types'=>'ics', 'on'=>'file', 'allowEmpty' => true ),
            
            array( "icsfile", 'checkICS', 'on'=>'file_inline' ),
            
            array( "icalSourceHref", 'safe', 'on'=>'remote' ),
            array( "checktime", 'checkTime', 'on'=>'remote' ),
            array( "uri", 'SimpleURLValidator', 'on'=>'remote' ),
            array( "priority", 'in', 'range'=>array("trace","debug","info","warn","error","fatal") ),
            
            array( "selectMode", 'checkSelectMode' ),
            
            array( 'cockpitUri', 'SimpleURLValidator' ),
            array( 'cockpitOptions', 'safe' ),
            array( 'cockpitTime', 'safe' ),
            array( 'httpLogs', 'safe' )
    	);
        return $res;
	}
    public function attributeLabels()
	{
        $labels = array(
            'selectMode' => "Settings",
            'mode_dis'=>'Disabled',
            'mode_manual'=>'Manual settings',
            'mode_file'=>'From uploaded iCalendar file (ics)',
            'mode_remote'=>'From remote iCalendar file (ics)',
            'enable_log'=>'Automatically upload logs from the '.Yii::app()->branding->product,
			'enable_rpc'=>'Enable RPC concentrator',
            'concentrator'=>'Concentrator uri',
			'polling'=>'Polling interval',
			'notification_only'=>'Notification only',
			'log_source'=>'Log destination',
            'hourLog'=>'Time',
            'log_type'=>'Accounting logs only',
            'logs_accounting'=>'Accounting logs only',
            'logs_all'=>'All logs',
            'icalfile'=>'Schedule file',
            'icalSourceHref'=>'Schedule uri',
            'checktime'=>'Check calendar every',
            'uri'=>"Schedule uri",
            'priority'=>'Log level',
            'publishSrc' => 'Server URI',
            'cockpitUri' => 'Concentrator URI'
		);
        return $labels;
	}
    public function validateSource( $attribute, $params ) { 
        $value = $this->$attribute;
        if (!is_array($value) ) {
            $this->addError($attribute, "not an array");
            return;
        }
        foreach($value as $source) {
            if ( !$source->validate() ){
                if ( $source->getError( 'dest' ) )
                    $this->addError($attribute, $source->getError( 'dest' ) );
                if ( $source->getError( 'uri' ) )
                    $this->addError($attribute, $source->getError( 'uri' ) );
                if ( $source->getError( 'update' ) )
                    $this->addError($attribute, $source->getError( 'update' ) );
                if ( $source->getError( 'time' ) )
                    $this->addError($attribute, $source->getError( 'time' ) );
                if ( $source->getError( 'norecurs' ) )
                    $this->addError($attribute, $source->getError( 'norecurs' ) );
            }              
        }
    }
    function resetSyncSource() {
        $this->syncSources = array();
    }
    function addSyncSource( $source ){
        $this->syncSources[] = $source;
    }
    function checkTime( $attribute ) {
        $t = Tools::timestr_to_secs( $this->$attribute );
        if ( $t===false || $t<=0 ) {
            $this->addError($attribute, "Invalid time format");
        }
    }
    function checkICS( $attribute ) {
        $val = $this->$attribute;
        if ( empty($val) ){
            $this->addError($attribute, "Must provide an ICS file");
            return false;
        }
        if ( strpos($val, "BEGIN:VCALENDAR")=== false || 
                strpos($val, "END:VCALENDAR")=== false ){
            $this->addError($attribute, "Not a valid ICS file");
            return false;
        }        
    }
    function checkSelectMode() {
        switch ($this->selectMode) {            
            case PullModeConfig::PULL_FILE: 
                                            
                break;
            case PullModeConfig::PULL_REMOTE: $this->setScenario('remote'); 
                if ( empty($this->uri) ){
                    $this->addError('icalSourceHref', 'Schedule uri cannot be blanck');                    
                }
                break;            
        }
    }
    function formatTime( $h, $m ) {
        if ( intval($h, 10) == -1 ){
            $h = "XX";
        } else {
            if ( intval($h, 10)<10 ){
                $h = "0".intval($h, 10);
            }
        }
        if ( intval($m, 10)<10 )
            $m = "0".intval($m, 10);
        return "$h:$m";
    }
    
    function getTimeLog (){
        return $this->formatTime( $this->hourLog, $this->minLog );
    }
    
    function setTimeLog( $time ){
        $times = explode(":", $time);
        if ( count($times==2) && is_numeric($times[0]) && is_numeric($times[1]) ){
            $this->hourLog = intval($times[0], 10);
            $this->minLog = intval($times[1], 10);
        } else {
            $this->hourLog = $time;
            $this->minLog = "";
        }
    }
    function setSelectMode( $mode ) {
        if ( $mode == PullModeConfig::PULL_STATIC ){
            $this->setScenario('manual');             
            return;
        }
        $this->selectMode = $mode;
        switch ($mode) {
            case PullModeConfig::PULL_DISABLED: 
                $this->setScenario('disabled'); break;            
            case PullModeConfig::PULL_FILE: 
                if ($this->backup)
                    $this->setScenario('file_inline'); 
                else
                    $this->setScenario('file'); 
                break;
            case PullModeConfig::PULL_REMOTE:
                $this->setScenario('remote'); 
                $this->uri="";
                $this->checktime = "24h";
                break;
            default: 
                $this->addError('selectMode', 'Mode not supported');
        }
    }
    function setIcalSourceHref( $href ) {
        $this->uri=preg_replace("@\[serial\]@",Yii::app()->device->serviceSerial, $href);
    }
    function getIcalSourceHref() {
        return preg_replace("@".Yii::app()->device->serviceSerial."@","[serial]", $this->uri);
    }
    
    function load( $modeOnly = false )
    {
        $spxNS = 'http://www.spinetix.com/namespace/1.0/spxconf';
        $xlinkNS = 'http://www.w3.org/1999/xlink';


        $config = new DOMDocument;
        $config->loadXML(file_get_contents($this->config_name));
        $this->selectMode = PullModeConfig::PULL_DISABLED;
        
        
        $schedule=$config->getElementsByTagName("schedule");
        if ($schedule->length>0) {
            $link=$schedule->item(0)->getAttributeNS($xlinkNS,"href");
            if ( $link=="uploader.ics" || $link==$this->uploadercal ) {
                $this->selectMode = PullModeConfig::PULL_FILE;
            } else {
                $this->selectMode = PullModeConfig::PULL_REMOTE;
                $this->uri=$link;
                $checktime = $schedule->item(0)->getAttribute("check");
                if ($checktime=="") 
                    $this->checktime = "24h";
                else {
                    $this->checktime = Tools::secs_to_timestr($checktime);
                }
            }
        } 
        if ( $modeOnly ){
            return;
        }
        $proj=$config->getElementsByTagName("action");
        $nbPublish = 0;
        $this->syncSources = array();
        foreach ($proj as $action) {
            $name = $action->getAttribute("summary");
            if ($name == '' ) {
                $source = new SyncSource;
                $source->uri = $action->getAttributeNS($xlinkNS,"href");
                $source->dest = $action->getAttribute("dest");
                $source->norecurs = intval( $action->getAttribute("norecurs") );
                $source->update = 'manual';
                $this->syncSources[] = $source;
            }
            if ($name=='publish') {
                $update = $action->getAttribute("frequency");
                $time = $action->getAttribute("time");
                $h=floor($time/3600);
                
                if ( $update == 'hourly' && $h != 0){
                    continue;
                }
                $m=floor(($time-$h*3600)/60);
                if ( $m < 10 ){
                    $m = "0".$m;
                }
                if ( $h < 10 ){
                    $h = "0".$h;
                }
                $source = new SyncSource;
                $source->uri = $action->getAttributeNS($xlinkNS,"href");
                $source->dest = $action->getAttribute("dest");
                $source->norecurs = intval( $action->getAttribute("norecurs") );
                
                $source->update = $update;
                if ( $update == 'daily' ){
                    $source->time = $h .":" .$m;                                
                }
                $this->syncSources[] = $source;

            }
            if ($name=='upload') {
                $this->enable_log = 1;
                $this->log_source=$action->getAttributeNS($xlinkNS,"href");
                $time = $action->getAttribute("time");
                $this->hourLog=floor($time/3600);
                $this->minLog=floor(($time-$this->hourLog*3600)/60);
                $this->log_type=$action->getAttribute("type");                
            }
            if ( $name=='rpc' && !$action->getAttribute('cockpit') ) {
                $this->enable_rpc = 1;
                $this->concentrator=$action->getAttributeNS($xlinkNS,"href");
                if ( $action->hasAttribute("pooling" ) ) // backward compatibility
                    $this->polling = Tools::secs_to_timestr( $action->getAttribute("pooling") );
                else
                    $this->polling = Tools::secs_to_timestr( $action->getAttribute("polling") );
                $this->notification_only=$action->getAttribute("notification")=="true";                
            }
            if ( $name=='rpc' && $action->getAttribute('cockpit') ) {
                $this->cockpitUri = $action->getAttributeNS($xlinkNS,"href");
                $this->cockpitOptions = trim($action->textContent);
                $this->cockpitTime = intval( $action->getAttribute("time"), 10 );
            }
        }
        if ( $nbPublish > 1 ){
            $this->hourProj = -1;
        }

        $https_configs=$config->getElementsByTagName("https");
        foreach ($https_configs as $https) {
            if ( $https->getAttribute("useDefaultCA")=='false' )
                $this->https_useDefaultCA = false;
            else
                $this->https_useDefaultCA = true;
        }

        $networkLog = $config->getElementsByTagName("networkLog");
        if ($networkLog->length > 0){
            $this->_networkLog = $networkLog->item(0)->getAttributeNS($xlinkNS,"href");
        }
        $cacheLog = $config->getElementsByTagName("cacheLog");
        if ($cacheLog->length > 0){
            $this->_cacheLog = $cacheLog->item(0)->getAttributeNS($xlinkNS,"href");
        }
        $cookieLog = $config->getElementsByTagName("cookieLog");
        if ($cookieLog->length > 0){
            $this->_cookieLog = $cookieLog->item(0)->getAttributeNS($xlinkNS,"href");
        }
    }
    public function getHttpLogs() {
        $resp = "";
        if ( $this->_networkLog !== "" ) $resp = "network";
        if ( $this->_cookieLog !== "" ) $resp = ($resp===""? "": $resp . "," ) ."cookie";
        if ( $this->_cacheLog !== "" ) $resp = ($resp===""? "": $resp . "," ) ."cache";
        if ( $resp === "network,cookie,cache" )  $resp = "all";
        return $resp; 
    }
    public function setHttpLogs( $input ) {        
        $features = array_map('trim', explode(",", $input));

        if ( in_array("all", $features) ) $features = array("network","cookie","cache");

        $this->_networkLog = in_array("network", $features) ? "/srv/raperca/capture/HTTP-pullmode-network-".date('Y-m-d-His').".json" : "";
        $this->_cookieLog = in_array("cookie", $features) ? "/srv/raperca/capture/HTTP-pullmode-cookie-".date('Y-m-d-His').".json" : "";
        $this->_cacheLog = in_array("cache", $features) ? "/srv/raperca/capture/HTTP-pullmode-cache-".date('Y-m-d-His').".json" : "";
    }

    function loadLogConf( ){
        $logconfig = new DOMDocument;
        $logconfig->loadXML(file_get_contents($this->logconf));
        $priority = $logconfig->getElementsByTagName("priority");
        if ( $priority->length>0 ){
            $this->priority = $priority->item(0)->getAttribute("value");
        }
    }
    function saveLogConf( ){
        if ( $this->priority!=null ) {
            $logconfig = new DOMDocument;
			$logconfig->preserveWhiteSpace = false;
			$logconfig->formatOutput = true;
            $logconfig->loadXML(file_get_contents($this->logconf));
            $priority=$logconfig->getElementsByTagName("priority");
            if ( $priority->length>0) {
                $priority->item(0)->setAttribute("value",$this->priority);
                Tools::save_file($this->logconf, $logconfig->saveXML());
                exec('/etc/init.d/uploader reload');
            } else {
                $this->setError('priority', 'Cannot save priority');            
            }
        }  
    }
    function addPublishAction( $source, $time, $main, $config ) {
        
        $spxNS = 'http://www.spinetix.com/namespace/1.0/spxconf';
        $xlinkNS = 'http://www.w3.org/1999/xlink';
        $proj=$main->appendChild($config->createElementNS($spxNS,'action'));        
        $proj->setAttribute("guid", uniqid() );
        $proj->setAttributeNS($xlinkNS,"xlink:href", $source->uri);
        if ( $time ) {
            $proj->setAttribute("summary", 'publish');
            $t = explode(":", $time );
            $saveTime = $t[0]*3600+$t[1]*60;
            $proj->setAttribute("time", $saveTime);
        } 
        if ( $source->dest != "" ) {
            $proj->setAttribute("dest", $source->dest);
            $desc=$proj->appendChild($config->createElementNS($spxNS,'desc'));
            $desc->appendChild($config->createTextNode("-dest ".$source->dest));    
        }
        if ( $source->norecurs ) {
            $proj->setAttribute("norecurs", 1);
            $desc=$proj->appendChild($config->createElementNS($spxNS,'desc'));
            $desc->appendChild($config->createTextNode("-norecurs"));    
        }
        return $proj;
    }
    function checkAndFixCockpit(){   
        
        if( $this->selectMode == PullModeConfig::PULL_FILE && $this->scenario == 'file_inline' ){
            $ics = $this->icsfile;
            $matches = array();
            $fixedics = str_replace(array("\r\n ", "\n ", "\r "), "", $ics);
            $nbEvents = 0;
            if ( strpos( $fixedics, "PRODID:icalendar-ruby" ) ) {
                // Find each event 
                $st = strpos( $fixedics, "BEGIN:VEVENT");
                while( $st !== false ){                       
                    $ed = strpos( substr($fixedics, $st), "END:VEVENT");
                    if ( $ed === false ){
                        return;
                    }
                    $event = substr($fixedics, $st, $ed+strlen("END:VEVENT") );
                    if ( strpos("SUMMARY:rpc", $event) !== null && preg_match("/URL\:(https?:\/\/(?:cockpit-staging|concentrator)\.spinetix\.com.*)/", $event, $matches )) {
                        $this->cockpitUri = trim($matches[1]);
                        if ( preg_match("/DESCRIPTION\:(.*)/", $fixedics, $matches ) ){
                            $options = str_replace(array("\\,"), ",", $matches[1] );
                            $this->cockpitOptions = trim( str_replace(array("\\r\\n", "\\r", "\\n"), "\n", $options) );
                        }
                        if ( preg_match("/DTSTART\:\d*T(\d{2})(\d{2})(\d{2})/", $fixedics, $matches ) ){                        
                            $this->cockpitTime = intval( $matches[1], 10 )* 3600 +  intval( $matches[2], 10 ) *60 + intval( $matches[3], 10 );
                            $this->cockpitTime = $this->cockpitTime % 300;
                        }                       
                        // kill this event from the ICS file                        
                        $fixedics = substr($fixedics, 0, $st) . substr($fixedics, $st + $ed +strlen("END:VEVENT")+1 );
                        $this->icsfile = $fixedics;
                        $st = strpos( $fixedics, "BEGIN:VEVENT");
                    } else {
                        $nbEvents++;
                        $st = strpos( substr($fixedics, $st + $ed+strlen("END:VEVENT") ), "BEGIN:VEVENT") + $st + $ed+strlen("END:VEVENT");
                    }                    
                }  
                if ( $nbEvents == 0 ){
                    $this->icsfile = "";
                    $this->selectMode = PullModeConfig::PULL_DISABLED;
                }
            }            
        } 
        if ( $this->enable_rpc ){
            if ( preg_match("/https?:\/\/(?:cockpit-staging|concentrator)\.spinetix\.com/", $this->concentrator )) {
                $this->cockpitUri = $this->concentrator;                    
                $this->cockpitOptions = "-polling ". Tools::timestr_to_secs( $this->polling );
                $this->enable_rpc = 0;
            }            
        }
    }
    
    function save()
    {
        $spxNS = 'http://www.spinetix.com/namespace/1.0/spxconf';
        $xlinkNS = 'http://www.w3.org/1999/xlink';

        if ( $this->selectMode!=null ) {
            $config = new DOMDocument;
			$config->preserveWhiteSpace = false;
			$config->formatOutput = true;
            $main=$config->appendChild($config->createElementNS($spxNS,'uploader'));
            
            if ( $this->https_useDefaultCA !== null ) {
                $https = $main->appendChild( $config->createElementNS($spxNS,'https') );
                if ( $this->https_useDefaultCA ) {
                    $https->setAttribute( "useDefaultCA", 'true' );
                } else {
                    $https->setAttribute( "useDefaultCA", 'false' );
                }
            }

            $resource = $config->getElementsByTagName("resource");
            if ($resource->length == 0){
                $resource = $config->createElement( "resource" );
                $config->documentElement->appendChild($resource);
            }else {
                $resource = $resource->item(0);
            }
            
            $networkLog = $config->getElementsByTagName("networkLog");
            if ($networkLog->length == 0){
                $networkLog = $config->createElement( "networkLog" );
                $resource->appendChild($networkLog);
            } else {
                $networkLog = $networkLog->item(0);
            }
            $networkLog->setAttributeNS($xlinkNS, "xlink:href", $this->_networkLog);

            $cacheLog = $config->getElementsByTagName("cacheLog");
            if ($cacheLog->length == 0){
                $cacheLog = $config->createElement( "cacheLog" );
                $resource->appendChild($cacheLog);
            } else {
                $cacheLog = $cacheLog->item(0);
            }
            $cacheLog->setAttributeNS($xlinkNS, "xlink:href", $this->_cacheLog);


            $cookieLog = $config->getElementsByTagName("cookieLog");
            if ($cookieLog->length == 0){
                $cookieLog = $config->createElement( "cookieLog" );
                $resource->appendChild($cookieLog);
            } else {
                $cookieLog = $cookieLog->item(0);
            }
            $cookieLog->setAttributeNS($xlinkNS, "xlink:href", $this->_cookieLog);

            // detecting cockpit configuration
            $this->checkAndFixCockpit();
            switch ( $this->selectMode ) {
                case PullModeConfig::PULL_DISABLED:// disabled
                    $proj=$main->appendChild($config->createElementNS($spxNS,'disabled'));
                break;
                case PullModeConfig::PULL_FILE:// file
                    if ( $this->scenario == 'file_inline' ){
                        $this->setSchedule( $this->icsfile );
                    } else {
                        $obj = CUploadedFile::getInstance($this,'icalfile');
                        if ( $obj ) {
                            if( !$obj->saveAs( $this->uploadercal ) ) {
                                $this->addError('uploadercal', 'Saving the ics file failed');
                            }
                        }
                    }
                    $proj=$main->appendChild($config->createElementNS($spxNS,'schedule'));
                    $proj->setAttributeNS($xlinkNS,"xlink:href", "uploader.ics");
                break;
                case PullModeConfig::PULL_REMOTE:// remote
                    $proj=$main->appendChild($config->createElementNS($spxNS,'schedule'));
                    
                    $proj->setAttributeNS($xlinkNS,"xlink:href", $this->uri);
                    $time = Tools::timestr_to_secs($this->checktime);
                    if ( $time===false )
                        $this->addError("checktime", "Invalid time format");
                    if ( $time<10 ) {
                        $time = 10;
                        $this->checktime = "10s";
                    }
                    $proj->setAttribute("check", $time);
                break;
            }
            
            foreach ( $this->syncSources as $source ){
                if ( $source->update == 'manual' ) {
                    $this->addPublishAction( $source, null, $main, $config );
                } else if ( $source->update == 'hourly' ) {
                    for ( $i=0; $i<24; $i++ ){
                        $this->addPublishAction( $source, $i.":00", $main, $config )
                                ->setAttribute("frequency", "hourly");                        
                    }                    
                } else {
                    $this->addPublishAction( $source, $source->time, $main, $config )
                            ->setAttribute("frequency", "daily");
                }
            }
            
            $this->enable_log = intval( $this->enable_log );
            if ( $this->enable_log ){
                $this->enable_log = 1;
                $proj=$main->appendChild($config->createElementNS($spxNS,'action'));
                $proj->setAttribute("summary", 'upload');
                $proj->setAttribute("guid", uniqid() );
                $proj->setAttributeNS($xlinkNS,"xlink:href", $this->log_source);
                $saveTime2 = $this->hourLog*3600+$this->minLog*60;
                $proj->setAttribute("time", $saveTime2);


                if ( $this->log_type==='accounting' ) {
                    $proj->setAttribute("type", 'accounting');
                    $desc=$proj->appendChild($config->createElementNS($spxNS,'desc'));
                    $desc->appendChild($config->createTextNode("-accounting"));
                } else {
                    $proj->setAttribute("type", 'alllogs');
                    $desc=$proj->appendChild($config->createElementNS($spxNS,'desc'));
                    $desc->appendChild($config->createTextNode("-logs"));
                }
            }
            $this->enable_rpc = intval( $this->enable_rpc );
            if ( $this->enable_rpc ){
                $this->enable_rpc = 1;
                $proj=$main->appendChild($config->createElementNS($spxNS,'action'));
                $proj->setAttribute("summary", 'rpc');
                $proj->setAttribute("guid", uniqid() );
                $proj->setAttributeNS($xlinkNS,"xlink:href", $this->concentrator);
                $polling_interval = Tools::timestr_to_secs( $this->polling );

                $proj->setAttribute("polling", $polling_interval);
                $proj->setAttribute("notification", $this->notification_only?"true":"false");
                $desc=$proj->appendChild($config->createElementNS($spxNS,'desc'));
                $txt = "-polling " . $polling_interval;
                if ( $this->notification_only ) 
                    $txt .= "\n-notification";
                // We create one event valid for 24h.
                $proj->setAttribute("time", 0);
                $proj->setAttribute("end", 24*3600 );
                $desc->appendChild( $config->createTextNode( $txt ) );
            }
            if ( $this->cockpitUri === "" ){
                $this->cockpitUri = null;
            }
            if ( $this->cockpitUri ){
                $proj=$main->appendChild($config->createElementNS($spxNS,'action'));
                $proj->setAttribute("summary", 'rpc');
                $proj->setAttribute("guid", uniqid() );
                $proj->setAttribute("time", $this->cockpitTime );
                $proj->setAttribute("end", 24*3600 );
                $proj->setAttribute("cockpit", "1" );
                $proj->setAttributeNS($xlinkNS, "xlink:href", $this->cockpitUri);
                $desc=$proj->appendChild($config->createElementNS($spxNS, 'desc'));  
                $desc->appendChild( $config->createTextNode( $this->cockpitOptions ) );
            }
            if ( !$this->hasErrors() ){
                Tools::save_file($this->config_name,$config->saveXML());
                exec('/etc/init.d/uploader reload');
		    }
        }        
    }
    function setSchedule( $cnt ) {
        file_put_contents( $this->uploadercal, $cnt );
    }
    function getSchedule() {
        return file_get_contents($this->uploadercal);
    }
    function download() {
        if (file_exists($this->uploadercal) ){
            header("Content-type: text/calendar");
            header('Content-Disposition: attachment; filename="pullmode.ics"');
	        echo file_get_contents($this->uploadercal);
            return true;            
        } else {
            return false;
        }
    }
	function emptyselect( $attribute,$params ) {
		if ( ( $attribute=='project_source' && $this->project_source=="" && $this->enable_project) ||
			 ( $attribute=='log_source' && $this->log_source=="" && $this->enable_log) ||
			 ( $attribute=='concentrator' && $this->concentrator=="" && $this->enable_rpc) ){
			// error
			$this->addError($attribute, $this->getAttributeLabel($attribute).' cannot be blanck.' );
		}	
        if ( $attribute=='project_source' && !$this->enable_project) {
            $this->project_source = "";
        }
        if ( $attribute=='log_source' && !$this->enable_log) {
            $this->log_source = "";
        }
        if ( $attribute=='concentrator' && !$this->enable_rpc) {
            $this->concentrator = "";
        }
	}
    
}