<?php

class NTPSettings extends CFormModel
{
    var $file = "/var/spool/spxutils/dateadjust";

    var $fname;
    var $fname2 = "/etc/default/ntpdate";
    var $fname3 = "/etc/default/bootpause";
    var $active;
    var $server = "";
    var $nr_servers = 0;
    var $nr_sync_servers = 0;
    var $bootdelay = 0;

    var $ntpdelayList = array(
        '0' => '0 s',
        '90' => '90 s',
        '120' => '120 s',
        '150' => '150 s',
    );
    var $ntpserver1, $ntpserver2, $ntpserver3, $ntpserver4, $ntpserver5;
    var $ntpnosel1, $ntpnosel2, $ntpnosel3, $ntpnosel4, $ntpnosel5;

    var $stats = array();
    var $message = "";

    protected $_date = ""; // for the validation
    var $dateday = false;
    var $datemonth;
    var $dateyear;

    protected $_time = ""; // for the validation
    var $timeh = false;
    var $timem;
    var $times;

    var $months = array(
        "01" => "January",
        "02" => "February",
        "03" => "March",
        "04" => "April",
        "05" => "May",
        "06" => "June",
        "07" => "July",
        "08" => "August",
        "09" => "September",
        "10" => "October",
        "11" => "November",
        "12" => "December",
    );
    var $days = array();
    var $years = array();
    var $houres = array();
    var $minutes = array();
    var $seconds = array();

    public function init()
    {
        parent::init();
        $this->fname = file_exists("/etc/ntp/local.conf")
            ? "/etc/ntp/local.conf"
            : "/etc/spinetix/spx-ntp.conf";
        for ($i = 1; $i < 10; $i++) {
            $this->days['0' . $i] = '0' . $i;
        }
        for ($i = 10; $i <= 31; $i++) {
            $this->days[$i] = $i;
        }
        for ($i = 2022; $i <= 2037; $i++) {
            $this->years[$i] = $i;
        }
        for ($i = 0; $i < 10; $i++) {
            $this->houres['0' . $i] = '0' . $i;
        }
        for ($i = 10; $i < 24; $i++) {
            $this->houres[$i] = $i;
        }
        for ($i = 0; $i < 10; $i++) {
            $this->minutes['0' . $i] = '0' . $i;
            $this->seconds['0' . $i] = '0' . $i;
        }
        for ($i = 10; $i < 60; $i++) {
            $this->minutes[$i] = $i;
            $this->seconds[$i] = $i;
        }
    }

    /**
     * @return array validation rules for model attributes.
     */
    public function rules()
    {
        return array(
            array('bootdelay', 'in', 'range' => array("0", "90", "120", "150")),
            array(
                'ntpserver1, ntpserver2, ntpserver3, ntpserver4, ntpserver5',
                'checkServerName',
            ),
            array('ntpserver1', 'required', 'on' => 'ntp'),
            array(
                'ntpnosel1, ntpnosel2, ntpnosel3, ntpnosel4, ntpnosel5, active',
                'safe',
            ),
            array('dateday, datemonth, dateyear, timeh, timem, times', 'safe'),
            array(
                'date',
                'type',
                'type' => 'date',
                'dateFormat' => 'dd-MM-yyyy',
                'allowEmpty' => true,
                'on' => 'manual',
            ),
            array(
                'time',
                'type',
                'type' => 'time',
                'timeFormat' => 'hh:mm:ss',
                'allowEmpty' => true,
                'on' => 'manual',
            ),
            array('date, time', 'checkSet', 'on' => 'manual'),
        );
    }

    public function attributeLabels()
    {
        return array(
            'ntpserver1' => 'Server 1',
            'ntpserver2' => 'Server 2',
            'ntpserver3' => 'Server 3',
            'ntpserver4' => 'Server 4',
            'ntpserver5' => 'Server 5',
            'bootdelay' => 'Pause device at startup to wait for NTP servers by',
            'dateday' => 'Date',
            'timeh' => 'Time',
            'active' => 'Automatic time from Internet (NTP)',
        );
    }

    public function checkServerName($attribute, $params)
    {
        $value = $this->$attribute;
        if ($value == "") {
            return true;
        }
        if (preg_match('/^\[([\s\S]*)\]$/', $value, $matches)) {
            if (!filter_var($matches[1], FILTER_VALIDATE_IP, 
                FILTER_FLAG_IPV6
            )) {
                $this->addError($attribute, 'Invalid format');
                return false;
            }
        } else {
            if (!filter_var($value, FILTER_VALIDATE_DOMAIN,
                    FILTER_FLAG_HOSTNAME
                ) 
                && !filter_var($value, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)
            ) {
                $this->addError($attribute, 'Invalid format');
                return false;
            }
        }
        return true;
    }
    protected function formatNTPServer($ip, $name)
    {
        return $ip->resolveLinkLocalAddress($name);
    }
    protected function cleanNTPServer($name)
    {
        $val = explode("%", $name);
        return $val[0];
    }

    public function checkSet()
    {
        if ($this->_time == "" && $this->_date != "") {
            $this->addError('time', "Time cannot be empty if date is set");
            return false;
        }
        if ($this->_time != "" && $this->_date == "") {
            $this->addError('date', "Date cannot be empty if time is set");
            return false;
        }
    }
    function getDate()
    {
        if ($this->_date != "") {
            return $this->_date;
        }
        if ($this->dateday === false) {
            return "";
        }
        return $this->dateday . "-" . $this->datemonth . "-" . $this->dateyear;
    }
    function getTime()
    {
        if ($this->_time != "") {
            return $this->_time;
        }
        if ($this->timeh === false) {
            return "";
        }
        $h = $this->timeh;
        //if ( $h<10 ) $h = "0".$h;
        $m = $this->timem;
        //if ( $m<10 ) $m = "0".$m;
        $s = $this->times;
        //if ( $s<10 ) $s = "0".$s;
        return $h . ":" . $m . ":" . $s;
    }
    function setDate($date)
    {
        $this->_date = $date;
        $dates = explode("-", $date);
        if (count($dates) == 3) {
            $this->dateday = $dates[0];
            $this->datemonth = $dates[1];
            $this->dateyear = $dates[2];
        }
    }
    function setTime($time)
    {
        $this->_time = $time;
        $times = explode(":", $time);
        if (count($times) == 3) {
            $this->timeh = $times[0];
            $this->timem = $times[1];
            $this->times = $times[2];
        }
    }
    function loadTime()
    {
        $now = time();
        if (file_exists($this->file)) {
            $adjust = file_get_contents($this->file);
            $offsets = explode(" ", $adjust);
            if (!count($offsets) != 8) {
                $now +=
                    $offsets[0] * 24 * 3600 +
                    $offsets[2] * 3600 +
                    $offsets[4] * 60 +
                    $offsets[6];
            }
        } else {
            $now += 60;
        }
        $this->times = '00';
        $this->timem = date("i", $now);
        $this->timeh = date("H", $now);
        if ($this->timem < 10) {
            $this->timem = '0' . $this->timem;
        }
        //if ( $this->timeh<10 )
        //	$this->timeh = '0'.$this->timeh;
        $this->dateday = date("j", $now);
        if ($this->dateday < 10) {
            $this->dateday = '0' . $this->dateday;
        }
        $this->datemonth = date("m", $now);
        $this->dateyear = date("Y", $now);
    }
    function load()
    {
        $this->bootdelay = 0;
        if (
            preg_match(
                '/^BOOTPAUSE=\"*([^\"\s]+)/m',
                file_get_contents($this->fname3),
                $matches
            )
        ) {
            $this->bootdelay = $matches[1];
        }
        $ntpconf = file_get_contents($this->fname);
        $this->active = true;
        $this->nr_servers = 0;
        $this->nr_sync_servers = 0;
        if (isset($ntpconf)) {
            $lines = explode("\n", $ntpconf);

            foreach ($lines as $line) {
                if (preg_match('/^# Not active/', $line, $matches)) {
                    $this->active = false;
                }
                if (preg_match('/^#?\s*server\s+(\S+)(.*)/', $line, $matches)) {
                    $this->nr_servers++;
                    $name = "ntpserver" . $this->nr_servers;
                    $nosel = "ntpnosel" . $this->nr_servers;
                    $address = $matches[1];
                    if (strpos($address, ":") !== false) {
                        $this->$name =
                            "[" . $this->cleanNTPServer($address) . "]";
                    } else {
                        $this->$name = $address;
                    }
                    if (strpos($matches[2], "noselect") === false) {
                        $this->$nosel = "no";
                        $this->nr_sync_servers++;
                    } else {
                        $this->$nosel = "yes";
                    }
                }
            }
        }

        if ($this->nr_servers == 0) {
            $this->active = false;
        }
        $this->loadTime();
    }

    function save($ip = null)
    {
        if (
            !$this->active &&
            $this->dateday !== false &&
            $this->timeh !== false
        ) {
            $now = time();
            $in = strtotime($this->getDate() . "T" . $this->getTime());
            $updateOffset = $in - $now;

            $Days = round($updateOffset / (60 * 60 * 24));
            $Hours = round(($updateOffset - $Days * 60 * 60 * 24) / (60 * 60));
            $Minutes = round(
                ($updateOffset - $Days * 60 * 60 * 24 - $Hours * 60 * 60) / 60
            );
            $Sec =
                $updateOffset -
                $Days * 60 * 60 * 24 -
                $Hours * 60 * 60 -
                $Minutes * 60;

            Tools::save_file(
                $this->file,
                "$Days day $Hours hour $Minutes min $Sec sec"
            );
            Tools::addReason("Time changed");
        }
        $startup_servers = array();
        $gopts = "iburst";
        $lopts = "minpoll 4 maxpoll 8";
        $acopts = "nomodify notrap";
        $conf = "# Local ntp configuration\n"; # Should not be empty
        if (!$this->active) {
            $conf = "# Not active\n"; # Should not be empty
        }
        if (!$ip) {
            $ip = new IPConfig();
            $ip->loadIP();
        }

        for ($i = 1; $i <= 5; $i++) {
            $name = "ntpserver" . $i;
            $nosel = "ntpnosel" . $i;
            if (isset($this->$name) && $this->$name != "") {
                $address = $this->$name;
                if (substr($address, 0, 1) == "[") {
                    $address = 
                        $this->formatNTPServer($ip, substr($address, 1, -1));
                }
                # Do not add extra options to default / pool servers
                $sopts = $gopts;
                if (!preg_match("/pool.ntp.org\s*$/", $address)) {
                    $sopts .= " " . $lopts;
                }
                if (isset($this->$nosel) && $this->$nosel == "yes") {
                    $sopts .= " noselect";
                }
                if (!$this->active) {
                    $conf .= "#"; // comment it
                }
                $conf .= "server " . $address . " " . $sopts . "\n";
                if (!$this->active) {
                    $conf .= "#"; // comment it
                }
                $conf .=
                    "restrict " .
                    $address .
                    " mask 255.255.255.255 " .
                    $acopts .
                    "\n";
                if ($this->active) {
                    $startup_servers[] = trim($address);
                }
            }
        }

        $ok = Tools::save_file($this->fname, $conf);

        //boot time server:
        $oldFile = file_get_contents($this->fname2);
        $ok &= Tools::save_file(
            $this->fname2,
            preg_replace(
                "/^NTPSERVERS=.*$/m",
                "NTPSERVERS=" . escapeshellarg(implode(" ", $startup_servers)),
                $oldFile
            )
        );

        $pause_conf = file_get_contents($this->fname3);
        $val = $this->bootdelay ? $this->bootdelay : '';
        $pause_conf = preg_replace(
            '/^BOOTPAUSE=.*/m',
            "BOOTPAUSE=" . $val,
            $pause_conf
        );
        $ok &= Tools::save_file($this->fname3, $pause_conf);

        Tools::addReason("NTP configuration changed");
        return $ok;
    }

    public function refresh($ip) {
        $this->load();
        $this->save($ip);
    }

    function loadStats()
    {
        static $tally = array(
            ' ' => 'reject',
            'x' => 'bad', // falseticker
            '.' => 'discard', // excess
            '-' => 'outlier',
            '+' => 'candidate',
            '#' => 'selected',
            '*' => 'peer',
            'o' => 'peer',
        );
        # Codes from http://doc.ntp.org/4.2.0/debug.html
        static $refidmsg = array(
            '.ACST.' => 'anycast',
            '.AUTH.' => 'auth failed',
            '.AUTO.' => 'autokey failed',
            '.BCST.' => 'broadcast',
            '.CRYP.' => 'crypto failed',
            '.DENY.' => 'denied',
            '.DROP.' => 'dropped',
            '.RSTR.' => 'restricted',
            '.INIT.' => 'initializing',
            '.MCST.' => 'manycast',
            '.NKEY.' => 'no key',
            '.RATE.' => 'rate exceeded',
            '.RMOT.' => 'remote control',
            '.STEP.' => 'step change',
        );
        $stdout = array();
        $status = -1;
        if (PHP_OS == "WINNT") {
            $status = 0;
            $stdout = explode("\n", file_get_contents("/srv/ntpq.txt"));
            $ntpd_stat = 0;
        } else {
            exec("ntpq -n -w -c 'timeout 300' -c peers", $stdout, $status);
            if (file_exists('/var/run/ntp.pid')) {
                $ntpd_stat = stat('/var/run/ntp.pid');
            } elseif (file_exists('/var/run/ntpd.pid')) {
                $ntpd_stat = stat('/var/run/ntpd.pid');
            } else {
                $ntpd_stat = 0;
            }
        }
        $ntpd_started = $ntpd_stat ? $ntpd_stat['mtime'] : 0;
        if ($status != 0 || count($stdout) < 2) {
            if ($ntpd_started + 60 * 5 >= time() && $ntpd_started < time()) {
                $this->message =
                    "NTP service was recently restarted (" .
                    (time() - $ntpd_started) .
                    " seconds ago), please " .
                    "wait a few minutes until it gathers statistics " .
                    "and refresh this page.\n";
            } else {
                $this->message = "Failed querying local NTP service.\n";
            }
            return -1;
        }
        $this->stats = array();
        $nr_servers = 0;
        $nr_peers = 0;
        $server_ip = null;
        $server_status = null;
        for ($i = 2; $i < count($stdout); $i++) {
            $data = preg_split('/\s+/', substr($stdout[$i], 1));
            if (count($data) == 1 && substr($data[0], 0, 3) !== "===") {
                $server_status = substr($stdout[$i], 0, 1);
                $server_ip = $data[0];
                continue;
            }
            if (count($data) == 9 && $server_ip) {
                array_unshift($data, $server_ip);
            }
            if ($data[0] == "" && $server_ip) {
                $data[0] = $server_ip;
            }

            if ($server_status === null) {
                $server_status = substr($stdout[$i], 0, 1);
            }

            if (count($data) < 10) {
                $server_status = null;
                $server_ip = null;
                continue;
            }
            $server_refid = $data[1];
            $server_type = $data[3];
            if ($server_type != 'l') {
                // do not count local server
                $nr_servers++;
                $server_ip = $data[0];
                if ($server_status == '*' || $server_status == 'o') {
                    $nr_peers++;
                }
            } else {
                $server_ip = 'loopback';
            }
            $server_stratum = $data[2];
            $server_poll = $data[5];
            // the reachability register is a shifting log of the
            // successful / failed status of the last 8 polls printed out in octal
            $server_reachbits = intval($data[6], 8);
            $server_reach = 0;
            while ($server_reachbits) {
                if (($server_reachbits & 1) != 0) {
                    $server_reach++;
                }
                $server_reachbits = $server_reachbits >> 1;
            }
            $server_reach = $server_reach / 8;
            $server_delay = $data[7];
            $server_offset = $data[8];
            $server_jitter = $data[9];

            $stats = array();
            $stats['server_ip'] = $server_ip;
            $stats['stratum'] = $server_reach != 0 ? $server_stratum : "-";
            if (isset($refidmsg[$server_refid]) && $refidmsg[$server_refid]) {
                $stats['status'] = $refidmsg[$server_refid];
            } elseif ($server_reach != 0 || $server_status != ' ') {
                $stats['status'] = isset($tally[$server_status])
                    ? $tally[$server_status]
                    : 'unknown';
            } else {
                $stats['status'] = 'unreachable';
            }

            $stats['reach'] = round($server_reach * 100);
            $stats['offset'] =
                $server_reach != 0 ? round($server_offset) . " ms" : "-";
            $stats['delay'] =
                $server_reach != 0 ? round($server_delay) . " ms" : "-";
            $stats['dispersion'] =
                $server_reach != 0 ? round($server_jitter) . " ms" : "-";
            $stats['poll'] = $server_poll;

            $this->stats[] = $stats;
            $server_ip = null;
            $server_status = null;
        }

        if ($nr_peers == 0 && $this->nr_sync_servers != 0) {
            $this->message .=
                "<p><b>Warning</b>: there are currently no external NTP peers and thus " .
                "this device is currently not time synchronized.<br/>\n";

            if ($ntpd_started + 60 * 8 > time()) {
                // allow for a few minutes for ntpd to start sync
                $this->message .=
                    "The NTP service was recently restarted (" .
                    (time() - $ntpd_started) .
                    " " .
                    "seconds ago) so this may be a harmless transient situation while NTP " .
                    "gathers statistics, please refresh this page in a few minutes.<br/>\n" .
                    "However, if this situation persists, " .
                    "synchronized playback and timely display are not possible. \n";
            } else {
                $this->message .=
                    "Synchronized playback and timely display are not possible. \n";
            }
            $this->message .=
                "Most common causes are firewalls preventing NTP network traffic (UDP port " .
                "123), NTP servers which have no reference clock or have recently restarted, " .
                "or mistyped NTP server addresses.</p>";
        }
        if (!file_exists('/var/lib/spx-ntp')) {
            $drift_file = '/var/lib/ntp/drift';
        } else {
            $drift_file = '/var/lib/spx-ntp/drift';
        }
        if (file_exists($drift_file)) {
            $drift = trim(file_get_contents($drift_file));
        } else {
            $drift = "";
        }
        if ($drift === "" || $drift == 0) {
            // catches empty and non-existent drift file as well
            if ($nr_peers == 0) {
                if (is_numeric($drift) && $drift == 0) {
                    $this->message .=
                        "<p><b>Warning</b>: the local clock is calibrated with " .
                        "a dummy value of zero from the loopback reference.\n";
                } else {
                    $this->message .=
                        "<p><b>Warning</b>: the local clock is not calibrated.\n";
                }
                $this->message .=
                    "To calibrate configure at least one NTP server so that it " .
                    "is reachable and can become a peer. Calibration starts " .
                    "automatically when external peers are found and takes " .
                    "60 minutes.<br/>\n";
            } else {
                if (is_numeric($drift) && $drift == 0) {
                    $this->message .=
                        "<p><b>Warning</b>: the local clock is calibrated with " .
                        "a dummy value of zero from the loopback reference " .
                        "but calibration is ongoing,\n";
                } else {
                    $this->message .=
                        "<p><b>Warning</b>: the local clock is not calibrated but " .
                        "calibration is ongoing,\n";
                }
                $this->message .=
                    "keep this device running without " .
                    "interruption for at least 60 minutes with a reachable " .
                    "external NTP peer for calibration to complete.<br/>\n";
            }
            $this->message .=
                "An uncalibrated local clock severely degrades time synchronization and " .
                "makes the device's clock drift away from real time much more rapidly " .
                "when no NTP servers are reachable. Calibration is a one time process, " .
                "initial calibration data is kept across reboots and updated as necessary.\n";
            $this->message .= "</p>\n";
        } else {
            $this->message .=
                "<p>The local clock is calibrated (saved correction is " .
                htmlspecialchars($drift) .
                " ppm).</p>\n";
            if ($drift < -80 || $drift > 80) {
                $this->message .=
                    "<p><b>Warning</b>: the value of the saved correction is unusually large. " .
                    "This is often indicative of a malfunctioning NTP server. Verify that your NTP " .
                    "servers are working reliably. Once the NTP servers are working reliably the " .
                    "calibration will be smoothly corrected automatically, but this is a long " .
                    "process. You may restart the calibration from scratch by resetting it " .
                    "from the maintenance page.</p>\n";
            }
        }
        return $nr_servers;
    }
    function clearServers()
    {
        for ($i = 1; $i <= 5; $i++) {
            $name = "ntpserver" . $i;
            $nosel = "ntpnosel" . $i;
            $this->$name = "";
            $this->$nosel = "no";
        }
    }
}
