<?php

/**
 * This is the model class for table "Users".
 *
 * The followings are the available columns in table 'Users':
 * @property integer $id
 * @property string $username
 * @property string $password
 */
class User extends CActiveRecord
{
    public $verifyPassword;
    public $lowSecurityPassword = false;
    public $oldPassword = null;    
    public $srpPassword = null;
    public $digestPassword = null;
    public $crypted = false;
    public $userRights = null;
    
    var $passfile; # referred in pro files directly!
    var $passfileDigest;
    var $groupfile; 
    protected $pass_conf_name="/etc/raperca/spxpasswd.xml";    
    protected $srpTargetfile = "/etc/spxmanage/srpsecrets/passwd.srpv";
    protected $appacheConfigfile = "/etc/apache2/conf.d/01-spxmanage-defs.conf";
    static $digest="Player Credentials";
    
    protected $users = null;
    protected $currentUser;
    protected $updateLang =  false;
    public $validLanguage;
    
    var $needreload = false;
    
    public function init() {
        parent::init();
        if ( file_exists("/etc/spxmanage") ) {
            $this->passfile = "/etc/spxmanage/htpasswd";
            $this->passfileDigest = "/etc/spxmanage/htsecrets/htdigest";
            $this->groupfile = "/etc/spxmanage/htgroup";
        } else {
            $this->passfile = "/etc/raperca/htpasswd";
            $this->passfileDigest = "/etc/raperca/htsecrets/htdigest";
            $this->groupfile = "/etc/raperca/htgroup";
        }
        $this->validLanguage = array( 
			"auto"=>'Automatic',
			"ar"=>"العربية",
			//"cs"=>"Česky",
			//"da"=>"Dansk",
			"de"=>"Deutsch",
			"en"=>"English",
			"es"=>"Español",
			//"fi"=>"Suomi",*/
			"fr"=>"Français",
			//"he"=>"עִבְרִית",
			//"hu"=>"Magyar",
			//"id"=>"Bahasa Indonesia",
			"it"=>"Italiano",
			"ja"=>"日本語",
			"nl"=>"Dutch",            
			//"pl"=>"Polski",
            "pt" => "Português",
			"ru"=>"Pусский",
			//"se"=>"Svenska",
			"th"=>"ภาษาไทย",
			"zh-cn"=>"简体字",
			"zh-tw"=>"繁體字"
		);
    }

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return 'Users';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		// NOTE: you should only define rules for those attributes that
		// will receive user inputs.
		$rules = array(
			array('username, password', 'required'),
            array('username', 'unique'),
            array('username', 'match', 'pattern'=>'/^[^\t\n\:.][^\t\n\:]*$/', 'message' => "The username must not start with a period and must not contain tabs, line feeds or colons."),
            array('username', 'compare', 'compareValue' => 'uploader', 'operator' => '!=', 'message' => "'uploader' is a reserved username."),
            array('verifyPassword', 
                    'compare', 'compareAttribute'=>'password',
                    'except' => 'skipPassword'
            ),
            array('oldPassword', 'checkOldPass', 
                    'except' => 'skipPassword'
            ),
			array('username, password', 'length', 'max'=>64),
            array('password', 'validatePasswordStrength', 'except' => 'skipPassword'),
            array('lang', 'in', 'range'=>array_keys( $this->validLanguage ), 
                    'allowEmpty'=>true),
            // The following rule is used by search().
			// @todo Please remove those attributes that should not be searched.
			array('id, username', 'safe', 'on'=>'search'),
            array('srpPassword', 'match', 'pattern'=>'/^[A-Za-z0-9\.\/]+\t[A-Za-z0-9\.\/]+\t[A-Za-z0-9\.\/]+$/'),
            array('digestPassword', 'match', 'pattern'=>'/^[a-f0-9]+$/')
		);        
        return $rules;
	}
    
	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		// NOTE: you may need to adjust the relation name and the related
		// class name for the relations automatically generated below.
		return array(
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id' => 'ID',
            'username' => 'Username',
            'hasSRP' => 'SRP',
            'oldPassword' => 'Old Password',
            'password' => 'New Password',
            'verifyPassword'=> "Retype new Password",
            'lang'=> "Language",
            'userRights'=> "Rights",
		);
	}
    public function getSupportedlanguages( ){        
        $lang = array_slice(array_keys( $this->validLanguage ), 1);
		array_unshift( $lang, "en" );
		return $lang;
    }
    public function checkOldPass( $attribute ){
        
        $oldPass = $this->$attribute;
        if ( $oldPass === null ){
            return;
        }
        $currentUser = $this->getUser( $this->username );
        if ( !$this->verify($this->username, $oldPass, $currentUser['pass'])){
            
            $this->addError($attribute, Yii::t('yii','{attribute} is invalid.', array('{attribute}'=>$this->getAttributeLabel($attribute))));
			//$this->addError($attribute, "Old password do not match" );	
		}
    }

    public function validatePasswordStrength($attribute, $params) {
        $minLength = 8;
        $minUppercaseLetters = 1;
        $minLowercaseLetters = 1;
        $minDigits = 1;

        $punctuationMarks = '~!@#$%^&*_-+=`|\(){}[]:;"<>,.?/\'';
        $punctuationMarksSet = '~!@#$%^&*_\-+=`|\\\\(){}[\]:;"<>,.?\/\'';
        $minPunctuationMarks = 1;

        if ($this->lowSecurityPassword || $this->crypted) {
            return;
        }

        $hasInvalidChar = preg_match("/[^A-Za-z0-9{$punctuationMarksSet}]/", $this->$attribute);
        if ( $hasInvalidChar || $hasInvalidChar === false) {
            $this->addError($attribute, "The chosen password must contain only the following letters, digits and punctuation marks: A-Z, a-z, 0-9 and {$punctuationMarks}.");
        }

        if (mb_strlen($this->$attribute, Yii::app()->charset) < $minLength) {
            $this->addError($attribute, "Security alert, the chosen password must be at least {$minLength} characters long.");
        }
        if (preg_match_all('/[A-Z]/', $this->$attribute) < $minUppercaseLetters) {
            $this->addError($attribute, "Security alert, the chosen password must contain at least {$minUppercaseLetters} uppercase letter (A-Z).");
        }
        if (preg_match_all('/[a-z]/', $this->$attribute) < $minLowercaseLetters) {
            $this->addError($attribute, "Security alert, the chosen password must contain at least {$minLowercaseLetters} lowercase letter (a-z).");
        }
        if (preg_match_all('/[0-9]/', $this->$attribute) < $minDigits) {
            $this->addError($attribute, "Security alert, the chosen password must contain at least {$minDigits} digit (0-9).");
        }
        if (preg_match_all("/[{$punctuationMarksSet}]/", $this->$attribute) < $minPunctuationMarks) {
            $this->addError($attribute, "Security alert, the chosen password must contain at least {$minPunctuationMarks} punctuation mark ({$punctuationMarks}).");
        }
        if (stripos($this->$attribute, $this->username) !== false) {
            $this->addError($attribute, "Security alert, the chosen password must not contain the username ({$this->username}).");
        }
    }
    
    public function setAttributes( $values, $safeOnly=true ){
        if ( isset($values['password']) && empty($values['password']) ){
            unset($values['password']);
        }
        if ( isset($values['verifyPassword']) && empty($values['verifyPassword']) ){
            unset($values['verifyPassword']);
        }
       
        if ( isset( $values['username'] ) && !empty($values['username']) && $this->currentUser && $this->currentUser['username'] != $values['username'] ){            
            $this->password = null;
        }
        if ( !isset($values['verifyPassword']) && !isset($values['password']) ){
            $this->scenario = 'skipPassword';
            $this->crypted = true;
        }
        if ( isset( $values['lang'] ) ){
            $this->updateLang = $values['lang'] != $this->lang;
        }
        parent::setAttributes( $values, $safeOnly );
    }
    protected function afterFind()
    {
        $this->userRights = new UserRights;
        $this->userRights->id = $this->id;
        
        $this->currentUser = $this->getUser( $this->username );
        if ( isset( $this->currentUser['digestpass'] ) && $this->currentUser['digestpass']!==null && $this->currentUser['digestpass']!=="" ){ 
            $this->digestPassword = $this->currentUser['digestpass'];
        }
        parent::afterFind();
    }
    protected function afterSave()
    {
        if ( !$this->userRights ){
            $this->userRights = new UserRights;
            $this->userRights->id = $this->id;
        }      
        if ( $this->updateLang ){
            Yii::app()->user->setState('lang', $this->lang);
            Tools::$redirect = array('/controlcenter/user/profile');
        }
        parent::afterSave();
    }
    protected function afterDelete()
    {
        $this->userRights = null;
        parent::afterDelete();
    }
    protected function createSRPPassword( $username, $password ){
        $template = "SRPXXXXXX";
        $tmpdir = "--tmpdir=/tmp";
        $srptmpdir = exec("mktemp -d " . escapeshellarg($tmpdir) . " " . escapeshellarg($template));
        $srptmpfile = $srptmpdir . "/tmppass.srpv";
        
        touch( $srptmpfile);
        $STRENGTH = 3072;
        
        $USER = $username;
        $cmd = "openssl srp -verbose -srpvfile " . escapeshellarg($srptmpfile) . " -gn " . escapeshellarg($STRENGTH) . " -passout stdin -add " . escapeshellarg($USER);
        $descriptorspec = array(
            0 => array("pipe", "r"),
            1 => array("file", "/dev/null", "w"),
            2 => array("file", "/dev/null", "w")
        );

        $process = proc_open($cmd, $descriptorspec, $pipes);
        $result = false;
        
        if (is_resource($process)) {

            fwrite($pipes[0], $password);
            fclose($pipes[0]);
            
            if ( proc_close($process) == 0 ){
                $str = file_get_contents($srptmpfile);
                $data = explode( "\t", $str );
                $result = $data[1] ."\t". $data[2]."\t". $data[4];
            } else {
                syslog(LOG_WARNING, "Failed running command: $cmd");
            }            
        } else {
            syslog(LOG_WARNING, "proc_open failed on command: $cmd");
            
        }
        GlobalSettings::rrmdir( $srptmpdir, true );
        return $result;
        
    }
    public function getHasRSP() {
        if ( !$this->srpPassword ){
            $this->srpPassword = $this->loadSRP( $this->username );
        }
        return $this->srpPassword !== null;
    }
    protected function getSRPString( $username, $srp, $groups ){
        $s = explode( "\t", $srp );
        return "V\t" . $s[0] . "\t" . $s[1] . "\t" . $username . "\t" . $s[2] . "\t" . implode(":",$groups ) . "\n";
    }

    protected function loadSRP( $username ){
        if ( !file_exists($this->srpTargetfile) ){
            return null;
        }
        $data = explode("\n", file_get_contents( $this->srpTargetfile ) );
        foreach( $data as $row){
            $d = explode( "\t", $row );
            if ( count($d) == 6 && $d[0] == 'V' && $d[3] == $username){
                return $d[1] ."\t". $d[2]."\t". $d[4];                
            }            
        }     
        return null;
    }

    protected function updateSRP( $username, $srp = null, $groups = array() ){
        if ( file_exists($this->srpTargetfile) ){
            $data = explode("\n", file_get_contents( $this->srpTargetfile ) );
        } else {
            $data = array();
        }
        $result = "";
        $found = false;
        $hasUsers = false;        
        foreach( $data as $row){   
            if ( trim($row) == "" ){
                continue;
            }
            $d = explode( "\t", $row );
            if ( count($d) <= 3 || $d[0] !== 'V' || $d[3] !== $username){
                $result .= $row . "\n"; 
                if ( $d[0] == 'V' ){
                    $hasUsers = true;
                }
                continue;
            }
            $this->needreload = true;
            if ( $srp !== null ){
                // update the line                
                $result .= $this->getSRPString( $username, $srp, $groups );
                $found = true;
                $hasUsers = true;
            }
        }
        if ( !$found && $srp !== null ){            
            $result .= $this->getSRPString( $username, $srp, $groups );
            $this->needreload = true;
            $hasUsers = true;
            // enable SRP on the server
            $config = file_get_contents( $this->appacheConfigfile );
            if ( preg_match( "/^\s*Define\s+SSL_DISABLE_SRP/m", $config ) ) {
                MaintenanceJobs::saveFile( $this->appacheConfigfile, preg_replace( "/^\s*Define\s+SSL_DISABLE_SRP/m", "#Define SSL_DISABLE_SRP", $config ) );
                syslog(LOG_INFO, "Enabling SSL SRP");
                Tools::addReason("Enabling SSL SRP");
            }
        }
        
        if ( !$hasUsers ){            
            $config = file_get_contents( $this->appacheConfigfile );
            if ( preg_match( "/^#\s*Define\s+SSL_DISABLE_SRP/m", $config ) ) {
                MaintenanceJobs::saveFile( $this->appacheConfigfile, preg_replace( "/^#\s*Define\s+SSL_DISABLE_SRP/m", "Define SSL_DISABLE_SRP", $config ) );
                syslog(LOG_INFO, "Disabling SSL SRP");
                Tools::addReason("Disabling SSL SRP");
            }    
        }
        if ( !MaintenanceJobs::saveFile( $this->srpTargetfile, $result ) ) {
            syslog(LOG_WARNING, "failed saving: $this->srpTargetfile");
        }
    }
    protected function afterValidate()
    {
        if ( !$this->hasErrors() ){
            if ( !$this->crypted ) {
                $this->srpPassword = $this->createSRPPassword( $this->username, $this->password);
                $this->digestPassword = md5($this->username.":".self::$digest.":". $this->password);
                $this->password = self::encrypt( $this->username, $this->password );
                $this->verifyPassword = "";
            }
            $this->crypted = true;
        }
        parent::afterValidate();
    }
    public function updateHttpdSecurity() {

        if ( $this->currentUser && $this->scenario == 'skipPassword' ){
            $password = $this->currentUser['pass'];
            $digestPassword = $this->currentUser['digestpass'];
            $srpPassword = $this->loadSRP( $this->username );
        } else {
            $password = $this->password;
            $digestPassword = $this->digestPassword;
            $srpPassword = $this->srpPassword;
        }
        if ( $this->userRights === null || ( $this->currentUser && $this->currentUser['username'] != $this->username ) ) {
            $this->clearUser( $this->currentUser['username'] );
            $this->updateSRP( $this->currentUser['username'] );
            
        }        
        if ( $this->userRights !== null ){            
            $this->addUser( $this->username, $password, $this->userRights->getUserRights(), true, $digestPassword);
            $this->updateSRP( $this->username, $srpPassword, $this->userRights->getActiveGroups() );
        }

        return $this->saveUsers();
    }
    
    static function encrypt( $username, $password ){ 
        $info = new BasicInfo();
        $fw = explode( ".", $info->getFirmware() );
        if ( $fw[0] > 3 && defined("CRYPT_BLOWFISH") && CRYPT_BLOWFISH ){
            $salt = '$2y$07$' . self::getRandomString(22) . '$';
        } else {
            $salt = '$1$' . self::getRandomString(8) . '$';
        }
        return crypt($password, $salt);
    }
    static function verify( $username, $password, $hash ){         
        return $hash == crypt($password, $hash);
    }
	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 *
	 * Typical usecase:
	 * - Initialize the model fields with values from filter form.
	 * - Execute this method to get CActiveDataProvider instance which will filter
	 * models according to data in model fields.
	 * - Pass data provider to CGridView, CListView or any similar widget.
	 *
	 * @return CActiveDataProvider the data provider that can return the models
	 * based on the search/filter conditions.
	 */
	public function search()
	{
		// @todo Please modify the following code to remove attributes that should not be searched.

		$criteria=new CDbCriteria;

		$criteria->compare('id',$this->id);
		$criteria->compare('username',$this->username,true);

		return new CActiveDataProvider($this, array(
			'criteria'=>$criteria,
		));
	}

	/**
	 * Returns the static model of the specified AR class.
	 * Please note that you should have this exact method in all your CActiveRecord descendants!
	 * @param string $className active record class name.
	 * @return User the static model class
	 */
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}
    
    function clearAllUsers() {
        // remove all users except admin        
        $users = self::model()->findAll();
        foreach ( $users as $user ){
            if ( $user->id > 1 ){
                $this->clearUser( $user->username );
                $this->updateSRP( $user->username );
                $user->delete();
            }            
        }
        return $this->saveUsers();        
    }
    function getUsers( ){
        
        $this->users=array();
        
        if ( file_exists($this->passfile) ){
            $groups = array();
            if ( file_exists($this->groupfile) ){
                $groupFile = file_get_contents($this->groupfile);
            } else {
                $groupFile = "";
            }
            $groupsStr = explode("\n",$groupFile);
            foreach ( $groupsStr as $groupStr) {
                $data = explode( ": ", $groupStr );
                if ( count($data) == 2 ){
                    $groups[$data[0]] = explode( " ", $data[1] );
                }
            }
            if ( file_exists($this->passfile) ){
                $passwordsFile = file_get_contents($this->passfile);
            } else {
                $passwordsFile = "";
            }
            $passwords = explode("\n",$passwordsFile);
            $passelements = array();
            
            foreach ($passwords as $line){
                if (preg_match("/(.*):(.*)/", $line, $passelements)){
                    $user = array(
                        'username'=>$passelements[1], 
                        'pass'=>$passelements[2], 
                        'groups' => array()
                    );
                    foreach ( $groups as $group => $items ){
                        if (in_array( $user['username'], $items) ){
                            $user['groups'][$group] = 1;
                        } else {
                            $user['groups'][$group] = 0;
                        }
                    }
                    $this->users[] = $user;
                }            
            }            
        }
        if ( file_exists($this->passfileDigest) ){
            $passwords=file_get_contents($this->passfileDigest);
            $passwords=explode("\n",$passwords);            
            foreach ($passwords as $line){            
                if (preg_match("/(.*):(.*):(.*)/", $line, $passelements)){
                    foreach ( $this->users as &$user ){
                        if ( $user['username'] == $passelements[1] ){
                            $user['digestpass'] = $passelements[3];
                        }
                    }                    
                }                
            }
        }        
        return $this->users;
    }
    
    static function getRandomString( $length ) {
        $bytes = "";
        do {
            $bytes .= random_bytes( $length );
            $secret = strtr( base64_encode( $bytes ), array( "/"=>"", "+"=>"", "="=>"" ) );
        } while ( strlen( $secret ) < $length );

        return substr( $secret, 0, $length );
    }
    
    function saveUsers( ){
        if ( $this->users === null ){
            $this->getUsers();
        }                    
        $xlinkNS = 'http://www.w3.org/1999/xlink';
        
        $pass_config = new DOMDocument;
        if ( file_exists($this->pass_conf_name) ){
            $pass_config->loadXML(file_get_contents($this->pass_conf_name));
        }
        $parent = $pass_config->getElementsByTagName('credits');
        if ($parent->length<1) {
            $parent = $pass_config->appendChild($pass_config->createElement('credits'));
        } else {
            $parent=$parent->item(0);
        }
        
        $str="";
        $strDigest = "";
        $groupStr = array();
        
        // add a random password for the uploader to be able to access the content, the logs and the interface
        $username = 'uploader';
        $pass = "";
        $cred = $pass_config->getElementById("admin");
        if ( $cred ) {
            $pass = $cred->getAttribute("password");
        }
        if ( $pass == "" ){
            $pass = self::getRandomString(16);
        }
        $this->addUser( $username, $pass, array( 'admin' => true ) );
        
        foreach( $this->users as $user ){
            $str .= $user['username'].":".$user['pass']."\n";      
            if ( isset($user['digestpass']) && $user['digestpass'] !== null){
                $strDigest .= $user['username'].":".self::$digest.":".$user['digestpass']."\n";      
            }
            if ( isset($user['groups']) ){
                foreach ( $user['groups'] as $group => $active ){
                    if ( $active ) {
                        if ( !isset( $groupStr[$group] )){
                            $groupStr[$group] = "$group:";
                        }
                        $groupStr[$group] .= " ".$user['username'];
                    }
                }
            }
        }
        
        if ( empty($str) ){
            if ( file_exists( $this->passfile) )
                unlink( $this->passfile );
            if ( file_exists( $this->passfileDigest) )
                unlink( $this->passfileDigest );
            if ( file_exists( $this->groupfile) )
                unlink( $this->groupfile );
            // remove destination file
            $ret = MaintenanceJobs::setSafeModeData( "htpasswd", null ) && MaintenanceJobs::setSafeModeData( "htgroup", null );
            return $ret;
        } else {
            $ret = MaintenanceJobs::saveFile( $this->passfile, $str, true, 'www-data' );
            if ( $ret=== true ){
                if ( !MaintenanceJobs::saveFile( $this->groupfile, implode("\n", $groupStr ) ) ) {
                    $ret = false;
                    $this->addError( 'password', "Failed setting password groups, please set again." );
                }
                if ( !MaintenanceJobs::saveFile( $this->passfileDigest, $strDigest, true, 'www-data' ) ) {
                    $ret = false;
                    $this->addError( 'password', "Failed setting digest passwords, please set again." );
                }
            } else {
                $this->addError( 'password', "Failed setting password, please set again." );
            }
            // save data in the XML file also        
            if ( $ret===true ) {    
                $id_set = array(
                    "admin"=>'http://localhost/', 
                    "content"=>'http://localhost:81/', 
                    "admin2"=>'http://127.0.0.1/', 
                    "content2"=>'http://127.0.0.1:81/'
                );
                foreach ($id_set as $id=>$dest ) {
                    $cred = $pass_config->getElementById($id);
                    if ( !$cred ) {
                        $cred=$pass_config->createElement('creditentials');
                        $parent->appendChild($cred);
                    } 
                    $cred->setAttribute("username", $username);
                    $cred->setAttribute("password", $pass);
                    $cred->setAttributeNS($xlinkNS,"xlink:href", $dest);
                    $cred->setAttribute("xml:id", $id);
                } 
                if ( !MaintenanceJobs::saveFile($this->pass_conf_name, $pass_config->saveXML(), true) ) {
                    $ret = false;
                    $this->addError( 'password', "Failed setting password credentials, please set again." );
                }      
            }
            if ( $ret===true ) {
                if ( ! ( MaintenanceJobs::setSafeModeData( "htpasswd", $this->passfile ) && MaintenanceJobs::setSafeModeData( "htgroup", $this->groupfile )) ) {
                    $this->addError( 'password', "Failed setting password for recovery console, please set again." );
                    return false;
                }
            }
            return $ret;
        }
        
    }
    function clearUser( $name ) {
        if ( $this->users === null ){
            $this->getUsers();
        }
        foreach ( $this->users as $idx => $user ){
            if ( $user['username'] == $name ){
                unset( $this->users[$idx] );
            }
        }
        return $this->users;
    }
    function getUser( $name, &$idx=0 ){
        if ( $this->users === null ){
            $this->getUsers();
        }
        $currentUser = false;
        foreach( $this->users as $idx => $user){
            if ( $user['username'] == $name ) {
                $currentUser = $user;
                break;
            }
        }
        return $currentUser;
    }
    function addUser( $username, $newpass, $groups = null, $encripted=false, $digestPassword=null){
        if ( $this->users === null ){
            $this->getUsers();
        }
        $idx=-1;
	    $currentUser = $this->getUser($username, $idx);

        if (!$currentUser){
            $currentUser = array();
            $currentUser['username'] = $username;            
            $idx=-1;
        } 
        if ( $groups ){                
            $currentUser['groups'] = $groups;
        }
        if ( $newpass ) {
            if ( $encripted ) {
                $currentUser['pass'] = $newpass;
                $currentUser['digestpass'] = $digestPassword;
            } else {
                $currentUser['clearpass']=$newpass;
                $currentUser['pass'] = self::encrypt($currentUser['username'], $newpass);    
                $currentUser['digestpass'] = md5($currentUser['username'].":".self::$digest.":".$newpass);            
            }
        }
        if ($idx>=0)
            $this->users[$idx] = $currentUser;
        else
            $this->users[] = $currentUser;
        return $this->users;    
    }
        
    function crypt_apr1_md5($plainpasswd, $salt="") {
        if ($salt=="")
            $salt = substr(str_shuffle("abcdefghijklmnopqrstuvwxyz0123456789"), 0, 8);
        else 
            $salt = substr($salt,6,8);
     
        $len = strlen($plainpasswd);
        $text = $plainpasswd.'$apr1$'.$salt;
        $bin = pack("H32", md5($plainpasswd.$salt.$plainpasswd));
        for($i = $len; $i > 0; $i -= 16) { $text .= substr($bin, 0, min(16, $i)); }
        for($i = $len; $i > 0; $i >>= 1) { $text .= ($i & 1) ? chr(0) : $plainpasswd[0]; }
        $bin = pack("H32", md5($text));
        for($i = 0; $i < 1000; $i++) {
            $new = ($i & 1) ? $plainpasswd : $bin;
            if ($i % 3) $new .= $salt;
            if ($i % 7) $new .= $plainpasswd;
            $new .= ($i & 1) ? $bin : $plainpasswd;
            $bin = pack("H32", md5($new));
        }
        $tmp="";
        for ($i = 0; $i < 5; $i++) {
            $k = $i + 6;
            $j = $i + 12;
            if ($j == 16) $j = 5;
            $tmp = $bin[$i].$bin[$k].$bin[$j].$tmp;
        }
        $tmp = chr(0).chr(0).$bin[11].$tmp;
        $tmp = strtr(strrev(substr(base64_encode($tmp), 2)),
        "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/",
        "./0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz");
        return "$"."apr1"."$".$salt."$".$tmp;
    }
    function reloadConfig() {
        if ($this->needreload) {           
            Tools::reloadHttpd();
        }            
    }   
}
