/*!
 * jSignage.Social
 * http://www.spinetix.com/
 * Copyright SpinetiX S.A.
 * Released under the GPL Version 2 license.
 *
 * $Date: 2025-01-30 00:02:09 +0000 (Thu, 30 Jan 2025) $
 * $Revision: 39193 $
 */

(function () {
var version = new String("1.4.0");
version.major = 1;
version.minor = 4;
version.revision = 0;

function uwq(uri, qs) {
    var first = true, str = '';
    for (var q in qs) {
        if (qs[q] === undefined)
            continue;
        if (first)
            uri += '?';
        else
            uri += '&';
        first = false;
        uri += encodeURIComponent(q) + '=' + encodeURIComponent(qs[q]);
    }
    return uri;
}

function get(json, path) {
    if (!jSignage.isArray(path))
        path = path.split('.');
    var x = json;
    for (var i = 0; i < path.length; i++) {
        var key = path[i];
        if (!x || typeof (x) != 'object')
            return null;
        if (key in x)
            x = x[key];
        else
            return null;
    }
    return x;
}

function parse_json(text, report) {
    var json;
    try {
        json = jSignage.parseJSON(text);
    } catch (e) {
        report.error = '500 ' + e;
        return null;
    }
    if (!json || typeof (json) != 'object') {
        report.error = '500 Malformed json';
        return null;
    }
    return json;
}

var reStripTags = /<\/?([A-Za-z][A-Za-z0-9]*(?::[A-Za-z][A-Za-z0-9]*)?)[^>]*>/g;
var reSection = /^br|p|div|h[1-9]$/i;
var reMultipleSpaces = /[ \f\t\v\u00a0\u1680\u180e\u2000-\u200a\u2028\u2029\u202f\u205f\u3000\ufeff]+/g;
var reURLs = /(?:^|\s)https?\:\/\/\S*/g;
var reEntities = /&(quot|amp|apos|lt|gt|nbsp);/g;
var reNewLines = /[\r\n]+/g;

function cleanup_spaces(text) {
    var t = text.replace(reNewLines, '\n');
    t = t.replace(reMultipleSpaces, ' ');
    return t.trim();
}

function cleanup_text(text) {
    var t = text.replace(reStripTags, function (match, tag) {
        return reSection.test(tag) ? ' ' : '';
    });
    t = t.replace(reURLs, '');
    t = t.replace(reEntities, function (match, p1) {
        switch (p1) {
            case 'quot': return '"';
            case 'amp': return '&';
            case 'apos': return '\'';
            case 'lt': return '<';
            case 'gt': return '>';
            default: return ' ';
        }
    });
    return cleanup_spaces(t);
}

function forex_ecb(base, currency) {
    return {
        type: 'uri',
        src: 'https://www.ecb.europa.eu/stats/eurofxref/eurofxref-daily.xml',
        parser: [{
            type: 'xml',
            rows: 'Envelope > Cube > Cube > Cube',
            columns: [
                {
                    'name': 'rate',
                    'attr': 'rate'
                }, {
                    'name': 'currency',
                    'attr': 'currency'
                }
            ]
        }, {
            type: 'transform',
            inline: function (table) {
                var rates = { 'EUR': 1 };
                for (var i = 0; i < table.length; i++)
                    rates[table[i].currency] = Number(table[i].rate);
                var base_rate = rates[base] || 0;
                var base_sign = jSignage.currency.getPortableCurrencySign(base);
                table = [];
                for (var i = 0; i < currency.length; i++) {
                    var unit = currency[i];
                    table.push({
                        currency: unit,
                        rate: (unit in rates ? rates[unit] : NaN) / base_rate,
                        sign: jSignage.currency.getPortableCurrencySign(unit),
                        base: base,
                        base_currency: base,
                        base_sign: base_sign
                    });
                }
                return table;
            }
        }]
    };
}

var feedsAPIKey = '';

var financeAPI = 'https://feeds.services.spinetix.com/v1/finance';
if ( navigator.spxCloudStage && navigator.spxCloudStage !== 'prod' )
	financeAPI = 'https://feeds-staging.services.spinetix.com/' + navigator.spxCloudStage + '/finance';

function finance_get_result( text, report ) {
	var json = parse_json( text, report );
	if ( !json )
		return [];
	if ( json.status !== 200 ) {
		report.error = String( json.status ) + ' ' + json.message;
		return [];
	}
	if ( !json.data || !jSignage.isArray( json.data ) ) {
		report.error = '418 Missing data';
		return [];
	}
	return json.data;
}

function forex_alphavantage( base, currencies, report ) {
	// prevent injection
	currencies = currencies.map( function ( c ) {
		return c.replace( /[^A-Za-z0-9]/g, '' );
	} );
	base = base.replace( /[^A-Za-z0-9]/g, '' );
	var base_sign = jSignage.currency.getPortableCurrencySign( base );
	var pairs = currencies.map( function ( to ) {
		return base + '/' + to;
	} );
	var qs = {};
	if ( pairs.length === 1 ) {
		qs.p = pairs[0];
	} else {
		for ( var i = 0; i < pairs.length; i++ )
			qs['p['+i+']'] = pairs[i];
	}

    var feed = {
        type: 'uri',
        src: uwq( financeAPI + '/forex', qs ),
        parser: [ {
            type: 'custom',
            inline: function ( text ) {
                return finance_get_result( text, report );
            }
        }, {
            type: 'custom',
            src: 'rate',
            inline: function ( rate, row ) {
                row.sign = jSignage.currency.getPortableCurrencySign( row.currency );
                row.base_currency = row.base;
                row.base_sign = base_sign;
                row.date = new Date( row.date );
                return row;
            }
        } ]
    };

    if ( feedsAPIKey ) {
        feed.headers = {
            'x-api-key': feedsAPIKey,
        };
    }

    return feed;
}

function finance_alphavantage( symbols, names, report ) {
	// prevent injection
	symbols = symbols.map( function ( s ) {
		return s.replace( /[^A-Z0-9.=^-]/g, '' );
	} );

	var qs = {};
	if ( symbols.length == 1 ) {
		qs.s = symbols[0];
		if ( names[0] )
			qs.n = names[0];
	} else {
		for ( var i = 0; i < symbols.length; i++ ) {
			qs['s['+i+']'] = symbols[i];
			if ( names[i] )
				qs['n['+i+']'] = names[i];
		}
	}

	var feed = {
		type: 'uri',
		src: uwq( financeAPI + '/quote', qs ),
		parser: {
			type: 'custom',
			inline: function ( text ) {
				return finance_get_result( text, report );
			}
		}
    };

    if ( feedsAPIKey ) {
        feed.headers = {
            'x-api-key': feedsAPIKey,
        };
    }

    return feed;
}

var reIsFileName = /^(.*)\[(?:image|video|audio)\]$/;
var reSplitURL = /^([^:]+:\/\/[^\/]*\/)([^\?#]*)/;

function table2d_to_feed(cells, transpose, has_headers, colnames, baseURL, suffix) {
    if (!jSignage.isArray(cells))
        return [];
    var rows = cells.length, columns = 0, i, j;
    for (i = 0; i < rows; i++) {
        if (!jSignage.isArray(cells[i]))
            return [];
        if (cells[i].length > columns)
            columns = cells[i].length;
    }
    for (i = 0; i < rows; i++)
        if (cells[i].length < columns)
            for (j = cells[i].length; j < columns; j++)
                cells[i].push('');

    if (transpose) {
        var t = [];
        for (j = 0; j < columns; j++)
            t[j] = [];
        for (i = 0; i < rows; i++)
            for (j = 0; j < columns; j++)
                t[j][i] = cells[i][j];
        cells = t;
        t = rows;
        rows = columns;
        columns = t;
    }

    var names = [];
    for (i = 0; i < columns; i++) {
        if (i < 26) {
            names[i] = String.fromCharCode(i + 65);
        } else {
            names[i] = String.fromCharCode(i % 26 + 65);
            for (var n = Math.floor(i / 26); n > 0; n = Math.floor((n - 1) / 26))
                names[i] = String.fromCharCode((n - 1) % 26 + 65) + names[i];
        }
    }
    if (has_headers && rows >= 1) {
        var headers = cells.shift();
        --rows;
        for (i = 0; i < columns; i++)
            if (headers[i] || headers[i] === 0)
                names[i] = '' + headers[i].trim();
    }
    if (colnames) {
        for (i = 0; i < colnames.length; i++) {
            var c = colnames[i];
            if (c.number >= 1 && c.number <= columns && (c.name || c.name === 0))
                names[c.number - 1] = c.name;
        }
    }

    var table = [];
    for (i = 0; i < rows; i++) {
        var row = {};
        for (j = 0; j < columns; j++)
            row[names[j]] = cells[i][j];
        table.push(row);
    }

    if (baseURL) {
        var url, host, path;
        for (j = 0; j < columns; j++) {
            var cn = reIsFileName.exec(names[j]);
            if (cn) {
                cn = cn[1];
                if (!url) {
                    url = reSplitURL.exec(baseURL);
                    if (!url)
                        break;
                    host = url[1];
                    path = url[2].split('/');
                    path.pop();
                }
                for (i = 0; i < rows; i++) {
                    var fn = table[i][names[j]];
                    var abs = reSplitURL.exec(fn);
                    if (abs) {
                        table[i][cn] = fn + (suffix || '');
                    } else {
                        var fpath;
                        if (fn.charAt(0) == '/' || fn.charAt(0) == '\\') {
                            fpath = fn.split(/[\/\\]/).map(encodeURIComponent);
                        } else {
                            fpath = path.concat(fn.split(/[\/\\]/).map(encodeURIComponent));
                            for (var k = path.length; k < fpath.length; k++) {
                                if (fpath[k] == '.') {
                                    fpath.splice(k, 1);
                                    --k;
                                } else if (fpath[k] == '..') {
                                    if (k > 0) {
                                        fpath.splice(k - 1, 2);
                                        k -= 2;
                                    } else {
                                        fpath.splice(k, 1);
                                        --k;
                                    }
                                }
                            }
                        }
                        table[i][cn] = host + fpath.join('/') + (suffix || '');
                    }
                }
            }
        }
    }

    return table;
}

function csv_spreadsheet(uri, separator, quotes, range, transpose, headers, colnames) {
    return {
        type: 'uri',
        src: uri,
        parser: {
            type: 'custom',
            inline: function (text) {
                var json = jSignage.parseCSV(text, separator, 'json', false, quotes, range);
                if (!json)
                    return [];
                return table2d_to_feed(json, transpose, headers, colnames, uri);
            }
        }
    };
}

function xls_spreadsheet(workbook, worksheet, range, data, transpose, headers, colnames) {
    return {
        type: 'inline',
        data: data || [],
        parser: {
            type: 'custom',
            inline: function (json) {
                if (window.parseXLS) {
                    var r = window.parseXLS(workbook, worksheet, range);
                    if (r)
                        json = r;
                }
                if (!jSignage.isArray(json))
					json = [];
				for ( var i = 0; i < json.length; i++ ) {
					var row = json[i];
					if ( jSignage.isArray( json ) ) {
						for ( var j = 0; j < row.length; j++ ) {
							if ( row[j] && typeof ( row[j] ) == 'object' ) {
								if ( row[j].type == 'date' && typeof ( row[j].value ) == 'number' ) {
									var d = new Date( row[j].value );
									row[j] = new Date( d.getUTCFullYear(), d.getUTCMonth(), d.getUTCDate(), d.getUTCHours(), d.getUTCMinutes(), d.getUTCSeconds(), d.getUTCMilliseconds() );
								}
							}
						}
					}
				}
                return table2d_to_feed(json, transpose, headers, colnames, document.documentURI);
            }
        }
    };
}

function xls_chart(width, height, workbook, worksheet, chart, data) {
    return {
        type: 'inline',
        data: data || [],
        parser: {
            type: 'custom',
            inline: function (json) {
                if (window.parseXLSChart) {
                    var r = window.parseXLSChart(width, height, workbook, worksheet, chart);
                    if (r)
                        json = r;
                }
                if (!jSignage.isArray(json))
                    json = [];
                return json;
            }
        }
    };
}

function ics_calendar(args) {
    return {
        type: 'uri',
        src: args.calendar,
        parser: {
            type: 'ical',
            startDate: args.startDate || null,
            endDate: args.endDate || null,
            dur: args.dur || null,
            maxItems: args.count || 0
        }
    };
}

function webdav_photos(args) {
    var feed = {
        type: 'uri',
        src: args.uri,
        parser: [{
            type: 'dir',
            hidden: false,
            filter: args.filter || null,
            resourcetype: 'file'
        }, {
            type: 'transform',
            inline: function (table) {
                var r = [];
                if (table) for (var i = 0; i < table.length; i++) {
                    var row = table[i];
                    r.push({
                        title: row.filename,
                        description: '',
                        href: row.href,
                        owner: '',
                        profileImage: '',
                        tags: '',
                        taken: row.creationdate,
                        posted: row.getlastmodified,
                        place: '',
                        longitude: NaN,
                        latitude: NaN,
                        likes: 0,
                        liked: 0
                    });
                }
                return r;
            }
        }]
    };

    if (args.sort) {
        if (args.sort == 'alpha') {
            feed.sort = {
                column: 'title',
                type: 'string'
            };
        } else if (args.sort == 'modified') {
            feed.sort = {
                column: 'posted',
                type: 'date',
                reverse: true
            };
        } else if (args.sort == 'created') {
            feed.sort = {
                column: 'taken',
                type: 'date',
                reverse: true
            };
        }
    }

    if (args.count)
        feed.maxCount = args.count;

    return feed;
}

var googleSheetsAPI = 'https://sheets.googleapis.com/v4';

function google_get_result(text, array, report) {
    var json = parse_json(text, report);
    if (!json)
        return null;
    var result = get(json, array);
    if (!result || !jSignage.isArray(result)) {
        report.error = (get(json, 'error.code') || 418) + ' ' + (get(json, 'error.message') || 'Missing results');
        return null;
    }
    return result;
}

function google_spreadsheet(feed, workbook_id, sheet_id, range, transpose, headers, colnames, report) {
    var qs = {
        fields: 'sheets(data.rowData.values(effectiveValue,effectiveFormat.numberFormat.type))'
    };

    var data = {
        dataFilters: [{
            gridRange: {
                sheetId: Number(sheet_id)
            }
        }],
        includeGridData: true
    };

    if (range !== null) {
        function decodeXLSCellName(cell) {
            var x = -1, y = 0, i = 0;
            for (; i < cell.length && cell.charCodeAt(i) >= 65 && cell.charCodeAt(i) <= 90; i++)
                x = (x + 1) * 26 + (cell.charCodeAt(i) - 65);
            for (; i < cell.length && cell.charCodeAt(i) >= 48 && cell.charCodeAt(i) <= 57; i++)
                y = y * 10 + (cell.charCodeAt(i) - 48);
            --y;
            if (x < 0 || y < 0 || i != cell.length)
                return null;
            return [x, y];
        }

        var colon = range.indexOf(':'), topLeft, bottomRight;
        if (colon < 0) {
            topLeft = bottomRight = decodeXLSCellName(range);
        } else {
            topLeft = decodeXLSCellName(range.substring(0, colon));
            bottomRight = decodeXLSCellName(range.substring(colon + 1));
        }
        if (topLeft && bottomRight && topLeft[0] <= bottomRight[0] && topLeft[1] <= bottomRight[1]) {
            data.dataFilters[0].gridRange.startColumnIndex = topLeft[0];
            data.dataFilters[0].gridRange.endColumnIndex = bottomRight[0] + 1;
            data.dataFilters[0].gridRange.startRowIndex = topLeft[1];
            data.dataFilters[0].gridRange.endRowIndex = bottomRight[1] + 1;
        }
    }

    return {
        type: 'uri',
        src: uwq(googleSheetsAPI + '/spreadsheets/' + encodeURIComponent(workbook_id) + ':getByDataFilter', qs),
        requestType: 'POST',
        data: JSON.stringify(data),
        contentType: 'application/json',
        parser: {
            type: 'custom',
            inline: function (text) {
                var rowData = google_get_result(text, 'sheets.0.data.0.rowData', report);
                if (!rowData)
                    return [];
                var cells = [];
                for (var i = 0; i < rowData.length; i++) {
                    if (typeof (rowData[i]) == 'object' && jSignage.isArray(rowData[i].values)) {
                        var row = [], values = rowData[i].values;
                        for (var j = 0; j < values.length; j++) {
                            if (typeof (values[j]) == 'object' && typeof (values[j].effectiveValue) == 'object') {
                                var v = values[j].effectiveValue;
								if ( 'stringValue' in v ) {
									while ( row.length < j )
										row.push( '' );
									row.push( v.stringValue );
                                } else if ('numberValue' in v) {
									while ( row.length < j )
										row.push( '' );
                                    var t = get(values[j], 'effectiveFormat.numberFormat.type');
									if ( t == 'DATE' || t == 'TIME' || t == 'DATE_TIME' ) {
										var d = new Date( ( v.numberValue - 25569 ) * 86400000 );
                                        row.push(new Date(d.getUTCFullYear(), d.getUTCMonth(), d.getUTCDate(), d.getUTCHours(), d.getUTCMinutes(), d.getUTCSeconds(), d.getUTCMilliseconds()));
                                    } else {
                                        row.push(v.numberValue);
									}
                                }
                            }
						}
						if ( row.length > 0 ) {
							while ( cells.length < i )
								cells.push( [''] );
							while ( row.length < values.length )
								row.push( '' );
							cells.push( row );
						}
                    }
                }
                return table2d_to_feed(cells, transpose, headers, colnames);
            }
        }
    };
}

var googleCalendarAPI = 'https://www.googleapis.com/calendar/v3';
var googleCalendarBatchAPI = 'https://www.googleapis.com/batch/calendar/v3';
var googleBoundary = 'batch_A5kZdO7IPV4M_yC8G1a20Oquj';
var reAllDay = /^(\d{4})-(\d{2})-(\d{2})$/;

function makeHTTPMultiPart( gets ) {
	var mp = '';
	for ( var i = 0; i < gets.length; i++ ) {
		var get = 'GET ' + gets[i] + '\r\n\r\n';
		if ( mp.length > 0 )
			mp += '\r\n';
		mp += '--' + googleBoundary + '\r\n';
		mp += 'Content-Type: application/http\r\n';
		mp += 'Content-Length: ' + get.length + '\r\n';
		mp += 'Content-ID: ' + ( i + 1 ) + '\r\n';
		mp += '\r\n';
		mp += get;
	}
	mp += '\r\n--' + googleBoundary + '--\r\n';
	return mp;
}

function parseMultiPart( response ) {
    var parts = [], start = 0;
    if ( response.substr( 0, 2 ) === '\r\n' )
        start = 2;
	if ( response.substr( start, 2 ) !== '--' )
		return parts;
	var i = response.indexOf( '\r\n', start );
	if ( i === -1 )
		return parts;
	var boundary = response.substring( start, i );
	while ( true ) {
		start += boundary.length;
		if ( response.substr( start, 2 ) === '--' )
			break;
		if ( response.substr( start, 2 ) !== '\r\n' )
			break;
		start += 2;
		var end = response.indexOf( boundary, start );
		if ( end === -1 || response.substr( end - 2, 2 ) !== '\r\n' )
			break;
		parts.push( response.substring( start, end - 2 ) );
		start = end;
	}
	return parts;
}

function parseHTTPPart( part ) {
	var response = {
		httpStatus: 0,
		headers: {},
		content: ''
	};

	var i = 0, j;
	for ( j = part.indexOf( '\r\n', i ); j !== -1; j = part.indexOf( '\r\n', i ) ) {
		if ( j === i ) {
			i += 2;
			break;
		}
		var header = part.substring( i, j );
		i = j + 2;
		var colon = header.indexOf( ':' );
		if ( colon === -1 ) {
			response.headers[header.toLowerCase()] = '';
		} else {
			var name = header.substring( 0, colon ).toLowerCase();
			++colon;
			while ( colon < header.length && ( header[colon] === ' ' || header[colon] === '\t' ) )
				++colon;
			response.headers[name] = header.substr( colon );
		}
	}

	if ( response.headers['content-type'] !== 'application/http' )
		return response;

	if ( part.substr( i, 9 ) === 'HTTP/1.1 ' ) {
		i += 9;
		j = part.indexOf( ' ', i );
		if ( j !== -1 ) {
			response.httpStatus = parseInt( part.substring( i, j ) );
			if ( isFinite( response.httpStatus ) && response.httpStatus > 0 ) {
				j = part.indexOf( '\r\n\r\n', i );
				if ( j !== -1 )
					response.content = part.substr( j );
			}
		}
	}

	return response;
}

function parseGooglePart( part, responses ) {
	var http = parseHTTPPart( part );
	var id = http.headers['content-id'];
	if ( id && id.startsWith( 'response-' ) ) {
		var idx = parseInt( id.substr( 9 ), 10 );
		if ( isFinite( idx ) && idx > 0 )
			responses[idx - 1] = http;
	}
}

function googleCalendarQuery( calendar_id, from, to ) {
	var qs = {
		maxResults: 2500,
		orderBy: 'startTime',
		singleEvents: true,
		timeMin: from.toISOString(),
		timeMax: to.toISOString(),
		timeZone: 'UTC',
        fields: 'items(summary,description,start,end,location,created,updated,organizer/displayName,recurringEventId,status,visibility,attendees(email,displayName),colorId,attachments(mimeType,fileId))'
	};

	return uwq( googleCalendarAPI + '/calendars/' + encodeURIComponent( calendar_id || 'primary' ) + '/events', qs );
}

function googleCalendarEvent( args, x, num, calendarId, calendarName ) {
	var event = {};
	if ( num > 0 ) {
		event.calendarName = calendarName;
		event.calendarId = calendarId;
		event.calendarNum = num;
	}
	event.title = x.summary || '';
	event.description = x.description || '';
	var startDay = get( x, 'start.date' );
	if ( startDay ) {
		event.allDay = 1;
		startDay = reAllDay.exec( startDay );
		if ( startDay )
			event.startDate = new Date( startDay[1], Number( startDay[2] ) - 1, startDay[3] );
		else
			event.startDate = new Date( NaN );
		var endDay = reAllDay.exec( get( x, 'end.date' ) );
		if ( endDay )
			event.endDate = new Date( endDay[1], Number( endDay[2] ) - 1, endDay[3] );
		else
			event.endDate = new Date( NaN );
	} else {
		event.allDay = 0;
		event.startDate = new Date( get( x, 'start.dateTime' ) );
		event.endDate = new Date( get( x, 'end.dateTime' ) );
	}
	event.location = x.location || '';
	event.created = new Date( x.created );
	event.modified = new Date( x.updated );
	event.organizer = get( x, 'organizer.displayName' ) || '';
	event.recuring = 'recurringEventId' in x ? 1 : 0;
	event.cancelled = x.status === 'cancelled' ? 1 : 0;
	event.private = x.visibility === 'private' ? 1 : 0;
	var guests = [];
	if ( jSignage.isArray( x.attendees ) ) {
		for ( var j = 0; j < x.attendees.length; j++ ) {
			var y = x.attendees[j];
			if ( y && typeof ( y ) === 'object' ) {
				if ( y.displayName )
					guests.push( y.displayName );
				else if ( y.email )
					guests.push( y.email );
			}
		}
	}
	event.guests = guests.join( ', ' );
    event.color = x.colorId || '';

    var medias = [];
    if ( jSignage.isArray( x.attachments ) ) {
        for ( var k = 0; k < x.attachments.length; k++ ) {
            var a = x.attachments[k], url;
            if ( a.fileId ) {
                url = 'https://' + encodeURIComponent( args.account ) + ',cached,' + encodeURIComponent( a.mimeType || '' ) + '@www.googleapis.com/drive/v3/files/' + encodeURIComponent( a.fileId ) + '?alt=media';
                medias.push( url );
            }
        }
    }
    if ( medias.length > 0 ) {
        event.enclosure = medias[ 0 ];
        event.enclosures = medias.join( ' ' );
    } else {
        event.enclosure = event.enclosures = '';
    }

    return event;
}

function google_calendar( feed, calendar_id, from, to, args, report ) {
	var count = args.count || 0;
	if ( jSignage.isArray( calendar_id ) ) {
		var calendar_name = args.name || '';
		if ( !jSignage.isArray( calendar_name ) )
			calendar_name = [calendar_name];
		return {
			type: 'uri',
			src: googleCalendarBatchAPI,
			requestType: 'POST',
			contentType: 'multipart/mixed; boundary=' + googleBoundary,
			data: makeHTTPMultiPart( calendar_id.map( function ( x ) {
				return googleCalendarQuery( x, from, to );
			} ) ),
			parser: {
				type: 'custom',
				inline: function ( text ) {
					var parts = parseMultiPart( text ), responses = [];
					for ( var i = 0; i < parts.length; i++ )
						parseGooglePart( parts[i], responses );
					var table = [];
					for ( i = 0; i < responses.length; i++ ) {
						if ( responses[i] && responses[i].httpStatus === 200 ) {
							var json = google_get_result( responses[i].content, 'items', report );
							if ( json ) {
								for ( var j = 0; j < json.length && ( !count || j < count ); j++ )
									table.push( googleCalendarEvent( args, json[j], i + 1, calendar_id[i], calendar_name[i] ) );
							}
						}
					}
					table.sort( function ( a, b ) {
						return a.startDate.getTime() - b.startDate.getTime();
					} );
					if ( count && table.length > count )
						table.length = count;
					return table;
				}
			}
		};
	} else {
		return {
			type: 'uri',
			src: googleCalendarQuery( calendar_id, from, to ),
			parser: {
				type: 'custom',
				inline: function ( text ) {
					var json = google_get_result( text, 'items', report );
					if ( !json )
						return [];
					var table = [];
					var length = json.length;
					if ( count > 0 && length > count )
						length = count;
					for ( var i = 0; i < length; i++ )
						table.push( googleCalendarEvent( args, json[i], 1, calendar_id, args.name || '' ) );
					return table;
				}
			}
		};
    }
}

var googleDriveAPI = 'https://www.googleapis.com/drive/v3';
var reSplitWildcards = /\s*;\s*/;
var reSpecialChars = /[\\\^\$\+\.\(\)\|\{\}\[\]]/g;
var reExifDateTime = /^(\d{4}):(\d{2}):(\d{2}) (\d{2}):(\d{2}):(\d{2})$/;

function google_photos(feed, args, report) {
	var q;
	if ( !args.folderId && args.driveId == 'shared' )
		q = 'sharedWithMe';
	else
		q = 'parents in "' + ( args.folderId ? args.folderId : args.driveId ? args.driveId : 'root' ) + '"';
    var qs = {
        corpora: 'user',
        pageSize: 1000,
		q: q + ' and trashed=false and not mimeType contains "application/vnd.google-apps"',
        fields: 'files(id,name,description,createdTime,modifiedTime,size,version,mimeType,owners/displayName,owners/photoLink,starred,imageMediaMetadata/time,imageMediaMetadata/location)'
	};

	if ( args.driveId && args.driveId != 'shared' ) {
		qs.corpora = 'drive';
		qs.includeItemsFromAllDrives = 'true';
		qs.supportsAllDrives = 'true';
		qs.driveId = args.driveId;
	}

    if (args.sort) {
        if (args.sort == 'alpha')
            qs.orderBy = 'name';
        else if (args.sort == 'modified')
            qs.orderBy = 'modifiedTime desc';
        else if (args.sort == 'created')
            qs.orderBy = 'createdTime desc';
    }

    var filter = null;
    if (args.filter) {
        filter = args.filter.replace(reSpecialChars, '\\$&');
        filter = filter.replace(/\*/g, '.*');
        filter = filter.replace(/\?/g, '.');
        filter = new RegExp('^(?:' + filter.split(reSplitWildcards).join('|') + ')$', 'i');
    }

    feed = {
        type: 'uri',
        src: uwq(googleDriveAPI + '/files', qs),
        parser: {
            type: 'custom',
            inline: function (text) {
                var json = google_get_result(text, 'files', report);
                if (!json)
                    return [];
                var table = [];
                for (var i = 0; i < json.length; i++) {
                    var x = json[i];
                    if (!x.id || !x.name)
                        continue;
                    if (filter && !filter.test(x.name))
                        continue;
					var taken = reExifDateTime.exec( get( x, 'imageMediaMetadata.time' ) );
					if ( x.mimeType === 'video/avi' )
						x.mimeType = 'video/x-msvideo';
                    table.push({
                        title: x.name,
                        description: x.description || '',
                        href: 'https://' + encodeURIComponent(args.account) + ',' + encodeURIComponent(x.version || '') + ',' + encodeURIComponent(x.mimeType || '') + '@www.googleapis.com/drive/v3/files/' + encodeURIComponent(x.id) + '?alt=media',
                        owner: get(x, 'owners.0.displayName') || '',
                        profileImage: get(x, 'owners.0.photoLink') || '',
                        tags: '',
                        taken: taken ? new Date(taken[1], taken[2] - 1, taken[3], taken[4], taken[5], taken[6]) : new Date(x.modifiedTime),
                        posted: new Date(x.createdTime),
                        place: '',
                        longitude: get(x, 'imageMediaMetadata.location.longitude') || NaN,
                        latitude: get(x, 'imageMediaMetadata.location.latitude') || NaN,
                        likes: 0,
                        liked: x.starred ? 1 : 0,
                        mimeType: x.mimeType || ''
                    });
                }
                return table;
            }
        }
    };

    if (args.sort) {
        if (args.sort == 'alpha') {
            feed.sort = {
                column: 'title',
                type: 'string'
            };
        } else if (args.sort == 'modified') {
            feed.sort = {
                column: 'posted',
                type: 'date',
                reverse: true
            };
        } else if (args.sort == 'created') {
            feed.sort = {
                column: 'taken',
                type: 'date',
                reverse: true
            };
        }
    }

    if (args.count)
        feed.maxCount = args.count;

    return feed;
}

var graphAPI = 'https://graph.microsoft.com/v1.0';

function azure_get_result(text, report, varname, vartype) {
    var json = parse_json(text, report);
    if (!json)
        return null;
    if (typeof (json) != 'object' || typeof (json[varname]) != (vartype || 'object')) {
        report.error = (get(json, 'error.code') || 418) + ' ' + (get(json, 'error.message') || 'Missing results');
        return null;
    }
    return json;
}

var reDateFormatChar = /[dmy]/;
var reRemoveProtectedChar = /\\./g;
var reRemoveFormatLiteral = /".*?"/g;
var reRemoveFormatTag = /\[.*?\]/g;

function azure_spreadsheet( feed, args, workbook_id, sheet_id, range, transpose, headers, colnames, path, account, report ) {
	var endPoint, root;
	if ( args.driveId ) {
		endPoint = '/drives/' + encodeURIComponent( args.driveId ) + '/items/';
		root = '@graph.microsoft.com/v1.0/drives/' + encodeURIComponent( args.driveId ) + '/root:';
	} else {
		endPoint = '/me/drive/items/';
		root = '@graph.microsoft.com/v1.0/me/drive/root:';
	}
    endPoint += encodeURIComponent(workbook_id) + '/workbook/worksheets/' + encodeURIComponent(sheet_id);
    if (range)
        endPoint += '/Range(address=\'' + range + '\')';
    else
        endPoint += '/UsedRange';
    endPoint += '?$select=rowIndex,rowCount,columnIndex,columnCount,values,numberFormat';

    return {
        type: 'uri',
        src: graphAPI + endPoint,
        requestType: 'GET',
        parser: {
            type: 'custom',
            inline: function (text) {
                var json = azure_get_result(text, report, 'values');
                if (!json)
                    return [];
                if (json.numberFormat) {
                    for (var i = 0; i < json.numberFormat.length; i++) {
                        for (var j = 0; j < json.numberFormat[i].length; j++) {
                            if (reDateFormatChar.test(json.numberFormat[i][j])) {
                                var x = json.numberFormat[i][j].replace(reRemoveProtectedChar, '').replace(reRemoveFormatLiteral, '').replace(reRemoveFormatTag, '');
								if ( reDateFormatChar.test( x ) && typeof ( json.values[i][j] == 'number' ) ) {
									var d = new Date( ( json.values[i][j] - 25569 ) * 86400000 );
									json.values[i][j] = new Date( d.getUTCFullYear(), d.getUTCMonth(), d.getUTCDate(), d.getUTCHours(), d.getUTCMinutes(), d.getUTCSeconds(), d.getUTCMilliseconds() );
								}
                            }
                        }
                    }
                }
                return table2d_to_feed(json.values, transpose, headers, colnames, path ? 'https://' + encodeURIComponent(account) + root + path.split('/').map(encodeURIComponent).join('/') : null, ':/content');
            }
        }
    };
}

function azure_chart( args, workbook_id, sheet_id, chart_id, report ) {
	var endPoint;
	if ( args.driveId )
		endPoint = '/drives/' + encodeURIComponent( args.driveId ) + '/items/';
	else
		endPoint = '/me/drive/items/';
    endPoint += encodeURIComponent(workbook_id) + '/workbook/worksheets/' + encodeURIComponent(sheet_id) + '/charts/' + encodeURIComponent(chart_id);
    endPoint += '/Image(width=' + (args.width || 0) + ',height=' + (args.height || 0) + ',fittingMode=\'fit\')';

    return {
        type: 'uri',
        src: graphAPI + endPoint,
        requestType: 'GET',
        parser: {
            type: 'custom',
            inline: function (text) {
                var json = azure_get_result(text, report, 'value', 'string');
                if (!json)
                    return [];
                return [{
                    link: 'data:image/png;base64,' + json.value
                }];
            }
        }
    };
}

function azure_calendar( feed, calendar_ids, from, to, args, report ) {
	if ( !jSignage.isArray( calendar_ids ) )
		calendar_ids = [calendar_ids];
	var calendarNames = args.name || [], calendarIds = [];
	if ( !jSignage.isArray( calendarNames ) )
		calendarNames = [calendarNames];
	for ( var i = 0; i < calendar_ids.length; i++ ) {
		var id = calendar_ids[i];
		if ( id.startsWith( 'room-' ) )
			calendarIds.push( id.substring( 5 ) );
		else if ( id.startsWith( 'email:' ) )
			calendarIds.push( id.substring( 6 ) );
		else
			calendarIds.push( id );
	}
	for ( i = calendarNames.length; i < calendarIds.length; i++ )
		calendarNames[i] = calendarIds[i];
	var requests = [];
	for ( i = 0; i < calendar_ids.length; i++ ) {
		var endPoint;
		id = calendar_ids[i];
		if ( id.startsWith( 'room-' ) )
			endPoint = '/users/' + encodeURIComponent( id.substring( 5 ) ) + '/calendar';
		else if ( id.startsWith( 'email:' ) )
			endPoint = '/users/' + encodeURIComponent( id.substring( 6 ) ) + '/calendar';
		else
			endPoint = '/me/calendars/' + encodeURIComponent( id );
		requests.push( {
			id: String( i + 1 ),
			method: 'GET',
            url: endPoint + '/calendarView?startDateTime=' + from.toISOString() + '&endDateTime=' + to.toISOString() + '&$orderby=start/dateTime&$expand=attachments($select=contentType,id,lastModifiedDateTime)'
		} );
	}

	var count = args.count || 0;
	if ( !isFinite( count ) || count < 0 )
		count = 0;

    return {
        type: 'uri',
        src: graphAPI + '/$batch',
		requestType: 'POST',
		contentType: 'application/json',
		data: JSON.stringify( { requests: requests } ),
        parser: {
            type: 'custom',
            inline: function (text) {
                var json = azure_get_result(text, report, 'responses');
				if ( !json || !jSignage.isArray( json.responses))
					return [];
				var table = [], responses = json.responses;
				for ( var i = 0; i < responses.length; i++ ) {
					var response = responses[i];
					if ( typeof response === 'object' && response.status === 200 && typeof response.body === 'object' && jSignage.isArray( response.body.value ) ) {
						var num = parseInt( response.id, 10 );
						if ( !isFinite( num ) || num < 0 )
							num = 0;
                        var ctx = response.body['@odata.context'];
                        if ( ctx ) {
                            var iStart = ctx.indexOf( '#' );
                            var iEnd = ctx.indexOf( '/calendarView' );
                            if ( iStart !== -1 && iEnd !== -1 ) {
                                ctx = ctx.substring( iStart + 1, iEnd );
                            }
                        }
						for ( var j = 0; j < response.body.value.length && ( !count || j < count ); j++ ) {
							var event = {};
							var x = response.body.value[j];
							if ( !x.start || !x.end || typeof x.start !== 'object' || typeof x.end !== 'object' )
								continue;
							if ( x.start.timeZone !== 'UTC' || x.end.timeZone !== 'UTC' )
								jSignage.error( 'Unknown timezone returned by outlook calendar: ' + x.start.timeZone + ' / ' + x.end.timeZone );
							if ( num > 0 ) {
								event.calendarName = calendarNames[num - 1];
								event.calendarId = calendarIds[num - 1];
								event.calendarNum = num;
							}
							event.title = x.subject || '';
							event.description = x.bodyPreview || '';
							event.startDate = new Date( x.start.dateTime + 'Z' );
							event.endDate = new Date( x.end.dateTime + 'Z' );
							event.location = get( x, 'location.displayName' ) || '';
							event.allDay = x.isAllDay ? 1 : 0;
							event.recuring = x.type === 'occurrence' ? 1 : 0;
							var guests = [];
							if ( jSignage.isArray( x.attendees ) ) {
								for ( var k = 0; k < x.attendees.length; k++ ) {
									var y = get( x.attendees[k], 'emailAddress.name' );
									if ( y )
										guests.push( y );
								}
							}
							event.guests = guests.join( ', ' );
							event.created = new Date( x.createdDateTime );
							event.modified = new Date( x.lastModifiedDateTime );
							if ( x.categories && jSignage.isArray( x.categories ) )
								event.categories = x.categories.join( ', ' );
							else
								event.categories = '';
							event.importance = x.importance || '';
							event.sensitivity = x.sensitivity || '';
							event.cancelled = x.isCancelled ? 1 : 0;
							event.showAs = x.showAs || '';
							if ( x.isOrganizer )
								event.organizer = 'me';
							else
                                event.organizer = get( x, 'organizer.emailAddress.name' ) || '';
                            var medias = [];
                            if ( ctx && x.attachments && jSignage.isArray( x.attachments ) ) {
                                for ( var k = 0; k < x.attachments.length; k++ ) {
                                    if ( typeof x.attachments[k] === 'object' ) {
                                        var a = x.attachments[k], url;
                                        if ( a['@odata.type'] === '#microsoft.graph.fileAttachment' ) {
                                            url = '/' + ctx + '/events/' + encodeURIComponent( x.id ) + '/attachments/' + encodeURIComponent( a.id ) + '/$value';
                                            url = 'https://' + encodeURIComponent( args.account ) + ',' + encodeURIComponent( a.lastModifiedDateTime || '' ) + ',' + encodeURIComponent( a.contentType || '' ) + "@graph.microsoft.com/v1.0" + url;
                                            medias.push( url );
                                        }
                                    }
                                }
                            }
                            if ( medias.length > 0 ) {
                                event.enclosure = medias[0];
                                event.enclosures = medias.join( ' ' );
                            } else {
                                event.enclosure = event.enclosures = '';
                            }
							table.push( event );
						}
					}
				}
				if ( calendar_ids.length > 1 ) {
					table.sort( function ( a, b ) {
						return a.startDate.getTime() - b.startDate.getTime();
					} );
					if ( count && table.length > count )
						table.length = count;
				}
                return table;
            }
        }
    };
}

function azure_photos(feed, args, report) {
	var endPoint;
	var drive = args.driveId ? '/drives/' + encodeURIComponent( args.driveId ) : '/me/drive';
	if ( args.folderId ) {
		endPoint = drive + '/items/' + encodeURIComponent( args.folderId );
    } else if (args.folder) {
        var path = args.folder.split('/');
        for (var i = 0; i < path.length; i++)
            path[i] = encodeURIComponent(path[i]);
        if (path.length > 0 && path[0] == '')
            path.shift();
        if (path.length > 0 && path[path.length - 1] == '')
            path.pop();
        if (path.length > 0)
            endPoint = drive + '/root:/' + path.join('/') + ':';
        else
            endPoint = drive + '/root';
    } else {
		endPoint = drive + '/root';
	}

	var root = '@graph.microsoft.com/v1.0' + drive + '/items/';

    var filter = null;
    if (args.filter) {
        filter = args.filter.replace(reSpecialChars, '\\$&');
        filter = filter.replace(/\*/g, '.*');
        filter = filter.replace(/\?/g, '.');
        filter = new RegExp('^(?:' + filter.split(reSplitWildcards).join('|') + ')$', 'i');
    }

    var graphURI = graphAPI + endPoint + '/children?$top=2000&$select=id,name,description,eTag,file,createdBy,createdDateTime,lastModifiedDateTime,photo';
    if ( args.size )
        graphURI += '&$expand=thumbnails($select=' + encodeURIComponent( args.size ) + ')';

    feed = {
        type: 'uri',
        src: graphURI,
        requestType: 'GET',
        parser: {
            type: 'custom',
            inline: function (text) {
                var json = azure_get_result(text, report, 'value');
                if (!json || !jSignage.isArray(json.value))
                    return [];
                json = json.value;
                var table = [], href;
                for (var i = 0; i < json.length; i++) {
                    var x = json[i];
                    if (!x.name || !x.id || !x.file)
                        continue;
                    if (filter && !filter.test(x.name))
						continue;
					if ( x.file.mimeType === 'video/avi' )
                        x.file.mimeType = 'video/x-msvideo';
                    href = null;
                    if ( args.size ) {
                        href = get( x, 'thumbnails.0.' + args.size + '.url' );
                        if ( href )
                            x.file.mimeType = 'image/jpeg';
                    }
                    if ( !href )
                        href = 'https://' + encodeURIComponent( args.account ) + ',' + encodeURIComponent( x.eTag || '' ) + ',' + encodeURIComponent( x.file.mimeType || '' ) + root + x.id + '/content';
                    table.push({
                        title: x.name,
                        description: x.description || '',
                        href: href,
                        owner: get(x, 'createdBy.user.displayName') || '',
                        profileImage: '',
                        tags: '',
                        taken: new Date((x.photo && x.photo.takenDateTime) || x.lastModifiedDateTime),
                        posted: new Date(x.createdDateTime),
                        place: '',
                        longitude: NaN,
                        latitude: NaN,
                        likes: 0,
                        liked: 0,
                        mimeType: x.file.mimeType || ''
                    });
                }
                return table;
            }
        }
    };

    if (args.sort) {
        if (args.sort == 'alpha') {
            feed.sort = {
                column: 'title',
                type: 'string'
            };
        } else if (args.sort == 'modified') {
            feed.sort = {
                column: 'posted',
                type: 'date',
                reverse: true
            };
        } else if (args.sort == 'created') {
            feed.sort = {
                column: 'taken',
                type: 'date',
                reverse: true
            };
        }
    }

    if (args.count)
        feed.maxCount = args.count;

    return feed;
}

var reQuickLink = /^items\[\d+\]\.sourceItem\.url$/;

function sharepoint_posts(feed, args, report) {
    var endpoint = '/sites/' + ( args.id ? encodeURIComponent( args.id ) : 'root' ) + '/pages/microsoft.graph.sitePage';
    var qs = {
        $orderby: 'lastModifiedDateTime desc',
        $select: 'promotionKind,title,description,webUrl,createdBy,lastModifiedBy,createdDateTime,lastModifiedDateTime,publishingState,reactions,titleArea,id',
        $expand: 'webparts($filter=microsoft.graph.standardWebPart/webPartType eq \'cbe7b0a9-3504-44dd-a3a3-0e5cacd07788\')',
        $top: 100
    };
    var count = args.count || 100;
    var all = !!args.all;

	if (args.timeline !== 'all') {
        qs.$filter = 'promotionKind eq \'newsPost\'';
    }

    if ( all ) {
        qs.$expand = 'webparts';
    }

    var getDriveUrl = function (url) {
        if (!url.startsWith('http://') && !url.startsWith('https://') && args.url) {
            if (url.startsWith('/')) {
                var is = args.url.indexOf('/', 8);
                if (is >= 0)
                    url = args.url.substring(0, is) + url;
            } else {
                url = args.url + url;
            }
        }
        if (url.startsWith('http://') || url.startsWith('https://')) {
            if (args.url && args.drives && url.startsWith(args.url + '/')) {
                for (var path in args.drives) {
                    var prefix = args.url + path + '/';
                    if (url.startsWith(prefix)) {
                        url = 'https://' + encodeURIComponent(args.account) + ',0@' + graphAPI.substring(8) + '/drives/' + encodeURIComponent(args.drives[path]) + '/root:/' + url.substring(prefix.length) + ':/content';
                        break;
                    }
                }
            }
            return url;
        }
        return null;
    }

    var mods = {
        type: 'uri',
        requestType: 'GET',
        parser: {
            type: 'custom',
            inline: function ( text ) {
                var json = azure_get_result(text, report, 'value');
                if (!json || !json.value || !jSignage.isArray(json.value))
                    return [];
                var table = [];
                for (var i = 0; i < json.value.length && count > 0; i++) {
                    var x = json.value[i];
                    if (x && typeof (x) === 'object' && (x.promotionKind === 'newsPost' || (x.promotionKind === 'page' && args.timeline === 'all')) && typeof (x.publishingState) === 'object' && x.publishingState.level === 'published') {
                        --count;
                        var post = {
                            title: x.title || '',
                            created: new Date(x.lastModifiedDateTime || x.createdDateTime),
                            user: get(x, 'lastModifiedBy.user.email') || get(x, 'createdBy.user.email') || 0,
                            name: get(x, 'lastModifiedBy.user.displayName') || get(x, 'createdBy.user.displayName') || 0,
                            jobTitle: '',
                            description: x.description || '',
                            text: x.description || x.title || '',
                            raw: x.description || x.title || '',
                            url: x.webUrl || '',
                            urls: x.webUrl || '',
                            likes: get(x, 'reactions.likeCount') || 0,
                            shares: get(x, 'reactions.shareCount') || 0,
                            media: '',
                            medias: ''
                        };
                        if (x.titleArea && typeof (x.titleArea) === 'object') {
                            var image = x.titleArea.imageWebUrl;
                            if (image) {
                                var url = getDriveUrl(image);
                                if (url) {
                                    post.media = url;
                                    post.medias = url;
                                }
                            }
                            if (x.titleArea.authors && jSignage.isArray(x.titleArea.authors) && x.titleArea.authors.length > 0) {
                                var author = x.titleArea.authors[0];
                                if (author && typeof(author) === 'object') {
                                    post.user = author.email || '';
                                    post.name = author.name || '';
                                    post.jobTitle = author.role || '';
                                }
                            }
                        } else if (x.webParts && jSignage.isArray(x.webParts) && x.webParts[0] && typeof x.webParts[0] === 'object' && x.webParts[0].webPartType === 'cbe7b0a9-3504-44dd-a3a3-0e5cacd07788') {
							var img = get(x.webParts[0], 'data.serverProcessedContent.imageSources.0');
                            if (!all && img && typeof(img) === 'object' && img.key === 'imageSource' && img.value) {
                                var url = getDriveUrl(img.value);
                                if (url) {
                                    post.media = url;
                                    post.medias = url;
                                }
                            }
                            var author = get(x.webParts[0], 'data.properties.authors.0');
                            if (author && typeof(author) === 'object') {
                                post.user = author.email || '';
                                post.name = author.name || '';
                                post.jobTitle = author.role || '';
                            }
                        }
                        if (all && x.webParts && jSignage.isArray(x.webParts)) {
                            var html = [], urls = [], medias = [];
                            if (post.url)
                                urls.push(post.url);
                            if (post.media)
                                medias.push(post.media);
                            for (var j = 0; j < x.webParts.length; j++) {
                                var part = x.webParts[j];
                                if (part && typeof (part) === 'object') {
                                    switch (part['@odata.type']) {
                                        case '#microsoft.graph.textWebPart':
                                            html.push(part.innerHtml);
                                            break;
                                        case '#microsoft.graph.standardWebPart':
                                            switch (part.webPartType) {
                                                case 'd1d91016-032f-456d-98a4-721247c305e8': // image
                                                case 'cbe7b0a9-3504-44dd-a3a3-0e5cacd07788': // banner
                                                    var img = get(part, 'data.serverProcessedContent.imageSources.0');
                                                    if (img && typeof(img) === 'object' && img.key === 'imageSource' && img.value) {
                                                        var url = getDriveUrl(img.value);
                                                        if (url)
                                                            medias.push(url);
                                                    }
                                                    break;
                                                case 'ac0e47ca-30af-452c-bdbb-510b715d0e46': // video
                                                    var url = get(part, 'data.properties.link') || get(part, 'data.properties.file');
                                                    if (url)
                                                        url = getDriveUrl(url);
                                                    if (url)
                                                        medias.push(url);
                                                    break;
                                                case '6410b3b6-d440-4663-8744-378976dc041e': // link
                                                    var link = get(part, 'data.serverProcessedContent.links.0');
                                                    if (link && typeof(link) === 'object' && link.key === 'url' && link.value) {
                                                        if (link.value.startsWith('http://') || link.value.startsWith('https://'))
                                                            urls.push(link.value);
                                                    }
                                                    break;
                                                 case '0f087d7f-520e-42b7-89c0-496aaf979d58': // call to action
                                                    var link = get(part, 'data.serverProcessedContent.links.0');
                                                    if (link && typeof(link) === 'object' && link.key === 'linkUrl' && link.value) {
                                                        if (link.value.startsWith('http://') || link.value.startsWith('https://'))
                                                            urls.push(link.value);
                                                    }
                                                    break;
                                                case 'c70391ea-0b10-4ee9-b2b4-006d3fcad0cd': // quick links
                                                    var links = get(part, 'data.serverProcessedContent.links');
                                                    if (links && jSignage.isArray(links)) {
                                                        for (var k = 0; k < links.Length; k++) {
                                                            var link = links[k];
                                                            if (link && typeof (link) === 'object' && reQuickLink.test(link.key) && link.value) {
                                                                if (link.value.startsWith('http://') || link.value.startsWith('https://'))
                                                                    urls.push(link.value);
                                                            }
                                                        }
                                                    }
                                                    break;
                                                case '275c0095-a77e-4f6d-a2a0-6a7626911518': // streaming video
                                                    var link = get(part, 'data.serverProcessedContent.links.0');
                                                    if (link && typeof(link) === 'object' && link.key === 'videoSource' && link.value) {
                                                        if (link.value.startsWith('http://') || link.value.startsWith('https://'))
                                                            urls.push(link.value);
                                                    }
                                                    break;
                                                default:
                                                    break;
                                            }
                                            break;
                                        default:
                                            break;
                                    }
                                }
                            }
                            if (html.length > 0) {
                                post.raw = html.join('\n');
                                post.text = html.map(cleanup_text).join('\n');
                            }
                            if (urls.length > 0) {
                                post.url = urls[0];
                                post.urls = urls.join(' ');
                            }
                            if (medias.length > 0) {
                                post.media = medias[0];
                                post.medias = medias.join(' ');
                            }
                        }
                        table.push(post);
                    }
                }
                return table;
            }
        }
    };

    mods.src = uwq(graphAPI + endpoint, qs);
    return mods;
}

function sharepoint_spreadsheet(feed, workbook_id, sheet_id, report) {

    var endPoint = '/sites/' + (workbook_id ? encodeURIComponent(workbook_id) : 'root') + '/lists/' + encodeURIComponent(sheet_id) + '?expand=columns,items(expand=fields)';

    return {
        type: 'uri',
        src: graphAPI + endPoint,
        requestType: 'GET',
        parser: {
            type: 'custom',
            inline: function (text) {
                var json = azure_get_result(text, report, 'list');
                if (!json || !json.columns || !jSignage.isArray(json.columns) || !json.items || !jSignage.isArray(json.items))
                    return [];
                var columns = [];
                for (var i = 0; i < json.columns.length; i++) {
                    if (typeof (json.columns[i]) == 'object') {
                        var x = json.columns[i], column = {};
                        if (x.hidden || x.readOnly || x.lookup || x.columnGroup == '_Hidden' || x.personOrGroup || !x.displayName || !x.name)
                            continue;
                        column.id = x.name;
                        column.name = x.displayName;
                        if (x.text || x.choice) {
                            column.type = 0;
                        } else if (x.currency || x.number) {
                            column.type = 1;
                        } else if (x.boolean) {
                            column.type = 2;
                        } else if (x.dateTime && typeof (x.dateTime) == 'object') {
                            column.type = (x.dateTime.format == 'dateOnly') ? 4 : 3;
                        } else if (x.name == 'Hyperlink' || x.name == 'Picture') {
                            column.type = 5;
                        } else if (x.calculated && typeof (x.calculated) == 'object') {
                            if (x.calculated.outputType == 'number' || x.calculated.outputType == 'currency') {
                                column.type = 1;
                            } else if (x.calculated.outputType == 'boolean') {
                                column.type = 2;
                            } else if (x.calculated.outputType == 'dateTime') {
                                if (x.calculated.format == 'dateOnly')
                                    column.type = 4;
                                else
                                    column.type = 3;
                            } else {
                                column.type = 0;
                            }
                        } else {
                            continue;
                        }
                        columns.push(column);
                    }
                }
                var table = [];
                for (var i = 0; i < json.items.length; i++) {
                    var x = json.items[i], row = {};
                    row.created = new Date(x.createdDateTime);
                    row.createdBy = get(x, 'createdBy.user.displayName') || '';
                    row.lastModified = new Date(x.lastModifiedDateTime);
                    row.lastModifiedBy = get(x, 'lastModifiedBy.user.displayName') || '';
                    if (typeof (x.fields) == 'object') {
                        for (var j = 0; j < columns.length; j++) {
                            var column = columns[j], day;
                            switch (column.type) {
                                case 0:
                                    row[column.name] = column.id in x.fields ? String(x.fields[column.id]) : '';
                                    break;
                                case 1:
                                    row[column.name] = column.id in x.fields ? Number(x.fields[column.id]) : NaN;
                                    break;
                                case 2:
                                    row[column.name] = x.fields[column.id] ? 1 : 0;
                                    break;
                                case 3:
                                    row[column.name] = new Date(x.fields[column.id]);
                                    break;
                                case 4:
                                    day = new Date(x.fields[column.id]);
                                    row[column.name] = new Date(day.getUTCFullYear(), day.getUTCMonth(), day.getUTCDate());
                                    break;
                                case 5:
                                    row[column.name] = column.id in x.fields ? String(x.fields[column.id].Url) : '';
                                    break; 
                                default:
                                    break;
                            }
                        }
                    }
                    table.push(row);
                }
                return table;
            }
        }
    };
}

var flickrAPI = 'https://api.flickr.com/services/rest/';

function flickr_get_photos(text, report) {
    var json = parse_json(text, report);
    if (!json)
        return null;
    if (get(json, 'stat') != "ok") {
        report.error = '418 ' + (get(json, 'message') || get(json, 'stat') || 'Missing status');
        return null;
    }
    var photo = get(json, 'photos.photo') || get(json, 'photoset.photo');
    if (!jSignage.isArray(photo)) {
        report.error = '418 Missing photo element';
        return null;
    }
    return photo;
}

var reFlickrDate = /^(\d+)-(\d+)-(\d+) (\d+):(\d+):(\d+)/;

function flickr_photos(feed, args, report) {
    var mods = {
        type: 'uri',
        src: flickrAPI,
        count: args.count || 0,
        parser: {
            type: 'custom',
            inline: function (text) {
                var json = flickr_get_photos(text, report);
                if (!json)
                    return [];
                var table = [];
                for (var i = 0; i < json.length; i++) {
                    var photo = json[i];
                    var r = {};
                    if (photo && typeof (photo) == 'object' && photo.id) {
                        r.href = 'https://farm' + photo.farm + '.staticflickr.com/' + photo.server + '/' + photo.id + '_' + photo.secret;
                        if (args.size) {
                            if (args.size != 'b' || photo.url_l)
                                r.href += '_' + args.size;
                        }
                        r.href += '.jpg';
                        r.title = 'title' in photo ? cleanup_text(photo.title) : '';
                        if (photo.description && typeof (photo.description) == 'object' && '_content' in photo.description)
                            r.description = cleanup_text(photo.description._content);
                        else
                            r.description = '';
                        var m = reFlickrDate.exec(photo.datetaken);
                        if (m)
                            r.taken = new Date(m[1], parseFloat(m[2]) - 1, m[3], m[4], m[5], m[6]);
                        else
                            r.taken = new Date(NaN);
                        r.posted = new Date(photo.dateupload * 1000);
                        r.owner = photo.ownername || '';
                        r.profileImage = '';
                        var tags = [];
                        if ('tags' in photo)
                            tags = String(photo.tags).split(' ');
                        r.tags = tags.join(',');
                        if (photo.accuracy) {
                            r.longitude = Number(photo.longitude) || NaN;
                            r.latitude = Number(photo.latitude) || NaN;
                        } else {
                            r.longitude = NaN;
                            r.latitude = NaN;
                        }
                        r.place = '';
                        r.liked = 0;
                        r.likes = 0;
                        table.push(r);
                    }
                }
                return table;
            }
        }
    };
    var api, qs = {
        format: 'json',
        nojsoncallback: 1,
        page: 1,
        extras: 'description,date_upload,date_taken,owner_name,geo,tags,url_l'
    };
    switch (args.selector) {
        case 'roll':
            api = 'flickr.people.getPhotos';
            qs.user_id = 'me';
            break;
        default:
        case 'stream':
            api = 'flickr.people.getPhotos';
            qs.user_id = 'me';
            qs.privacy_filter = 1;
            break;
        case 'likes':
            api = 'flickr.favorites.getList';
            break;
        case 'album':
            api = 'flickr.photosets.getPhotos';
            qs.photoset_id = args.id || 0;
            break;
        case 'gallery':
            api = 'flickr.galleries.getPhotos';
            qs.gallery_id = args.id || 0;
            break;
        case 'tag':
            api = 'flickr.photos.search';
            qs.media = 'photos';
            qs.tag_mode = 'all';
            var tag = args.tag || '';
            if (tag.length > 0 && tag.charAt(0) == '#')
                tag = tag.substring(1);
            qs.tags = tag;
            break;
        case 'user':
            api = 'flickr.people.getPhotos';
            qs.user_id = args.id || 0;
            break;
        case 'geo':
            api = 'flickr.photos.search';
            qs.media = 'photos';
            qs.lat = jSignage.parseLatLon(args.latitude, true);
            qs.lon = jSignage.parseLatLon(args.longitude, false);
            qs.radius = jSignage.parseDistance(args.radius);
            qs.min_upload_date = Math.floor(Date.now() / 1000) - 126230400;
            break;
    }
    if (args.selector === 'tag' || args.selector === 'geo') {
        if (args.restriction) {
            if (args.restriction == 'mine')
                qs.user_id = 'me';
            else if (args.restriction == 'by-nc')
                qs.license = '1,2,3,4,5,6,7,8';
            else if (args.restriction == 'by')
                qs.license = '4,5,6,7,8';
            else if (args.restriction == 'free')
                qs.license = '7,8';
        }
        if (args.sort) {
            if (args.sort == 'posted')
                qs.sort = 'date-posted-desc';
            else if (args.sort == 'taken')
                qs.sort = 'date-taken-desc';
            else if (args.sort == 'interest')
                qs.sort = 'interestingness-desc';
            else if (args.sort == 'relevance')
                qs.sort = 'relevance';
        }
    }
    qs.method = api;
    if (typeof mods.count !== 'number')
        mods.count = parseFloat(mods.count);
    if (!isFinite(mods.count) || mods.count < 0)
        mods.count = 0;
    if (mods.count && mods.count)
        qs.per_page = mods.count;
    else
        qs.per_page = 100;
    mods.src = uwq(mods.src, qs);
    return mods;
}

var instagramAPI = 'https://api.instagram.com/v1';

function instagram_get_photos(text, report) {
    var json = parse_json(text, report);
    if (!json)
        return null;
    if (get(json, 'meta.code') != 200) {
        report.error = (get(json, 'meta.code') || 418) + ' ' + (get(json, 'meta.error_message') || '');
        return null;
    }
    if (!jSignage.isArray(json.data)) {
        report.error = '418 Missing data';
        return null;
    }
    return json.data;
}

function instagram_photos(feed, args, report) {
    var mods = {
        type: 'uri',
        parser: {
            type: 'custom',
            inline: function (text) {
                var json = instagram_get_photos(text, report);
                if (!json)
                    return [];
                var table = [];
                for (var i = 0; i < json.length; i++) {
                    var photo = json[i];
                    var r = {};
                    if (photo && typeof (photo) == 'object' && photo.images && typeof (photo.images) == 'object') {
                        var img = photo.images.standard_resolution;
                        if (args.size) {
                            if (args.size == 't' && photo.images.thumbnail)
                                img = photo.images.thumbnail;
                            else if (args.size == 's' && photo.images.low_resolution)
                                img = photo.images.low_resolution;
                        }
                        if (!img || typeof (img) != 'object')
                            continue;
                        r.href = img.url;
                        if (photo.caption && typeof (photo.caption) == 'object')
                            r.title = 'text' in photo.caption ? cleanup_text(photo.caption.text) : '';
                        else
                            r.title = '';
                        r.description = '';
                        r.taken = r.posted = new Date((Number(photo.created_time) || NaN) * 1000);
                        if (photo.user && typeof (photo.user) == 'object') {
                            r.owner = photo.user.username || '';
                            r.profileImage = photo.user.profile_picture;
                        } else {
                            r.owner = '';
                            r.profileImage = '';
                        }
                        r.liked = Boolean(photo.user_has_liked) ? 1 : 0;
                        if (photo.likes && typeof (photo.likes) == 'object')
                            r.likes = Number(photo.likes.count) || 0;
                        else
                            r.likes = 0;
                        var tags = [];
                        if (photo.tags && jSignage.isArray(photo.tags))
                            tags = photo.tags;
                        r.tags = tags.join(',');
                        if (photo.location && typeof (photo.location) == 'object') {
                            r.longitude = Number(photo.location.longitude) || NaN;
                            r.latitude = Number(photo.location.latitude) || NaN;
                            r.place = 'name' in photo.location ? photo.location.name : '';
                        } else {
                            r.longitude = NaN;
                            r.latitude = NaN;
                            r.place = '';
                        }
                        table.push(r);
                    }
                }
                return table;
            }
        }
    };
    var endpoint, qs = {};
    switch (args.selector) {
        default:
        case 'stream':
            endpoint = '/users/self/media/recent';
            break;
        case 'likes':
            endpoint = '/users/self/media/liked';
            break;
        case 'tag':
            var tag = args.tag || '';
            if (tag.length > 0 && tag.charAt(0) == '#')
                tag = tag.substring(1);
            endpoint = '/tags/' + encodeURIComponent(tag) + '/media/recent';
            break;
        case 'user':
            endpoint = '/users/' + encodeURIComponent(args.id || 0) + '/media/recent';
            break;
        case 'geo':
            endpoint = '/media/search';
            qs.lat = jSignage.parseLatLon(args.latitude, true);
            qs.lng = jSignage.parseLatLon(args.longitude, false);
            qs.radius = jSignage.parseDistance(args.radius) * 1000;
            break;
    }
    var count = args.count || 0;
    if (isFinite(count) && count > 0)
        qs.count = count;
    mods.src = uwq(instagramAPI + endpoint, qs);
    return mods;
}

var twitterAPI = 'https://api.twitter.com/1.1';

function twitter_get_statuses(text, report) {
    var json = parse_json(text, report);
    if (!json)
        return null;
    if (jSignage.isArray(json))
        return json;
    if (jSignage.isArray(json.statuses))
        return json.statuses;
    return null;
}

var reTwitterDate = /^([A-Za-z]+) ([A-Za-z]+) (\d+) (\d+):(\d+):(\d+) \+0000 (\d+)$/;
var twitterMonth = { Jan: 0, Feb: 1, Mar: 2, Apr: 3, May: 4, Jun: 5, Jul: 6, Aug: 7, Sep: 8, Oct: 9, Nov: 10, Dec: 11 };

function twitter_posts(feed, timeline, username, query, count, retweets, report) {
    var args = {
        type: 'uri',
        parser: {
            type: 'custom',
            inline: function (text) {
                var json = twitter_get_statuses(text, report);
                if (!json)
                    return [];
                var table = [];
                for (var i = 0; i < json.length; i++) {
                    var tweet = {};
                    var x = json[i];
                    if (x.retweeted_status && typeof (x.retweeted_status) == 'object')
                        x = x.retweeted_status;
                    var y = x;
                    if (x.truncated && x.extended_tweet && typeof (x.extended_tweet) == 'object') {
                        y = x.extended_tweet;
                        tweet.text = tweet.raw = y.full_text;
                    } else {
                        tweet.text = tweet.raw = x.text;
                    }
                    var r = reTwitterDate.exec(x.created_at);
                    if (r && r[2] in twitterMonth)
                        tweet.created = new Date(Date.UTC(r[7], twitterMonth[r[2]], r[3], r[4], r[5], r[6]));
                    else
                        tweet.created = new Date(NaN);
                    if (x.user) {
                        tweet.user = '@' + x.user.screen_name;
                        tweet.name = x.user.name || '';
                        tweet.verified = x.user.verified ? 1 : 0;
                        tweet.profileImage = x.user.profile_image_url || x.user.profile_image_url_https || '';
                    } else {
                        tweet.user = tweet.name = tweet.profileImage = '';
                        tweet.verified = 0;
                    }
                    tweet.shared = x.retweet_count || 0;
                    tweet.likes = x.favorite_count || 0;
                    tweet.liked = Boolean(x.favorited) ? 1 : 0;
                    tweet.lang = x.lang || 'und';
                    if (x.place && typeof (x.place) == 'object' && x.place.name)
                        tweet.place = x.place.name;
                    else
                        tweet.place = '';
                    tweet.longitude = tweet.latitude = NaN;
                    if (x.coordinates && typeof (x.coordinates) == 'object') {
                        if (jSignage.isArray(x.coordinates)) {
                            tweet.longitude = x.coordinates[0];
                            tweet.latitude = x.coordinates[1];
                        } else if (x.coordinates.type == 'point' && jSignage.isArray(x.coordinates.coordinates)) {
                            tweet.longitude = x.coordinates.coordinates[0];
                            tweet.latitude = x.coordinates.coordinates[1];
                        }
                    }
                    tweet.tag = tweet.tags = tweet.mention = tweet.mentions = tweet.url = tweet.urls = tweet.media = tweet.medias = '';
                    if (y.entities && typeof (y.entities) == 'object') {
                        if (jSignage.isArray(y.entities.hashtags) && y.entities.hashtags.length > 0) {
                            tweet.tag = '#' + y.entities.hashtags[0].text;
                            tweet.tags = y.entities.hashtags.map(function (h) { return '#' + h.text; }).join(',');
                        }
                        if (jSignage.isArray(y.entities.user_mentions) && y.entities.user_mentions.length > 0) {
                            tweet.mention = '@' + y.entities.user_mentions[0].screen_name;
                            tweet.mentions = y.entities.user_mentions.map(function (h) { return '@' + h.screen_name; }).join(',');
                        }
                        if (jSignage.isArray(y.entities.urls) && y.entities.urls.length > 0) {
                            tweet.url = y.entities.urls[0].expanded_url || y.entities.urls[0].url;
                            tweet.urls = y.entities.urls.map(function (h) { return h.expanded_url || h.url }).join(',');
                        }
                        if (jSignage.isArray(y.entities.media) && y.entities.media.length > 0) {
                            tweet.media = y.entities.media[0].media_url || y.entities.media[0].media_url_https;
                            if (y.extended_entities && typeof (y.extended_entities) == 'object' && jSignage.isArray(y.extended_entities.media) && y.extended_entities.media.length > 1)
                                tweet.medias = y.extended_entities.media.map(function (h) { return h.media_url || h.media_url_https; }).join(',');
                            else
                                tweet.medias = y.entities.media.map(function (h) { return h.media_url || h.media_url_https; }).join(',');
                        }
                    }
                    tweet.type = 'tweet';
                    tweet.text = cleanup_text(tweet.text);
                    table.push(tweet);
                }
                return table;
            }
        }
    };
    var qs = {}, endpoint;
    switch (timeline) {
        case 'home':
            endpoint = '/statuses/home_timeline';
            qs.exclude_replies = retweets ? 'false' : 'true';
            break;
        case 'me':
            endpoint = '/statuses/user_timeline';
            qs.exclude_replies = retweets ? 'false' : 'true';
            qs.include_rts = retweets ? 'true' : 'false';
            break;
        case 'liked':
            endpoint = '/favorites/list';
            break;
        case 'retweets':
            endpoint = '/statuses/retweets_of_me';
            break;
        case 'mentions':
            endpoint = '/statuses/mentions_timeline';
            break;
        case 'user':
            endpoint = '/statuses/user_timeline';
            qs.exclude_replies = retweets ? 'false' : 'true';
            qs.include_rts = retweets ? 'true' : 'false';
            if (username.charAt(0) == '@')
                username = username.substring(1);
            qs.screen_name = username;
            break;
        case 'likes':
            endpoint = '/favorites/list';
            if (query.charAt(0) == '@')
                query = query.substring(1);
            qs.screen_name = query;
            break;
        default:
        case 'search':
            endpoint = '/search/tweets';
            qs.q = query;
            break;
    }
    if (typeof count !== 'number')
        count = parseFloat(count);
    if (isFinite(count) && count > 0)
        qs.count = count;
    qs.tweet_mode = 'compat-extended';
    args.src = uwq(twitterAPI + endpoint + '.json', qs);
    return args;
}

var yammerAPI = 'https://www.yammer.com/api/v1';

var reYammerDateTime = /^(\d{4})\/(\d{2})\/(\d{2}) (\d{2}):(\d{2}):(\d{2}) ([+-])(\d{2})(\d{2})$/;

function yammer_date(str) {
    var d = reYammerDateTime.exec(str);
    if (d) {
        var tUTC = Date.UTC(d[1], Number(d[2]) - 1, d[3], d[4], d[5], d[6]);
        var utcOffsetMinutes = d[8] * 60 + d[9];
        if (d[7] == '-')
            tUTC += utcOffsetMinutes * 60000;
        else
            tUTC -= utcOffsetMinutes * 60000;
        return new Date(tUTC);
    }
    return new Date(NaN);
}

var reYammerRef = /\[\[(user|tag):(\d+)\]\]/g;
var reYammerURLs = /(?:^|[\s<])(https?\:\/\/[^\s>]*)>?/g;

function cleanup_yammer_urls(text, urls) {
    var t = text.replace(reYammerURLs, function (m, url) {
        urls.push(url);
        return '';
    });
    return cleanup_spaces(t);
}

function yammer_posts(feed, args, report) {
    var mods = {
        type: 'uri',
        parser: {
            type: 'custom',
            inline: function (text) {
                var json = parse_json(text, report);
                if (!json)
                    return [];
                if (typeof (json.messages) == 'object' && json.messages.messages)
                    json = json.messages;
                if (!json.messages || !jSignage.isArray(json.messages))
                    return [];
                var users = {}, topics = {};
                var me = get(json, 'meta.current_user_id') || 0;
                if (json.references && jSignage.isArray(json.references)) {
                    for (var i = 0; i < json.references.length; i++) {
                        var x = json.references[i];
                        if (x && typeof (x) == 'object' && x.id) {
                            switch (x.type) {
                                case 'user':
                                    users[x.id] = {
                                        user: x.name || '',
                                        name: x.full_name || '',
                                        profileImage: x.mugshot_url_template ? x.mugshot_url_template.replace('{width}', 360).replace('{height}', 360) : x.mugshot_url ? x.mugshot_url : '',
                                        jobTitle: x.job_title || ''
                                    };
                                    break;
								case 'topic':
									topics[x.id] = {
										name: x.name || ''
									};
									break;
                                default:
                                    break;
                            }
                        }
                    }
                }
                var table = [];
                for (var i = 0; i < json.messages.length; i++) {
                    var x = json.messages[i];
                    var post = {};
                    if (x && typeof (x) == 'object') {
                        post.type = x.message_type || 'update';
                        post.shares = 0;
                        post.lang = x.language || 'und';
                        post.longitude = NaN;
                        post.latitude = NaN;
                        post.place = '';
                        post.created = yammer_date(x.created_at);

                        if (x.sender_id && x.sender_id in users) {
                            post.user = users[x.sender_id].user;
                            post.name = users[x.sender_id].name;
                            post.profileImage = users[x.sender_id].profileImage;
                            post.jobTitle = users[x.sender_id].jobTitle;
                        } else {
                            post.user = post.name = post.profileImage = post.jobTitle = '';
                        }
                        post.verified = 0;
                        var mentions = [], tags = [], urls = [], medias = [], ref;
                        while ((ref = reYammerRef.exec(get(x, 'body.parsed'))) !== null) {
                            if (ref[1] == 'user' && ref[2] in users)
                                mentions.push(users[ref[2]].name);
                            else if (ref[1] == 'tag' && ref[2] in topics)
                                tags.push(topics[ref[2]].name);
                        }
                        if (mentions.length > 0) {
                            post.mention = mentions[0];
                            post.mentions = mentions.map(function (h) { return h.replace(',', ' '); }).join(',');
                        } else {
                            post.mention = post.mentions = '';
                        }
                        if (tags.length > 0) {
                            post.tag = '#' + tags[0];
                            post.tags = tags.map(function (h) { return '#' + h; }).join(',');
                        } else {
                            post.tag = post.tags = '';
                        }
                        post.likes = get(x, 'liked_by.count') || 0;
                        post.liked = 0;
                        var likes = get(x, 'liked_by.names');
                        if (likes && jSignage.isArray(likes)) {
                            for (var j = 0; j < likes.length; j++) {
                                if (typeof (likes[j]) == 'object' && likes[j].user_id === me) {
                                    post.liked = true;
                                    break;
                                }
                            }
                        }
                        post.raw = get(x, 'body.plain') || '';
                        if (post.type === 'poll' && x.poll_options && jSignage.isArray(x.poll_options)) {
                            for (var j = 0; j < x.poll_options.length; j++) {
                                var opt = x.poll_options[j];
                                if (typeof (opt) == 'object' && opt.answer) {
                                    post.raw += '\n • ' + opt.answer;
                                }
                            }
                        }
                        post.text = cleanup_yammer_urls(post.raw, urls);
                        var bodyURLs = get(x, 'body.urls');
                        if (jSignage.isArray(bodyURLs))
                            urls = bodyURLs;
                        if (urls.length > 0) {
                            post.url = urls[0];
                            post.urls = urls.join(',');
                        } else {
                            post.url = post.urls = '';
                        }
                        post.title = x.content_excerpt ? cleanup_yammer_urls(x.content_excerpt, []) : post.text;

                        if (x.attachments && jSignage.isArray(x.attachments)) {
                            for (var j = 0; j < x.attachments.length; j++) {
                                if (typeof (x.attachments[j]) == 'object') {
                                    var a = x.attachments[j], url;
                                    if (a.type == 'image') {
                                        if (a.width && a.height && a.scaled_url && (a.width > 1280 || a.height > 1280)) {
                                            var w, h;
                                            if (a.width >= a.height) {
                                                w = 1280;
                                                h = Math.max(Math.round(1280 * a.height / a.width), 1);
                                            } else {
                                                h = 1280;
                                                w = Math.max(Math.round(1280 * a.width / a.height), 1);
                                            }
                                            url = a.scaled_url.replace('{{width}}', w).replace('{{height}}', h);
                                        } else if (a.download_url) {
                                            url = a.download_url;
                                        }
                                        if (url.substring(0, 23) == "https://www.yammer.com/")
                                            url = "https://" + encodeURIComponent(args.account) + ',' + encodeURIComponent(a.latest_version_id || '') + ',' + encodeURIComponent(a.content_type || '') + "@www.yammer.com/" + url.substring(23);
                                        medias.push(url);
									} else if ( a.type == 'file' && a.content_class == 'Video' ) {
										if ( a.download_url ) {
											url = a.download_url;
											if ( url.substring( 0, 23 ) == "https://www.yammer.com/" )
                                                url = "https://" + encodeURIComponent(args.account) + ',' + encodeURIComponent(a.latest_version_id || '') + ',' + encodeURIComponent(a.content_type || '') + "@www.yammer.com/" + url.substring(23);
											medias.push( url );
										}
                                    } else if (a.type == 'praise') {
                                        post.type = 'praise';
                                        post.title = a.description;
                                        post.raw = a.description + '\n' + a.comment;
                                        post.text = cleanup_spaces(post.raw);
                                        post.icon = a.icon;
                                    }
                                }
                            }
                        }
                        if (medias.length > 0) {
                            post.media = medias[0];
                            post.medias = medias.join(' ');
                        } else {
                            post.media = post.medias = '';
                        }
                        post.description = post.text;

                        table.push(post);
                    }
                }
                return table;
            }
        }
    };
    var endpoint, qs = {
        threaded: false
    };
    switch (args.timeline) {
        default:
        case 'all':
        case 'top':
            endpoint = '/messages.json';
            break;
        case 'follow':
            endpoint = '/messages/following.json';
            break;
        case 'sent':
            endpoint = '/messages/sent.json';
            break;
        case 'mentions':
            endpoint = '/messages/received.json';
            break;
        case 'group':
            endpoint = '/messages/in_group/' + encodeURIComponent(args.id) + '.json';
            break;
        case 'topic':
            endpoint = '/messages/about_topic/' + encodeURIComponent(args.id) + '.json';
            break;
        case 'search':
            endpoint = '/search.json';
            break;
    }
    var count = args.count || 0;
    if (args.timeline == 'search') {
        qs.search = args.query || '';
        if (isFinite(count) && count > 0)
            qs.num_per_page = Math.min(count, 100);
    } else {
        if (isFinite(count) && count > 0)
            qs.limit = Math.min(count, 100);
	}
	if ( args.threaded && ( args.timeline === 'top' || args.timeline === 'follow' || args.timeline === 'sent' || args.timeline === 'mentions' || args.timeline === 'group' ) )
		qs.threaded = true;
    mods.src = uwq(yammerAPI + endpoint, qs);
    return mods;
}

var powerBIEmbedUrl = 'https://download.spinetix.com/widgets/powerbi/powerbi.html';
function powerbi_dashboard(args) {
    function fragment( params ) {
        return '#' + encodeURIComponent(JSON.stringify(params));
    }

    var params = {
        account: args.account
    };

    for ( var x of [ 'embedUrl', 'workspaceId', 'dashboardId', 'reportId', 'tileId', 'pageId', 'pageDur' ] ) {
        if (args[x])
            params[ x ] = args[ x ];
    }

    if ( args.dashboardId )
        params.type = 'dashboard';
    else if (args.reportId)
        params.type = args.pageId ? 'page' : 'report';

    return [{
        url: powerBIEmbedUrl + fragment(params)
    }];
}

var weatherAPI = 'https://feeds.services.spinetix.com/v1/weather/proxy';
if ( navigator.spxCloudStage && navigator.spxCloudStage !== 'prod' )
	weatherAPI = 'https://feeds-staging.services.spinetix.com/' + navigator.spxCloudStage + '/weather/proxy';

function date_range( days ) {
	if ( days === 'today' ) {
		return [0, 1];
	} else if ( days === 'tomorrow' ) {
		return [1, 2];
	} else {
		var n = Math.floor( days );
		if ( isFinite( n ) && n >= 1 ) {
			return [0, ++n];    // today + n days forecast
		} else {
			return [0, 1];    // today
		}
	}
}

function weather_proxy( args, report ) {
	var qs = {
		units: args.units === 'us' ? 'us' : 'metric',
		name: args.label || args.name || '',
		lang: jSignage.getLocale()
	};

    if ( args.provider && args.provider != 'default' && args.provider != 'yahoo' && args.provider != 'darksky' )
		qs.src = args.provider;

	if ( args.loc )
		qs.loc = args.loc;
	else
		qs.id = args.id;

	if ( args.key )
		qs.key = args.key;

	if ( args.days !== 'current' ) {
		var range = date_range( args.days );
		qs.days = range[0].toFixed() + '-' + range[1].toFixed();
	}

	var i = qs.lang.indexOf( '-' );
	if ( i >= 0 )
		qs.lang = qs.lang.substring( 0, i );

	var feed = {
		type: 'uri',
		src: uwq( weatherAPI, qs ),
		parser: {
			type: 'custom',
			inline: function ( text ) {
				var json = parse_json( text, report );
				if ( !json )
					return [];
				if ( json.status !== 200 ) {
					report.error = String( json.status ) + ' ' + json.message;
					return [];
				}
				if ( !json.data || !jSignage.isArray( json.data ) ) {
					report.error = '418 Missing forecast';
					return [];
				}
				return json.data;
			}
		}
    };

    if ( feedsAPIKey ) {
        feed.headers = {
            'x-api-key': feedsAPIKey,
        };
    }

    return feed;
}

function authorize(feed, service_id, account_id) {
    if (feed.type == 'uri' && typeof (feed.src) == 'string' && feed.src.substr(0, 8) == 'https://') {
        feed.src = 'https://' + encodeURIComponent(account_id) + '@' + feed.src.substr(8);
    }
}

var feedDateFields = {
    weather: ['date'],
    calendar: ['startDate', 'endDate'],
    posts: ['created'],
    photos: ['taken', 'posted']
};

jSignage.Social = {

    version: version,

    minimumRefreshTime: function (feed) {
        if (feed.type === 'weather') {
            var args = feed.parser;
            if (jSignage.isArray(args))
                args = args[0];
			if ( args && typeof ( args ) == 'object' ) {
				if ( args.key )
					return 60;
                else if ( args.days == 'current' )
                    return 3600;
                else
                    return 10800;
            }
        }
        return 60;
    },

    getFeed: function (feed, callback) {
        var args = feed.parser;
        if (typeof (args) == 'object') {
            if (jSignage.isArray(args))
                args = args[0];
            if (typeof (args) != 'object' || args.type != 'args')
                args = {};
        } else {
            args = {};
        }

        function bad_provider() {
            jSignage.setTimeout(function () {
                callback(null, '400 Unknown provider for ' + feed.type + ' feed: ' + args.provider);
            }, 0);
            return null;
        }
        function respond_with(data) {
            jSignage.setTimeout(function () {
                callback(data);
            }, 0);
            return null;
        }

        var report = {};

        function parse(data, error) {
            if (data !== null) {
                var parsed = jSignage.parseFeed(feed, data);
                if (report.error) {
                    jSignage.warn('feed parsing error: ' + report.error);
                    callback(null, report.error);
                } else {
                    callback(parsed);
                }
            } else {
                callback(null, error);
            }
        }

        switch (feed.type) {
            case 'forex':
                // requires base and currency parameters
                var base = args.base || 'EUR';
                var currency = args.currency || 'USD';
                if (!jSignage.isArray(currency))
                    currency = [currency];
                if (args.provider === 'ecb') {
                    feed = forex_ecb(base, currency);
                } else if (args.provider === 'yahoo' || args.provider === 'alphavantage') {
					feed = forex_alphavantage( base, currency, report );
                } else {
                    return bad_provider();
                }
                break;
            case 'finance':
                if (args.provider === 'yahoo' || args.provider === 'alphavantage') {
                    var symbol = args.symbol || 'XAUUSD=X';
                    if (!jSignage.isArray(symbol))
						symbol = [symbol];
					var name = args.name || [];
					if ( !jSignage.isArray( name ) )
						name = [name];
					feed = finance_alphavantage( symbol, name, report );
                } else {
                    return bad_provider();
                }
                break;
			case 'weather':
				feed = weather_proxy( args, report );
                break;
            case 'spreadsheet':
                if (args.provider == 'google') {
                    feed = google_spreadsheet(feed, args.workbook, args.sheet || 0, args.range || null, args.transpose || false, args.headers || false, args.columns || null, report);
                    authorize(feed, 'google', args.account);
                } else if (args.provider == 'azure') {
                    feed = azure_spreadsheet(feed, args, args.workbook, args.sheet, args.range || null, args.transpose || false, args.headers || false, args.columns || null, args.workbookPath || null, args.account, report);
                    authorize(feed, 'azure', args.account);
                } else if (args.provider == 'sharepoint') {
                    feed = sharepoint_spreadsheet(feed, args.workbook, args.sheet, report);
                    authorize(feed, 'azure', args.account);
                } else if (args.provider == 'csv') {
                    feed = csv_spreadsheet(args.workbook, args.separator || ',', args.quotes || false, args.range || null, args.transpose || false, args.headers || false, args.columns || null);
                } else if (args.provider == 'xls') {
                    feed = xls_spreadsheet(args.workbook, args.sheet || null, args.range || null, args.data || null, args.transpose || false, args.headers || false, args.columns || null);
                } else {
                    return bad_provider();
                }
                break;
            case 'chart':
                if (args.provider == 'azure') {
                    feed = azure_chart(args, args.workbook, args.sheet, args.chart, report);
                    authorize(feed, 'azure', args.account);
                } else if (args.provider == 'xls') {
                    feed = xls_chart(args.width, args.height, args.workbook, args.sheet || null, args.chart || null, args.data || null);
                } else {
                    return bad_provider();
                }
                break;
            case 'calendar':
                if (args.provider == 'google') {
                    var range = jSignage._icalDateRange(args);
                    feed = google_calendar(feed, args.id || 0, range[0], range[1], args, report);
                    authorize(feed, 'google', args.account);
                } else if (args.provider == 'azure') {
                    var range = jSignage._icalDateRange(args);
                    feed = azure_calendar(feed, args.id, range[0], range[1], args, report);
                    authorize(feed, 'azure', args.account);
                } else if (args.provider == 'ics') {
                    feed = ics_calendar(args);
                } else {
                    return bad_provider();
                }
                break;
            case 'posts':
                if (args.provider == 'twitter') {
                    feed = twitter_posts(feed, args.timeline || 'home', args.username, args.query || '', args.count || 0, args.retweets || false, report);
                    authorize(feed, 'twitter', args.account);
                } else if (args.provider == 'yammer') {
                    feed = yammer_posts(feed, args, report);
                    authorize(feed, 'yammer', args.account);
                } else if (args.provider == 'sharepoint') {
                    feed = sharepoint_posts(feed, args, report);
                    authorize(feed, 'azure', args.account);
                } else {
                    return bad_provider();
                }
                break;
            case 'photos':
                if (args.provider == 'flickr') {
                    feed = flickr_photos(feed, args, report);
                    authorize(feed, 'flickr', args.account);
                } else if (args.provider == 'instagram') {
                    feed = instagram_photos(feed, args, report);
                    authorize(feed, 'instagram', args.account);
                } else if (args.provider == 'uri') {
                    feed = webdav_photos(args);
                } else if (args.provider == 'azure') {
                    feed = azure_photos(feed, args, report);
                    authorize(feed, 'azure', args.account);
                } else if (args.provider == 'google') {
                    feed = google_photos(feed, args, report);
                    authorize(feed, 'google', args.account);
                } else {
                    return bad_provider();
                }
                break;
            case 'dashboard':
                if (args.provider === 'powerbi') {
                    return respond_with(powerbi_dashboard(args));
                } else {
                    return bad_provider();
                }
                break;
            default:
                return bad_provider();
        }
        jSignage.getFeed(feed, parse);
    },

    fromJSON: function (feed, table) {
        var dateFields = feedDateFields[feed.type];
        if (dateFields) {
            if (dateFields.length > 1) {
                for (var i = 0; i < table.length; i++) if (table[i] && typeof (table[i]) == 'object') {
                    var row = table[i];
                    for (var j = 0; j < dateFields.length; j++) {
                        var x = dateFields[j];
                        if (x in row)
                            row[x] = new Date(row[x]);
                    }
                }
            } else {
                var x = dateFields[0];
                for (var i = 0; i < table.length; i++) if (table[i] && typeof (table[i]) == 'object') {
                    var row = table[i];
                    if (x in row)
                        row[x] = new Date(row[x]);
                }
            }
        }
    },

    setFeedsSource: function ( opt ) {
        if ( opt.feedsAPIKey )
            feedsAPIKey = opt.feedsAPIKey;
        if ( opt.financeAPI )
            financeAPI = opt.financeAPI;
        if ( opt.weatherAPI )
            weatherAPI = opt.weatherAPI;
    }
};

})();
