#!/bin/bash
#
# Sets the hostname and apache header to the serial number of the device
# prepending the prefix from the resources to the hostname

RCINIT=/usr/share/resources/default/init/main
RCRECOVERY=/usr/share/resources/default/recovery/main

# Load resources
[ -f "$RCINIT" ] && . "$RCINIT"
[ -f "$RCRECOVERY" ] && . "$RCRECOVERY"

# Detect if running in recovery mode
[ -f /etc/init.d/getdata.sh ] && RECOVERY="yes" || RECOVERY="no"

# Source the init script functions
. /etc/init.d/functions

# This regexp matches any characters which are either not printable ASCII
# nor space, which have special meaning for the shell inside double quotes
# ('!', '"', '$', '`', '\') and the pipe symbol ('|') which is used
# as separator.
UNSAFE_RE="[^] #%&'()*+,./:;<=>?@[^_{}~0-9A-Za-z-]"

# This script requires the extglob option to parse and run
shopt -s extglob
# This script may be sourced by bash started as sh and thus in POSIX mode, turn that off
set +o posix

get_main_net_iface()
{
    MAIN_IFACE_NAME=
    MAIN_IFACE_HWADDR=

    # Search for the first on-board Ethernet network interface to get serial number from.
    # udev assigns stable names in the ID_NET_NAME_ONBOARD using prefix "en" for wired Ethernet,
    # followed by "o" for on-board devices and the SMBIOS or ACPI assigned number; port 0 on the
    # first on-board wired Ethernet interface thus gets named "eno1" by udev (port 0 gets no port
    # in the name), but that naming scheme may not be active so the kernel interface name may be
    # different
    type udevadm 2>/dev/null >/dev/null && has_udevadm=1 || has_udevadm=
    min_onboard_index=
    min_path=
    min_ifindex=
    best_iface=
    for iface in /sys/class/net/*; do
        [ -e "$iface"/type -a -e "$iface"/address -a -e "$iface"/ifindex -a -e "$iface"/uevent ] || continue
        # accept only Ethernet type interfaces (includes wlan)
        [ "$(< "$iface"/type)" == 1 ] || continue
        # ignore wireless interfaces (wlan, wwan, etc.)
        ! grep -q ^DEVTYPE=w "$iface"/uevent || continue
        ifindex="$(< "$iface"/ifindex)"
        # handle on-board interface, selecting lowest numbered one
        UDEV_ID_NET_NAME_ONBOARD=
        UDEV_ID_NET_NAME_PATH=
        UDEV_ID_BUS=
        [ -z "$has_udevadm" ] || eval `udevadm info --query=property --export-prefix=UDEV_ --export --path="$iface"`
        case "$UDEV_ID_NET_NAME_ONBOARD" in
            eno+([0-9]) )
            onboard_index=${UDEV_ID_NET_NAME_ONBOARD:3}
            if [ -z "$min_onboard_index" -o $onboard_index -lt ${min_onboard_index:-0} ]; then
                # lower numbered on-board interface => select interface as best candidate
                min_onboard_index=$onboard_index
                best_iface="$iface"
            fi
            ;;
        esac
        # if we already have a selection based on onboard index, no need to consider any other
        [ -z "$min_onboard_index" ] || continue
        # only consider remaining interfaces if on PCI bus or no bus identified, avoids USB devices
        [ -z "$UDEV_ID_BUS" -o "$UDEV_ID_BUS" = "pci" ] || continue
        # if there is a path based name select the one which is sorts first
        if [ -n "$UDEV_ID_NET_NAME_PATH" ] && [ -z "$min_path" -o "$UDEV_ID_NET_NAME_PATH" \< "$min_path" ]; then
            min_path="$UDEV_ID_NET_NAME_PATH"
            best_iface="$iface"
        fi
        # if we already have a selection based on path, no need to consider any other
        [ -z "$min_path" ] || continue
        # select lowest numbered interface as fallback
        [ -n "$min_ifindex" -a "$ifindex" -ge "${min_ifindex:-0}" ] || best_iface="$iface"
    done

    if [ -z "$best_iface" ]; then
        echo "no suitable main network interface found for ID"
        return 1
    fi

    MAIN_IFACE_NAME="${best_iface##*/}"
    MAIN_IFACE_HWADDR="$(< "$best_iface"/address)"
}

get_block_devs()
{
    local disk_kname name kname type mountpoint pkname junk
    local removable hotplug subsystems partuuid partlabel
    local -A disk_names

    MAIN_DISK_NAME=

    # The main disk is normally the one from which the root fs is mounted
    # (notable exceptions are recovery and NFS-root), so check if we can
    # find the disk whose one of its partitions is the root fs.
    # As relative ordering of disks and partitions in lsblk output may change and
    # --inverse is unreliable, accumulate all use the parent kernel name (PKNAME)
    # to match a partition to its containing disk.
    while read name kname type mountpoint pkname junk; do
        [ "$type" = "disk" ] && disk_names["$kname"]="$name"
        [ "$mountpoint" = "/" ] || continue
        [ "$type" != "disk" ] || echo "WARNING: root fs is mounted from whole disk"
        disk_kname="$pkname"
    done < <(lsblk --raw --noheadings -o NAME,KNAME,TYPE,MOUNTPOINT,PKNAME)
    if [ -n "$disk_kname" ]; then
        MAIN_DISK_NAME="${disk_names[$disk_kname]}"
        [ -n "$MAIN_DISK_NAME" ] || echo "WARNING: no disk found for root fs partition (wanted '$disk_kname')"
    fi

    # If root fs is not on a block device the first non-removable, non-hotplug disk,
    # skipping ramdisk and mtdblock, becomes the main disk.
    if [ -z "$MAIN_DISK_NAME" ]; then
        while read name kname removable hotplug subsystems type junk; do
            [ "$removable" = 0 ] || continue
            [ "${subsystems/usb/}" = "$subsystems" ] || continue # anything via USB is to be skipped
            # mmc is special as it can be SD card or eMMC and eMMC has hardware partitions
            if [ "${kname#mmcblk[0-9]}" != "$kname" ]; then
                # do not test for hotplug, as mmc is always considered hotplug but eMMC is not
                [ "${kname%boot[0-9]}" = "$kname" ] || continue # skip eMMC boot hardware partitions
                [ "${kname%gp[0-9]}" = "$kname" ] || continue # skip eMMC general purpose hardware partitions
            else
                [ "$hotplug" = 0 ] || continue
            fi
            [ "$type" = "disk" ] || continue
            MAIN_DISK_NAME="$name"
            break
        done < <(lsblk --nodeps --raw --noheadings --exclude 1,31 -o NAME,KNAME,RM,HOTPLUG,SUBSYSTEMS,TYPE)
    fi

    if [ -z "$MAIN_DISK_NAME" ]; then
        echo "Could not find the main disk"
        return 1
    fi

    # On ARM platforms the block devices are fixed and not handled via identifiers
    [ "$HOSTTYPE" = "arm" ] && return

    # On Fukiran the block devices are fixed
    if [ "$HW_TYPE" = "fukiran" ]; then
        FAILSAFE_DATA_BDEV1=/dev/mmcblk0boot0
        FAILSAFE_DATA_BDEV2=/dev/mmcblk0boot1
        PERSISTENT_DATA_BDEV=/dev/mmcblk0gp0
        return
    fi

    FAILSAFE_DATA_BDEV1=
    FAILSAFE_DATA_BDEV2=
    PERSISTENT_DATA_BDEV=
    # Find the partition within that device that matches the GPT partition type UUID and label
    while read name partuuid partlabel junk; do
        [ "$partuuid" = "0fc63daf-8483-4772-8e79-3d69d8477de4" ] || continue
        case "$partlabel" in
            "failsafe-data-1")
                FAILSAFE_DATA_BDEV1=/dev/"$name"
                ;;
            "failsafe-data-2")
                FAILSAFE_DATA_BDEV2=/dev/"$name"
                ;;
            "persistent-data")
                PERSISTENT_DATA_BDEV=/dev/"$name"
                ;;
        esac
    done < <(lsblk  --raw --noheadings -o NAME,PARTTYPE,PARTLABEL /dev/"$MAIN_DISK_NAME")
    [ -z "$FAILSAFE_DATA_BDEV1" ] && echo 'missing disk partitions for failsafe-data-1'
    [ -z "$FAILSAFE_DATA_BDEV2" ] && echo 'missing disk partitions for failsafe-data-2'
    [ -z "$PERSISTENT_DATA_BDEV" ] && echo 'missing disk partitions for persistent-data'
    return 0
}

get_service_serial_number()
{
    # on ARM platforms cpuinfo has serial number
    if [ "$HOSTTYPE" = "arm" ]; then
        SERVICE_SN="$(sed -ne '/^Serial\s\+/{s,.*:\s\+,,p}' /proc/cpuinfo)"
        if [ $? -ne 0 ]; then
            echo "failed recovering serial number from /proc/cpuinfo"
            return 1
        fi
        if [ "$SERVICE_SN" = "000000000000" -o "$SERVICE_SN" = "0000000000000000" ]; then
            SERVICE_SN=
        fi
    fi
    [ -z "$SERVICE_SN" ] || return 0

    # use MAC address from main interface as serial number, without separators
    [ -z "$MAIN_IFACE_HWADDR" ] || SERVICE_SN="$(echo "$MAIN_IFACE_HWADDR" | sed -e 's/[^0-9a-fA-F]//g;y/ABCDEF/abcdef/')"
    if [ -z "$SERVICE_SN" ]; then
        echo "No serial number and no network interface to get a valid one from."
        if [ "$RECOVERY" = "yes" ]; then
            SERVICE_SN=000000000000
            echo "Using dummy $SERVICE_SN as serial number for recovery mode"
        else
            return 1
        fi
    fi
}

serial_number_fixups()
{
    # a handful of Fukiran units have bogus user serial numbers but the LAN MAC
    # address (i.e. service serial number) is OK
    case "$SERVICE_SN" in
        001d50220068)
        [ "$USER_SN" = "X2L62593" ] && USER_SN="X2L625AZ"
        ;;
        001d5022006a)
        [ "$USER_SN" = "X2L62593" ] && USER_SN="X24625CA"
        ;;
        001d5022006b)
        [ "$USER_SN" = "X2L62593" ] && USER_SN="X2L625DT"
        ;;
    esac
    return 0
}

# Get a name from a file, filtering unsafe characters and dummy values
# arg1: file path
# arg2: optional additional regular expression to match for dummy values, concatenated after base one with |
get_name_from_file()
{
    # A (case insensitive) extended regexp that catches unset values like :
    # "Default String", "Type2 - Board Serial Number", "Type1 - 123456789",
    # "Chassis Serial Number", "SYS_CHASSIS_SERIAL_NUM_1" or only whitespace
    local unset_ere='default|oem|o\.e\.m\.|^type[0-9][_[:space:]]|^[[:space:]]*$'
    local file="$1"
    local extra_ere="$2"
    local val

    if [ -n "$extra_ere" ]; then
        unset_ere="$unset_ere|$extra_ere"
    fi

    if [ -f "$file" ]; then
        val="$(sed -e "s|$UNSAFE_RE||g;s|^\s*||;s|\s*$||;q" "$file")"
        if ! echo "$val" | egrep -iq "$unset_ere"; then
            echo "$val"
        fi
    fi
}

get_sn_from_file()
{
    get_name_from_file "$1" 'serial|[[:space:]]'
}

get_alt_serial_numbers()
{
    if [ -f /sys/class/spx-board-info/id/serial ]; then
        USER_SN="$(< /sys/class/spx-board-info/id/serial)"
    else
        USER_SN=
    fi
    BOARD_SN="$(get_sn_from_file /sys/devices/virtual/dmi/id/board_serial)"
    SYSTEM_SN="$(get_sn_from_file /sys/devices/virtual/dmi/id/product_serial)"
    CHASSIS_SN="$(get_sn_from_file /sys/devices/virtual/dmi/id/chassis_serial)"

    return 0
}

get_serial_numbers()
{
    SERVICE_SN=
    USER_SN=
    BOARD_SN=
    SYSTEM_SN=
    CHASSIS_SN=
    SN=

    get_alt_serial_numbers || return
    get_service_serial_number || return
    serial_number_fixups || return
    [ -n "$USER_SN" ] && SN="$USER_SN" || SN="$SERVICE_SN"
    return 0
}

get_hw_type()
{
    local board_name
    local sys_name
    local sys_vendor

    # on ARM platforms cpuinfo has hardware name
    if [ "$HOSTTYPE" = "arm" ]; then
        HW_TYPE="$(sed -ne '/^Hardware\s\+/{s,.*:\s\+,,p}' /proc/cpuinfo)"
        if [ $? -ne 0 ]; then
            echo "failed recovering hardware type from /proc/cpuinfo"
            return 1
        fi
    fi

    # If no hardware type try baseboard name from SMBIOS followed by CPU model string, sanitized
    if [ -z "$HW_TYPE" ]; then
        board_name="$(get_name_from_file /sys/devices/virtual/dmi/id/board_name)"
        sys_name="$(get_name_from_file /sys/devices/virtual/dmi/id/product_name)"
        sys_vendor="$(get_name_from_file /sys/devices/virtual/dmi/id/sys_vendor)"

        case "$board_name" in
            HMP400|Fukiran)
                HW_TYPE="fukiran"
                ;;
            MBD111-*)
                if [ "$sys_vendor" = "SpinetiX" ]; then
                    HW_TYPE="ume"
                fi
                ;;
            MBD654-*)
                if [ "$sys_vendor" = "SpinetiX" ]; then
                    HW_TYPE="tsubaki"
                fi
                ;;
        esac
        if [ -z "$HW_TYPE" ]; then
            HW_TYPE="${board_name:-$sys_name}|$(sed -n -e "/^model name\s*:/{s|^[^\s]*:\s*||;s|$UNSAFE_RE||g;p;q}" /proc/cpuinfo)"
        fi
    fi

    [ -n "$HW_TYPE" ] || HW_TYPE="unknown"

    HW_TYPE="$rc_hw_prefix""$HW_TYPE"
}

need_identifiers_update()
{
    # must have a service serial number
    [ -z "$SPX_SN" ] && return

    # main network interface index should not be present
    [ -n "$SPX_MAIN_IFACE_IFINDEX" ] && return

    # might need a fixup of user serial numbers
    [ "$SPX_USER_SN" != "$USER_SN" ] && return

    # no update needed
    return 1
}

case "$1" in
    start|restart)
    if [ -s /etc/hostname -a -s /etc/spinetix/identifiers -a -s /etc/apache2/conf.d/30-spxserial.conf ]; then
        exit 0
    fi
    # load current identifiers, if any
    [ -s /etc/spinetix/identifiers ] && . /etc/spinetix/identifiers
    udevadm settle --timeout 30 > /dev/null 2>&1
    if ! { get_main_net_iface || [ "$RECOVERY" = "yes" ]; } || \
        ! get_serial_numbers || ! get_hw_type || \
        ! { get_block_devs || [ "$RECOVERY" = "yes" ]; }
    then
        failure
        exit 1
    fi
    if [ ! -s /etc/hostname ]; then
        echo -n "Setting the hostname to service serial number "
        echo "${rc_host_prefix}${SERVICE_SN}" > /etc/hostname
        success
        echo
    fi
    # the identifiers file can be read concurrently as this can be called during updates, so
    # the file needs to be updated atomically
    if need_identifiers_update; then
        echo -n "Saving identifiers "
        echo "SPX_SN='${SERVICE_SN}'" > /etc/spinetix/identifiers.tmp
        [ -z "$USER_SN" ] || echo "SPX_USER_SN='${USER_SN}'" >> /etc/spinetix/identifiers.tmp
        [ -z "$BOARD_SN" ] || echo "SPX_BOARD_SN='${BOARD_SN}'" >> /etc/spinetix/identifiers.tmp
        [ -z "$SYSTEM_SN" ] || echo "SPX_SYSTEM_SN='${SYSTEM_SN}'" >> /etc/spinetix/identifiers.tmp
        [ -z "$CHASSIS_SN" ] || echo "SPX_CHASSIS_SN='${CHASSIS_SN}'" >> /etc/spinetix/identifiers.tmp
        echo "SPX_HW_TYPE='${HW_TYPE}'" >> /etc/spinetix/identifiers.tmp
        echo "SPX_MAIN_IFACE_NAME='${MAIN_IFACE_NAME}'" >> /etc/spinetix/identifiers.tmp
        echo "SPX_MAIN_IFACE_HWADDR='${MAIN_IFACE_HWADDR}'" >> /etc/spinetix/identifiers.tmp
        echo "SPX_MAIN_DISK_NAME='${MAIN_DISK_NAME}'" >> /etc/spinetix/identifiers.tmp
        if [ "$HOSTTYPE" != "arm" ]; then
            echo "SPX_FAILSAFE_DATA_BDEV1='${FAILSAFE_DATA_BDEV1}'" >> /etc/spinetix/identifiers.tmp
            echo "SPX_FAILSAFE_DATA_BDEV2='${FAILSAFE_DATA_BDEV2}'" >> /etc/spinetix/identifiers.tmp
            echo "SPX_PERSISTENT_DATA_BDEV='${PERSISTENT_DATA_BDEV}'" >> /etc/spinetix/identifiers.tmp
        fi
        mv /etc/spinetix/identifiers.tmp /etc/spinetix/identifiers
        success
        echo
    fi
    if [ ! -s /etc/apache2/conf.d/30-spxserial.conf ]; then
        echo -n "Setting apache httpd server name and serial number / hardware headers "
        sed -e "s/@@servername@@/${rc_host_prefix}${SERVICE_SN}.local/" \
            -e "s/@@serial@@/$SN/;s/@@hardware@@/${HW_TYPE}${rc_hwtype_suffix}/;" \
            /etc/apache2/conf.d/30-spxserial.conf.tmpl > /etc/apache2/conf.d/30-spxserial.conf
        success
        echo
    fi
    ;;
    stop|reload|force-reload)
    ;;
esac



