#!/bin/bash

#
# Copyright (C) 2017 SpinetiX
#

NAME=bootspx-common.sh

#
# Assorted initialization things common to normal firmware and recovery
# console, run late in rcS.
# It should run after bootmisc.sh
#

[ -f /etc/default/network ] && source /etc/default/network

# Sets the system time precisely from the hardware clock (RTC),
# normally this is initialized loosely by the kernel (with a 1 sec
# precision).
set_systime_from_hc() {
    hwclock --utc --noadjfile --hctosys
    if [ $? -ne 0 ]; then
        echo 'failed to precisely set system time from hardware clock'
    fi
}

set_fallback_systime() {
    if [ ! -f /etc/os-release ]; then
        echo "missing /etc/os-release, cannot check system time"
        return
    fi
    reftime=$(date -u -r /etc/os-release +%4Y%2m%2d%2H%2M)
    systime=$(date -u +%4Y%2m%2d%2H%2M)
    if [ "$reftime" -gt "$systime" ]; then
        echo 'system time older than firmware creation time, setting to latter'
        date -u ${reftime#????}${reftime%????????}
    fi
}

# Apply extra sysctl from /etc/sysctl.d, base /etc/sysctl.conf is
# already applied by bootmisc.sh
apply_extra_sysctl() {
    for file in /etc/sysctl.d/*.conf; do
        [ -e "$file" ] || continue # avoid error if no glob match
        sysctl -p "$file"
    done
}

# Applies global network configurations
apply_net_global_configs() {

    # non-IPv6 stuff first

    # IPv6 stuff, it returns immediately if no IPv6 support

    [ -f /proc/net/if_inet6 ] || modprobe ipv6
    [ -d /proc/sys/net/ipv6/conf/ ] || return

    # IPv6 auto-configuration, default is to have it enabled
    if [ "$IPV6_AUTOCONF" != "no" ]; then
        ipv6_autoconf=1
    else
        ipv6_autoconf=0
    fi
    for i in /proc/sys/net/ipv6/conf/* ; do
        iface=${i##*/}
        sysctl -q -w \
            net.ipv6.conf.${iface}.accept_ra=$ipv6_autoconf \
            net.ipv6.conf.${iface}.accept_redirects=$ipv6_autoconf \
            net.ipv6.conf.${iface}.autoconf=$ipv6_autoconf
    done
}

# applies all net.<proto>.conf.default sysctl settings to the already
# initialized interfaces; if an interface was already up (e.g.,
# NFS-root) before the net.<proto>.conf.default setting was applied
# then it would not have picked up the change, this propagates these
# settings to them
fixup_iface_sysctl() {
    # The sed command below, input to read loop, gets all
    # net.<proto>.conf.default.<parameter>=<value> in sysctl.conf
    # files and reports three space separated fields: the
    # net/<proto>/conf, <parameter> and <value>.
    #
    # Note that the read loop executes in a subshell, so it does not
    # affect this subshell's variables.
    sed -n -e '/^\s*net\.[^\s.]*\.conf\.default\.\S\+\s*=/{s:.default.: :;s:\.:/:g;s:=: :;p}' \
        /etc/sysctl.conf /etc/sysctl.d/*.conf 2>/dev/null | \
        while read -r path param value; do
            procpath=/proc/sys/$path
            [ -e "$procpath" ] || continue
            # apply to all interfaces (except pseudo "default" and "all")
            for file in $procpath/*/$param; do
                [ "$file" = "$procpath/default/$param" ] && continue
                [ "$file" = "$procpath/all/$param" ] && continue
                echo "$value" > "$file"
            done
        done
}

fixup_nfsroot_dhcpcd() {
    local dhconf
    dhconf=/etc/dhcpcd.conf
    # For NFS root compatibility dhcpcd must have persistent enabled and
    # release disabled, make sure that is the case if running NFS root.
    if grep -q '\bnfsroot=' /proc/cmdline && [ -f "$dhconf" ]; then
        if ! grep -q '^[[:space:]]*persistent[[:space:]]*$' "$dhconf"; then
            sed -i -e '/^#[[:space:]]*persistent[[:space:]]*$/s/#[[:space:]]*//' "$dhconf"
            if ! grep -q '^[[:space:]]*persistent[[:space:]]*$' "$dhconf"; then
                echo >> "$dhconf"
                echo '# Persist configuration for NFS root' >> "$dhconf"
                echo 'persistent' >> "$dhconf"
            fi
        fi
        if grep -q '^[[:space:]]*release[[:space:]]*$' "$dhconf"; then
            sed -i -e '/^[[:space:]]*release[[:space:]]*$/s/^/#/' "$dhconf"
        fi
    fi
}

case "$1" in
    start|restart)
        set_systime_from_hc
        set_fallback_systime
        apply_extra_sysctl
        apply_net_global_configs
        fixup_iface_sysctl
        fixup_nfsroot_dhcpcd
        ;;
    stop|reload|force-reload)
        ;;
    *)
esac
