<?php

/*

  Copyright (c) 2010, SpinetiX S.A.
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above
      copyright notice, this list of conditions and the following
      disclaimer in the documentation and/or other materials provided
      with the distribution.

    * Neither the name of SpinetiX S.A. nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
  FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
  COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
  ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  POSSIBILITY OF SUCH DAMAGE.

*/

if (!extension_loaded('json')) {
    trigger_error("json extension is not loaded", E_USER_ERROR);
}

require_once 'Updater.php';

/**
 * This class implements a simple JSON RPC 1.0 server.
 *
 * The JSON-RPC 1.0 specification is available at
 * http://json-rpc.org/wiki/specification.
 *
 * To use create an object of this class passing it an object which
 * implements the RPC method calls as class methods and call the run()
 * mehod. This will process the RPC notification in the HTTP request
 * and return the next pending RPC call to the device, if any, in the
 * HTTP response.
 *
 * The RPC method names may differ from the class method names by
 * providing a method map to the constructor. The map can also be used
 * to restrict the methods exposed via RPC. The map can also be useful
 * if the RPC method names are not legal PHP method names.
 *
 * The Content-Length header is included in the HTTP request so as to
 * enable use of keep-alive HTTP connections.
 *
 * Note that JSON-RPC calls must be done with the HTTP POST method and
 * have the Content-Type as specified in the $content_type property
 * below.
 *
 */
class JSONRPCServer
{
    /**
     * The content type required for JSON-RPC calls.
     * @var string
     */
    const CONTENT_TYPE = 'application/json';

    /**
     * Session name for cookie-based authentication.
     * @var string
     */
    const SESSION_NAME = 'uisess';

    /**
     * Session variable containing the username for cookie-based 
     * authentication.
     * @var string
     */
    const SESSION_USERNAME_KEY = '__name';

    /**
     * @var object The object implementing the RPC methods
     */
    private $object;

    /**
     * @var array The method map, NULL if none.
     */
    private $methods;

    /**
     * @var boolean If TRUE the incoming request is saved
     */
    private $save_request;

    /**
     * @var boolean TRUE if a response was sent, FALSE otherwise.
     */
    private $responded = false;

    /**
     * @var string The error message if the last RPC call returned an
     * error; NULL if none.
     */
    private $error = null;

    /**
     * @var string The raw JSON-RPC request received in the last RPC
     * call.
     */
    private $raw_request = null;

    /**
     * @var mixed Current logged user, false if not logged
     */
    private $username = false;

    /**
     * @var mixed Current logged user group, false if not logged
     */
    private $usergroup = false;

    /**
     * Sets the username from the session for cookie-based authentication.
     */
    private function setUsernameFromSession() {
        $old_session_name = session_name(self::SESSION_NAME);
        if ($old_session_name === false) {
            trigger_error(
                'session_name(' . self::SESSION_NAME . ') failed', 
                E_USER_ERROR);
            return;
        }
        if (session_start() == false) {
            trigger_error('session_start failed', E_USER_ERROR);
            session_name($old_session_name);
            return;
        }
        //trigger_error('session_id: ' . session_id(), E_USER_NOTICE);
        //trigger_error('_SESSION: ' . print_r($_SESSION, true), E_USER_NOTICE);
        foreach ($_SESSION as $key => $value) {
            if (strlen($key) >= strlen(self::SESSION_USERNAME_KEY) && strpos(
                $key, self::SESSION_USERNAME_KEY, 
                -strlen(self::SESSION_USERNAME_KEY)) !== false) {
                //trigger_error('username: ' . $value, E_USER_NOTICE);
                $this->username = $value;
                break;
            }
        }
        if (session_abort() == false) {
            trigger_error('session_abort failed', E_USER_WARNING);
        }
        if (session_name($old_session_name) === false) {
            trigger_error(
                'session_name(' . $old_session_name . ') failed', 
                E_USER_WARNING);
        }
    }

    /**
     * Creates a new server.
     *
     * @param object $object An instance of the class implementing
     * the RPC method calls.
     *
     * @param array $methods The map of method names, the keys are
     * the RPC method names and the values are the corresponding
     * class method names; if NULL all the class methods are RPC
     * methods by the same name.
     */
    function __construct($object, $methods = null, $save_request = false)
    {
        if (!is_null($methods) && !is_array($methods)) {
            trigger_error("methods must be an array or null", E_USER_ERROR);
        }
        $this->object = $object;
        $this->methods = $methods;
        $this->save_request = $save_request;
        //trigger_error('_SERVER: ' . print_r($_SERVER, true), E_USER_NOTICE);
        if (isset($_SERVER['PHP_AUTH_USER'])) {
            //trigger_error("PHP_AUTH_USER': {$_SERVER['PHP_AUTH_USER']}", E_USER_NOTICE);
            $this->username = $_SERVER['PHP_AUTH_USER'];
        } elseif (isset($_SERVER['REMOTE_USER'])) {
            //trigger_error("REMOTE_USER: {$_SERVER['REMOTE_USER']}", E_USER_NOTICE);
            $this->username = $_SERVER['REMOTE_USER'];
        } elseif (isset($_COOKIE[self::SESSION_NAME])) {
            $this->setUsernameFromSession();
        }
        // when admin endpoints are unprotected (explicitly or because not yet configured) we auto-login as admin
        if (!$this->username) {
        	$noauth = isset($_SERVER['SPXMANAGE_ADMIN_PROTECTED']) && $_SERVER['SPXMANAGE_ADMIN_PROTECTED'] == 'no';
        	if (!$noauth ) {
        		$job = new MaintenanceJobs;
        		$noauth = $job->isNotConfigured();
        	}
        	if ($noauth) {
        		$this->username = "admin";
        	}
        }
        $htgroup = file_exists("/etc/spxmanage")
            ? "/etc/spxmanage/htgroup"
            : "/etc/raperca/htgroup";
        if ($this->username && file_exists($htgroup)) {
            $this->usergroup = array();
            $groupFile = file_get_contents($htgroup);
            $groupsStr = explode("\n", $groupFile);
            foreach ($groupsStr as $groupStr) {
                $data = explode(": ", $groupStr);
                if (count($data) == 2) {
                    if (in_array($this->username, explode(" ", $data[1]))) {
                        $this->usergroup[] = $data[0];
                    }
                }
            }
        }
    }

    /**
     * Send the CORS headers
     */
    private function sendCorsHeaders()
    {
        // Allow from any origin
        if (isset($_SERVER['HTTP_ORIGIN'])) {
            header("Access-Control-Allow-Origin: " . $_SERVER['HTTP_ORIGIN']);
            header('Access-Control-Allow-Credentials: true');
            header('Access-Control-Max-Age: 86400'); // cache for 1 day
        }

        // Access-Control headers are received during OPTIONS requests
        if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
            if (isset($_SERVER['HTTP_ACCESS_CONTROL_REQUEST_METHOD'])) {
                header("Access-Control-Allow-Methods: POST, OPTIONS");
            }
            if (isset($_SERVER['HTTP_ACCESS_CONTROL_REQUEST_HEADERS'])) {
                header("Access-Control-Allow-Headers: Content-Type");
            }

            // that's it
            exit(0);
        }
    }

    /**
     * Send a JSON-RPC response, if appropriate.
     *
     * @param mixed $id The id of the call; if NULL no response is
     * generated (notification)
     *
     * @param mixed $result The call result; should be NULL on error.
     *
     * @param string $error The call's error; should be NULL on
     * success.
     */
    private function respond($id, $result, $error = null, $errorDetails = null)
    {
        $this->error = $error;
        if ($id === null) {
            return;
        }
        $this->responded = true;
        header("Content-Type: " . self::CONTENT_TYPE);
        $body = array(
            'result' => $result,
            'error' => $error,
            'id' => $id,
        );
        if ($errorDetails !== null) {
            $body['errorDetails'] = $errorDetails;
        }
        $body = json_encode($body);
        header("Content-Length: " . strlen($body));
        echo $body;
        flush();
    }

    /**
     * Processes the request in the POST body and dispatches the call
     * to the implementing class method. The response is sent if
     * required (i.e. the call is not a notification).
     *
     * @return null, if the request is not a valid JSON string, boolean TRUE if the method was called, FALSE otherwise.
     */
    public function run()
    {
        $this->responded = false;
        $this->error = null;
        $this->raw_request = null;

        $this->sendCorsHeaders();

        $contentType = "";

        if (
            isset($_SERVER['CONTENT_TYPE']) &&
            !empty($_SERVER['CONTENT_TYPE'])
        ) {
            $charSetPos = strpos($_SERVER['CONTENT_TYPE'], ";");
            if ($charSetPos === false) {
                $contentType = strtolower($_SERVER['CONTENT_TYPE']);
            } else {
                $contentType = strtolower(
                    substr(
                        $_SERVER['CONTENT_TYPE'],
                        0,
                        strpos($_SERVER['CONTENT_TYPE'], ";")
                    )
                );
            }
        }
        // check if called the correct way
        if (
            $_SERVER['REQUEST_METHOD'] != 'POST' ||
            $contentType != self::CONTENT_TYPE
        ) {
            return null;
        } // not a JSON call

        // check for well-formed request
        $request = file_get_contents("php://input");
        if ($request === false) {
            $this->respond(null, null, "failed reading request");
            return null;
        }
        if ($this->save_request) {
            $this->raw_request = $request;
        }
        if (strlen($request) == 0) {
            return null;
        } // empty request
        $request = json_decode($request, true);
        if ($request === null || !is_array($request)) {
            $this->respond(null, null, "invalid JSON-RPC");
            return null;
        }
        if (!array_key_exists('id', $request)) {
            trigger_error(
                "missing id in JSON-RPC, assuming null",
                E_USER_WARNING
            );
            $request['id'] = null; // be tolerant
        }
        try {
            if (empty($request['method'])) {
                throw new Exception("missing JSON-RPC method");
            }
            if (!is_string($request['method'])) {
                throw new Exception("invalid JSON-RPC method");
            }
            if (!array_key_exists('params', $request)) {
                throw new Exception("missing JSON-RPC params");
            }
            if (!is_array($request['params'])) {
                throw new Exception("invalid JSON-RPC params");
            }
        } catch (Exception $e) {
            $this->respond($request['id'], null, $e->getMessage());
            return false;
        }

        // authorization
        if (!$this->username || !$this->usergroup) {
            http_response_code(403);
            $this->respond($request['id'], null, "Unauthorized user");
            return false;
        }
        $allowed_groups = array();
        if (
            property_exists($this->object, 'methods_groups') &&
            array_key_exists($request['method'], $this->object->methods_groups)
        ) {
            $allowed_groups = $this->object->methods_groups[$request['method']];
        }
        $allowed_groups[] = 'admin'; // members of the admin group have access to all methods
        $found = false;
        foreach ($this->usergroup as $group) {
            if (in_array($group, $allowed_groups)) {
                $found = true;
                break;
            }
        }
        if (!$found) {
            $this->respond($request['id'], null, "Unauthorized user");
            return false;
        }

        // find called method
        try {
            $method = $request['method'];
            if ($this->methods !== null) {
                if (!array_key_exists($method, $this->methods)) {
                    throw new Exception(
                        "No such method (" . $request['method'] . ")"
                    );
                }
                $method = $this->methods[$method];
            }
            if (!method_exists($this->object, $method)) {
                throw new Exception(
                    "No such method (" . $request['method'] . ")"
                );
            }
        } catch (Exception $e) {
            $this->respond($request['id'], null, $e->getMessage());
            return false;
        }

        // check that the method is safe to run if the firmware is being
        // updated
        if (Updater::isUpdateInProgress() && !in_array(
            $method, $this->object->safe_methods)) {
            $this->respond(
                $request['id'], null, 'firmware update in progress, retry ' . 
                'later', array('code' => 'FirmwareUpdateInProgress', 'data' => 
                array('retryDelay' => 120)));
            return false;            
        }

        // do the actual call
        try {
            $this->object->callId = $request['id'];
            $result = call_user_func_array(
                array($this->object, $method),
                $request['params']
            );
            if (!isset($result) || is_null($result)) {
                // nothing returned
                $result = (object) array();
            } // empty object
        } catch (Exception $e) {
            $this->respond($request['id'], null, $e->getMessage());
            return true;
        }

        // send the response
        $this->respond($request['id'], $result);

        return true;
    }

    /**
     * Returns the raw JSON-RPC request of the last call, if saving
     * is enabled.
     *
     * @return string The Raw JSON encoded parameters of the last
     * call; NULL if none
     */
    public function request()
    {
        return $this->raw_request;
    }

    /**
     * Returns the error message of the last call.
     *
     * @return string The error message of the last call; NULL if
     * none.
     */
    public function error()
    {
        return $this->error;
    }

    /**
     * @return boolean TRUE if a response was sent in the HTTP
     * response, FALSE otherwise.
     */
    public function responded()
    {
        return $this->responded;
    }
}
